async function query(sql, args) {
	let result = [];
	await ssb.sqlAsync(sql, args, function (row) {
		result.push(row);
	});
	return result;
}

async function main() {
	let whoami = await ssb.getActiveIdentity();
	let following = Object.keys(await ssb.following([whoami], 2));
	let data = await query(
		`
		SELECT
			messages.id,
			content ->> 'title' AS title,
			content ->> '$.image.link' AS image,
			content ->> 'description' AS description,
			content ->> 'authors' AS authors
		FROM messages, json_each(?) AS following
		ON messages.author = following.value
		WHERE
			content ->> 'type' = 'bookclub' AND
			title IS NOT NULL AND
			image IS NOT NULL AND
			description IS NOT NULL
	`,
		[JSON.stringify(following)]
	);
	let books = Object.fromEntries(data.map((x) => [x.id, x]));
	for (let book of Object.values(books)) {
		try {
			book.authors = JSON.parse(book.authors);
		} catch {
			book.authors = [book.authors];
		}
		book.reviews = [];
	}
	let reviews = await query(
		`
		SELECT author, about, json(json_group_object(key, value)) AS content
		FROM (
			SELECT
				messages.author,
				messages.content ->> '$.about' AS about,
				fields.key,
				RANK() OVER (PARTITION BY messages.author, messages.content ->> '$.about', fields.key ORDER BY messages.sequence DESC) AS rank,
				fields.value
			FROM messages, json_each(messages.content) AS fields, json_each(?1) AS book, json_each(?2) AS following
			ON messages.author = following.value
			WHERE
				messages.content ->> 'type' = 'about'
				AND messages.content ->> '$.about' = book.value
				AND NOT fields.key IN ('about', 'type')
			UNION
			SELECT
				messages.author,
				messages.content ->> '$.updates' AS about,
				fields.key,
				RANK() OVER (PARTITION BY messages.author, messages.content ->> '$.updates', fields.key ORDER BY messages.sequence DESC) AS rank,
				fields.value
			FROM messages, json_each(messages.content) AS fields, json_each(?1) AS book, json_each(?2) AS following
			ON messages.author = following.value
			WHERE
				messages.content ->> 'type' = 'bookclubUpdate'
				AND messages.content ->> '$.updates' = book.value
				AND NOT fields.key IN ('about', 'updates', 'type')
		) WHERE rank = 1
		GROUP BY author, about
	`,
		[JSON.stringify(Object.keys(books)), JSON.stringify(following)]
	);
	for (let review of reviews) {
		review.content = JSON.parse(review.content);
		books[review.about].reviews.push(review);
	}
	await app.setDocument(
		utf8Decode(getFile('index.html')).replace('G_DATA', JSON.stringify(data))
	);
}

main().catch(function (e) {
	throw new Error(e.message);
});
