import {LitElement, html, unsafeHTML, range} from './lit-all.min.js';
import * as tfrpc from '/static/tfrpc.js';

class TfJournalEntryElement extends LitElement {
	static get properties() {
		return {
			whoami: {type: String},
			key: {type: String},
			journals: {type: Object},
			text: {type: String},
		};
	}

	constructor() {
		super();
		this.journals = {};
		this.key = new Date().toISOString().split('T')[0];
	}

	markdown(md) {
		var reader = new commonmark.Parser();
		var writer = new commonmark.HtmlRenderer({safe: true});
		var parsed = reader.parse(md || '');
		return writer.render(parsed);
	}

	on_discard(event) {
		this.text = undefined;
	}

	async on_publish() {
		console.log('publish', this.text);
		this.dispatchEvent(
			new CustomEvent('publish', {
				bubbles: true,
				detail: {
					key: this.shadowRoot.getElementById('date_picker').value,
					text: this.text,
				},
			})
		);
	}

	back_dates(count) {
		let now = new Date();
		let result = [];
		for (let i = 0; i < count; i++) {
			let next = new Date(now);
			next.setDate(now.getDate() - i);
			result.push(next.toISOString().split('T')[0]);
		}
		return result;
	}

	on_edit(event) {
		this.text = event.srcElement.value;
	}

	on_date_change(event) {
		this.key = event.srcElement.value;
		this.text = this.journals[this.key]?.text;
	}

	render() {
		console.log('RENDER ENTRY', this.key, this.journals?.[this.key]);
		return html`
			<select id="date_picker" @change=${this.on_date_change}>
				${this.back_dates(10).map(
					(x) => html` <option value=${x}>${x}</option> `
				)}
			</select>
			<div style="display: inline-flex; flex-direction: row">
				<button
					?disabled=${this.text == this.journals?.[this.key]?.text}
					@click=${this.on_publish}
				>
					Publish
				</button>
				<button @click=${this.on_discard}>Discard</button>
			</div>
			<div style="display: flex; flex-direction: row">
				<textarea
					style="flex: 1 1; min-height: 10em"
					@input=${this.on_edit}
					.value=${this.text ?? this.journals?.[this.key]?.text ?? ''}
				></textarea>
				<div style="flex: 1 1">
					${unsafeHTML(
						this.markdown(this.text ?? this.journals?.[this.key]?.text)
					)}
				</div>
			</div>
		`;
	}
}

customElements.define('tf-journal-entry', TfJournalEntryElement);
