#pragma once

/**
** \defgroup trace Performance Tracing
** Generates trace output that is compatible with speedscope.app,
** chrome://tracing or ui.perfetto.dev for scrutinizing what each thread is
** doing for optimization purposes.
** @{
*/

#include <inttypes.h>
#include <stddef.h>

/**
** A trace instance.
*/
typedef struct _tf_trace_t tf_trace_t;

/**
** An SQLite database instance.
*/
typedef struct sqlite3 sqlite3;

/**
** Create a trace instance.  Can be used to produce a Chrome trace event
** document for analyzing performance.  Clean up with tf_trace_destroy().
** @return A trace instance.
*/
tf_trace_t* tf_trace_create();

/**
** Destroy a trace instance that was created with tf_trace_create().
*/
void tf_trace_destroy(tf_trace_t* trace);

/**
** Set the name of the current process.
** @param trace The trace instance.
** @param name The name of the process.
*/
void tf_trace_set_process_name(tf_trace_t* trace, const char* name);

/**
** Record counter values.
** @param trace The trace instance.
** @param name The counter group name.
** @param argc The number of counters.
** @param arg_names The counter names.
** @param arg_values The counter values.
*/
void tf_trace_counter(tf_trace_t* trace, const char* name, int argc, const char** arg_names, const int64_t* arg_values);

/**
** Begin tracing a time interval.  End with tf_trace_end().
** @param trace The trace instance.
** @param name The interval name.
*/
void tf_trace_begin(tf_trace_t* trace, const char* name);

/**
** End tracing the next time interval previously started with tf_trace_begin().
** @param trace The trace instance.
*/
void tf_trace_end(tf_trace_t* trace);

/**
** Export all currently stored trace data to a string.
** @param trace The trace instance.
** @return A string representation of the trace data.
*/
char* tf_trace_export(tf_trace_t* trace);

/**
** A callback to collect trace data.
** @param trace The trace instance.
** @param buffer The trace data.
** @param size The size of the trace data.
** @param user_data User data registered with the callback.
*/
typedef void(tf_trace_write_callback_t)(tf_trace_t* trace, const char* buffer, size_t size, void* user_data);

/**
** Replace the trace recording behavior.
** @param trace The trace instance.
** @param callback A callback that will be called instead of collecting the trace data in a buffer.
** @param user_data User data to pass to the callback.
*/
void tf_trace_set_write_callback(tf_trace_t* trace, tf_trace_write_callback_t* callback, void* user_data);

/**
** Inject raw trace data.
** @param trace The trace instance.
** @param buffer The trace data.
** @param size The size of the trace data.
*/
void tf_trace_raw(tf_trace_t* trace, const char* buffer, size_t size);

/**
** Register for trace-worthy events from sqlite and record them going forward.
** @param trace The trace instance.
** @param sqlite The SQLite database.
*/
void tf_trace_sqlite(tf_trace_t* trace, sqlite3* sqlite);

/** @} */
