#include "httpd.js.h"

#include "file.js.h"
#include "http.h"
#include "log.h"
#include "mem.h"
#include "sha1.h"
#include "ssb.db.h"
#include "task.h"
#include "trace.h"
#include "util.js.h"
#include "version.h"

#include "sodium/crypto_sign.h"
#include "sodium/utils.h"

#define CYAN "\e[1;36m"
#define MAGENTA "\e[1;35m"
#define YELLOW "\e[1;33m"
#define RESET "\e[0m"

static JSClassID _httpd_request_class_id;

typedef struct _http_user_data_t
{
	char redirect[1024];
} http_user_data_t;

static JSValue _httpd_response_write_head(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	JS_SetPropertyStr(context, this_val, "response_status", JS_DupValue(context, argv[0]));
	JS_SetPropertyStr(context, this_val, "response_headers", JS_DupValue(context, argv[1]));
	return JS_UNDEFINED;
}

static int _object_to_headers(JSContext* context, JSValue object, const char** headers, int headers_length)
{
	int count = 0;
	JSPropertyEnum* ptab = NULL;
	uint32_t plen = 0;
	JS_GetOwnPropertyNames(context, &ptab, &plen, object, JS_GPN_STRING_MASK);
	for (; count < (int)plen && count < headers_length / 2; ++count)
	{
		JSPropertyDescriptor desc;
		JSValue key_value = JS_NULL;
		if (JS_GetOwnProperty(context, &desc, object, ptab[count].atom) == 1)
		{
			key_value = desc.value;
			JS_FreeValue(context, desc.setter);
			JS_FreeValue(context, desc.getter);
		}
		headers[count * 2 + 0] = JS_AtomToCString(context, ptab[count].atom);
		headers[count * 2 + 1] = JS_ToCString(context, key_value);
		JS_FreeValue(context, key_value);
	}
	for (uint32_t i = 0; i < plen; ++i)
	{
		JS_FreeAtom(context, ptab[i].atom);
	}
	js_free(context, ptab);
	return count;
}

static JSValue _httpd_response_end(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	tf_http_request_t* request = JS_GetOpaque(this_val, _httpd_request_class_id);
	size_t length = 0;
	const char* cstring = NULL;
	const void* data = NULL;
	JSValue buffer = JS_UNDEFINED;
	if (JS_IsString(argv[0]))
	{
		cstring = JS_ToCStringLen(context, &length, argv[0]);
		data = cstring;
	}
	else if ((data = tf_util_try_get_array_buffer(context, &length, argv[0])) != 0)
	{
	}
	else
	{
		size_t offset;
		size_t size;
		size_t element_size;
		buffer = tf_util_try_get_typed_array_buffer(context, argv[0], &offset, &size, &element_size);
		if (!JS_IsException(buffer))
		{
			data = tf_util_try_get_array_buffer(context, &length, buffer);
		}
	}
	JSValue response_status = JS_GetPropertyStr(context, this_val, "response_status");
	int status = 0;
	JS_ToInt32(context, &status, response_status);
	JS_FreeValue(context, response_status);

	const char* headers[64] = { 0 };
	JSValue response_headers = JS_GetPropertyStr(context, this_val, "response_headers");
	int headers_count = _object_to_headers(context, response_headers, headers, tf_countof(headers));
	JS_FreeValue(context, response_headers);

	tf_http_respond(request, status, headers, headers_count, data, length);

	for (int i = 0; i < headers_count * 2; i++)
	{
		JS_FreeCString(context, headers[i]);
	}
	JS_FreeValue(context, buffer);
	if (cstring)
	{
		JS_FreeCString(context, cstring);
	}
	return JS_UNDEFINED;
}

static JSValue _httpd_response_send(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	tf_http_request_t* request = JS_GetOpaque(this_val, _httpd_request_class_id);
	int opcode = 0x1;
	JS_ToInt32(context, &opcode, argv[1]);
	size_t length = 0;
	const char* message = JS_ToCStringLen(context, &length, argv[0]);
	tf_http_request_websocket_send(request, opcode, message, length);
	JS_FreeCString(context, message);
	return JS_UNDEFINED;
}

static void _httpd_websocket_close_callback(tf_http_request_t* request)
{
	JSContext* context = request->context;
	JSValue response_object = JS_MKPTR(JS_TAG_OBJECT, request->user_data);
	JSValue on_close = JS_GetPropertyStr(context, response_object, "onClose");
	JSValue response = JS_Call(context, on_close, JS_UNDEFINED, 0, NULL);
	tf_util_report_error(context, response);
	JS_FreeValue(context, response);
	JS_FreeValue(context, on_close);
	JS_SetPropertyStr(context, response_object, "onMessage", JS_UNDEFINED);
	JS_SetPropertyStr(context, response_object, "onClose", JS_UNDEFINED);
	JS_FreeValue(context, response_object);
}

static void _httpd_message_callback(tf_http_request_t* request, int op_code, const void* data, size_t size)
{
	JSContext* context = request->context;
	JSValue response_object = JS_MKPTR(JS_TAG_OBJECT, request->user_data);
	JSValue on_message = JS_GetPropertyStr(context, response_object, "onMessage");
	JSValue event = JS_NewObject(context);
	JS_SetPropertyStr(context, event, "opCode", JS_NewInt32(context, op_code));
	JS_SetPropertyStr(context, event, "data", JS_NewStringLen(context, data, size));
	JSValue response = JS_Call(context, on_message, JS_UNDEFINED, 1, &event);
	tf_util_report_error(context, response);
	JS_FreeValue(context, response);
	JS_FreeValue(context, event);
	JS_FreeValue(context, on_message);
}

static JSValue _httpd_websocket_upgrade(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	tf_http_request_t* request = JS_GetOpaque(this_val, _httpd_request_class_id);
	tf_http_request_ref(request);
	const char* header_connection = tf_http_request_get_header(request, "connection");
	const char* header_upgrade = tf_http_request_get_header(request, "upgrade");
	const char* header_sec_websocket_key = tf_http_request_get_header(request, "sec-websocket-key");
	if (header_connection && header_upgrade && header_sec_websocket_key && strstr(header_connection, "Upgrade") && strcasecmp(header_upgrade, "websocket") == 0)
	{
		static const char* k_magic = "258EAFA5-E914-47DA-95CA-C5AB0DC85B11";
		size_t key_length = strlen(header_sec_websocket_key);
		size_t size = key_length + 36;
		uint8_t* key_magic = alloca(size);
		memcpy(key_magic, header_sec_websocket_key, key_length);
		memcpy(key_magic + key_length, k_magic, 36);

		uint8_t digest[20];
		SHA1_CTX sha1 = { 0 };
		SHA1Init(&sha1);
		SHA1Update(&sha1, key_magic, size);
		SHA1Final(digest, &sha1);

		char key[41] = { 0 };
		tf_base64_encode(digest, sizeof(digest), key, sizeof(key));

		const char* headers[64] = { 0 };
		int headers_count = 0;

		headers[headers_count * 2 + 0] = "Upgrade";
		headers[headers_count * 2 + 1] = "websocket";
		headers_count++;

		headers[headers_count * 2 + 0] = "Connection";
		headers[headers_count * 2 + 1] = "Upgrade";
		headers_count++;

		headers[headers_count * 2 + 0] = "Sec-WebSocket-Accept";
		headers[headers_count * 2 + 1] = key;
		headers_count++;

		tf_ssb_t* ssb = tf_task_get_ssb(tf_task_get(context));
		const char* session = tf_http_get_cookie(tf_http_request_get_header(request, "cookie"), "session");
		JSValue jwt = tf_httpd_authenticate_jwt(ssb, context, session);
		tf_free((void*)session);
		JSValue name = !JS_IsUndefined(jwt) ? JS_GetPropertyStr(context, jwt, "name") : JS_UNDEFINED;
		const char* name_string = !JS_IsUndefined(name) ? JS_ToCString(context, name) : NULL;
		const char* session_token = tf_httpd_make_session_jwt(tf_ssb_get_context(ssb), ssb, name_string);
		const char* cookie = tf_httpd_make_set_session_cookie_header(request, session_token);
		tf_free((void*)session_token);
		JS_FreeCString(context, name_string);
		JS_FreeValue(context, name);
		JS_FreeValue(context, jwt);
		headers[headers_count * 2 + 0] = "Set-Cookie";
		headers[headers_count * 2 + 1] = cookie ? cookie : "";
		headers_count++;

		bool send_version = !tf_http_request_get_header(request, "sec-websocket-version") || strcmp(tf_http_request_get_header(request, "sec-websocket-version"), "13") != 0;
		if (send_version)
		{
			headers[headers_count * 2 + 0] = "Sec-WebSocket-Accept";
			headers[headers_count * 2 + 1] = key;
			headers_count++;
		}
		int js_headers_count = _object_to_headers(context, argv[1], headers + headers_count * 2, tf_countof(headers) - headers_count * 2);
		headers_count += js_headers_count;

		tf_http_request_websocket_upgrade(request);
		tf_http_respond(request, 101, headers, headers_count, NULL, 0);

		for (int i = headers_count - js_headers_count; i < headers_count * 2; i++)
		{
			JS_FreeCString(context, headers[i * 2 + 0]);
			JS_FreeCString(context, headers[i * 2 + 1]);
		}

		tf_free((void*)cookie);

		request->on_message = _httpd_message_callback;
		request->on_close = _httpd_websocket_close_callback;
		request->context = context;
		request->user_data = JS_VALUE_GET_PTR(JS_DupValue(context, this_val));
	}
	else
	{
		tf_http_respond(request, 400, NULL, 0, NULL, 0);
	}
	tf_http_request_unref(request);

	return JS_UNDEFINED;
}

JSValue tf_httpd_make_response_object(JSContext* context, tf_http_request_t* request)
{
	JSValue response_object = JS_NewObjectClass(context, _httpd_request_class_id);
	JS_SetOpaque(response_object, request);
	JS_SetPropertyStr(context, response_object, "writeHead", JS_NewCFunction(context, _httpd_response_write_head, "writeHead", 2));
	JS_SetPropertyStr(context, response_object, "end", JS_NewCFunction(context, _httpd_response_end, "end", 1));
	JS_SetPropertyStr(context, response_object, "send", JS_NewCFunction(context, _httpd_response_send, "send", 2));
	JS_SetPropertyStr(context, response_object, "upgrade", JS_NewCFunction(context, _httpd_websocket_upgrade, "upgrade", 2));
	return response_object;
}

bool tf_httpd_redirect(tf_http_request_t* request)
{
	http_user_data_t* user_data = tf_http_get_user_data(request->http);
	if (!user_data || !*user_data->redirect)
	{
		return false;
	}

	char redirect[1024];
	snprintf(redirect, sizeof(redirect), "%s%s", user_data->redirect, request->path);
	tf_http_respond(request, 303, (const char*[]) { "Location", redirect }, 1, NULL, 0);
	return true;
}

typedef struct _httpd_listener_t
{
	int padding;
} httpd_listener_t;

static void _httpd_listener_cleanup(void* user_data)
{
	httpd_listener_t* listener = user_data;
	tf_free(listener);
}

typedef struct _auth_query_work_t
{
	const char* settings;
	JSValue entry;
	JSValue result;
	JSValue promise[2];
} auth_query_work_t;

static void _httpd_auth_query_work(tf_ssb_t* ssb, void* user_data)
{
	auth_query_work_t* work = user_data;
	work->settings = tf_ssb_db_get_property(ssb, "core", "settings");
}

static void _httpd_auth_query_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	auth_query_work_t* work = user_data;
	JSContext* context = tf_ssb_get_context(ssb);
	JSValue name = JS_GetPropertyStr(context, work->entry, "name");
	const char* name_string = JS_ToCString(context, name);
	JSValue settings_value = work->settings ? JS_ParseJSON(context, work->settings, strlen(work->settings), NULL) : JS_UNDEFINED;
	JSValue out_permissions = JS_NewObject(context);
	JS_SetPropertyStr(context, work->result, "permissions", out_permissions);
	JSValue permissions = !JS_IsUndefined(settings_value) ? JS_GetPropertyStr(context, settings_value, "permissions") : JS_UNDEFINED;
	JSValue user_permissions = !JS_IsUndefined(permissions) ? JS_GetPropertyStr(context, permissions, name_string) : JS_UNDEFINED;
	int length = !JS_IsUndefined(user_permissions) ? tf_util_get_length(context, user_permissions) : 0;
	for (int i = 0; i < length; i++)
	{
		JSValue permission = JS_GetPropertyUint32(context, user_permissions, i);
		const char* permission_string = JS_ToCString(context, permission);
		JS_SetPropertyStr(context, out_permissions, permission_string, JS_TRUE);
		JS_FreeCString(context, permission_string);
		JS_FreeValue(context, permission);
	}
	JSValue error = JS_Call(context, work->promise[0], JS_UNDEFINED, 1, &work->result);
	JS_FreeValue(context, work->result);
	tf_util_report_error(context, error);
	JS_FreeValue(context, error);
	JS_FreeValue(context, work->promise[0]);
	JS_FreeValue(context, work->promise[1]);
	JS_FreeValue(context, user_permissions);
	JS_FreeValue(context, permissions);
	JS_FreeValue(context, settings_value);
	tf_free((void*)work->settings);
	JS_FreeCString(context, name_string);
	JS_FreeValue(context, name);
	tf_free(work);
}

static JSValue _httpd_auth_query(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	tf_task_t* task = tf_task_get(context);
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	JSValue headers = argv[0];
	if (JS_IsUndefined(headers))
	{
		return JS_UNDEFINED;
	}

	JSValue cookie = JS_GetPropertyStr(context, headers, "cookie");
	const char* cookie_string = JS_ToCString(context, cookie);
	const char* session = tf_http_get_cookie(cookie_string, "session");
	JSValue entry = tf_httpd_authenticate_jwt(ssb, context, session);
	tf_free((void*)session);
	JS_FreeCString(context, cookie_string);
	JS_FreeValue(context, cookie);

	JSValue result = JS_UNDEFINED;
	if (!JS_IsUndefined(entry))
	{
		JSValue value = JS_NewObject(context);
		JS_SetPropertyStr(context, value, "session", entry);

		auth_query_work_t* work = tf_malloc(sizeof(auth_query_work_t));
		*work = (auth_query_work_t) {
			.entry = entry,
			.result = value,
		};
		result = JS_NewPromiseCapability(context, work->promise);
		tf_ssb_run_work(ssb, _httpd_auth_query_work, _httpd_auth_query_after_work, work);
	}
	return result;
}

typedef struct _magic_bytes_t
{
	const char* type;
	uint8_t bytes[12];
	uint8_t ignore[12];
} magic_bytes_t;

static bool _magic_bytes_match(const magic_bytes_t* magic, const uint8_t* actual, size_t size)
{
	if (size < sizeof(magic->bytes))
	{
		return false;
	}

	int length = (int)tf_min(sizeof(magic->bytes), size);
	for (int i = 0; i < length; i++)
	{
		if ((magic->bytes[i] & ~magic->ignore[i]) != (actual[i] & ~magic->ignore[i]))
		{
			return false;
		}
	}
	return true;
}

const char* tf_httpd_magic_bytes_to_content_type(const uint8_t* bytes, size_t size)
{
	const char* type = "application/binary";
	if (bytes)
	{
		const magic_bytes_t k_magic_bytes[] = {
			{
				.type = "image/jpeg",
				.bytes = { 0xff, 0xd8, 0xff, 0xdb },
			},
			{
				.type = "image/jpeg",
				.bytes = { 0xff, 0xd8, 0xff, 0xe0, 0x00, 0x10, 0x4a, 0x46, 0x49, 0x46, 0x00, 0x01 },
			},
			{
				.type = "image/jpeg",
				.bytes = { 0xff, 0xd8, 0xff, 0xee },
			},
			{
				.type = "image/jpeg",
				.bytes = { 0xff, 0xd8, 0xff, 0xe1, 0x00, 0x00, 0x45, 0x78, 0x69, 0x66, 0x00, 0x00 },
				.ignore = { 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
			},
			{
				.type = "image/png",
				.bytes = { 0x89, 0x50, 0x4e, 0x47, 0x0d, 0x0a, 0x1a, 0x0a },
			},
			{
				.type = "image/gif",
				.bytes = { 0x47, 0x49, 0x46, 0x38, 0x37, 0x61 },
			},
			{
				.type = "image/gif",
				.bytes = { 0x47, 0x49, 0x46, 0x38, 0x39, 0x61 },
			},
			{
				.type = "image/webp",
				.bytes = { 0x52, 0x49, 0x46, 0x46, 0x00, 0x00, 0x00, 0x00, 0x57, 0x45, 0x42, 0x50 },
				.ignore = { 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00 },
			},
			{
				.type = "image/svg+xml",
				.bytes = { 0x3c, 0x73, 0x76, 0x67 },
			},
			{
				.type = "audio/mpeg",
				.bytes = { 0x00, 0x00, 0x00, 0x00, 0x66, 0x74, 0x79, 0x70, 0x6d, 0x70, 0x34, 0x32 },
				.ignore = { 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
			},
			{
				.type = "video/mp4",
				.bytes = { 0x00, 0x00, 0x00, 0x00, 0x66, 0x74, 0x79, 0x70, 0x69, 0x73, 0x6f, 0x6d },
				.ignore = { 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
			},
			{
				.type = "video/mp4",
				.bytes = { 0x00, 0x00, 0x00, 0x00, 0x66, 0x74, 0x79, 0x70, 0x6d, 0x70, 0x34, 0x32 },
				.ignore = { 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
			},
			{
				.type = "audio/midi",
				.bytes = { 0x4d, 0x54, 0x68, 0x64 },
			},
		};

		for (int i = 0; i < tf_countof(k_magic_bytes); i++)
		{
			if (_magic_bytes_match(&k_magic_bytes[i], bytes, size))
			{
				type = k_magic_bytes[i].type;
				break;
			}
		}
	}
	return type;
}

const char* tf_httpd_ext_to_content_type(const char* ext, bool use_fallback)
{
	if (ext)
	{
		typedef struct _ext_type_t
		{
			const char* ext;
			const char* type;
		} ext_type_t;

		const ext_type_t k_types[] = {
			{ .ext = ".html", .type = "text/html; charset=UTF-8" },
			{ .ext = ".js", .type = "text/javascript; charset=UTF-8" },
			{ .ext = ".mjs", .type = "text/javascript; charset=UTF-8" },
			{ .ext = ".css", .type = "text/css; charset=UTF-8" },
			{ .ext = ".png", .type = "image/png" },
			{ .ext = ".json", .type = "application/json" },
			{ .ext = ".map", .type = "application/json" },
			{ .ext = ".svg", .type = "image/svg+xml" },
		};

		for (int i = 0; i < tf_countof(k_types); i++)
		{
			if (strcmp(ext, k_types[i].ext) == 0)
			{
				return k_types[i].type;
			}
		}
	}
	return use_fallback ? "application/binary" : NULL;
}

static void _httpd_request_finalizer(JSRuntime* runtime, JSValue value)
{
	tf_http_request_t* request = JS_GetOpaque(value, _httpd_request_class_id);
	tf_http_request_unref(request);
}

static void _httpd_endpoint_trace(tf_http_request_t* request)
{
	if (tf_httpd_redirect(request))
	{
		return;
	}

	tf_task_t* task = request->user_data;
	tf_trace_t* trace = tf_task_get_trace(task);
	char* json = tf_trace_export(trace);
	const char* headers[] = {
		"Content-Type",
		"application/json; charset=utf-8",
		"Access-Control-Allow-Origin",
		"*",
	};
	tf_http_respond(request, 200, headers, tf_countof(headers) / 2, json, json ? strlen(json) : 0);
	tf_free(json);
}

static void _httpd_endpoint_mem(tf_http_request_t* request)
{
	if (tf_httpd_redirect(request))
	{
		return;
	}

	char* response = NULL;
	size_t length = 0;

	int count = 0;
	tf_mem_allocation_t* alloc = tf_mem_summarize_allocations(&count);
	for (int i = 0; i < count; i++)
	{
		const char* stack = tf_util_backtrace_to_string(alloc[i].frames, alloc[i].frames_count);
		int line = snprintf(NULL, 0, "%zd bytes in %d allocations\n%s\n\n", alloc[i].size, alloc[i].count, stack);
		response = tf_resize_vec(response, length + line);
		snprintf(response + length, line, "%zd bytes in %d allocations\n%s\n\n", alloc[i].size, alloc[i].count, stack);
		length += line - 1;
		tf_free((void*)stack);
	}
	tf_free(alloc);

	const char* headers[] = {
		"Content-Type",
		"text/plain; charset=utf-8",
		"Access-Control-Allow-Origin",
		"*",
	};
	tf_http_respond(request, 200, headers, tf_countof(headers) / 2, response, length);
	tf_free(response);
}

static void _httpd_endpoint_add_slash(tf_http_request_t* request)
{
	const char* host = tf_http_request_get_header(request, "x-forwarded-host");
	if (!host)
	{
		host = tf_http_request_get_header(request, "host");
	}
	char url[1024];
	snprintf(url, sizeof(url), "%s%s%s/", "http://", host, request->path);
	const char* headers[] = {
		"Location",
		url,
	};
	tf_http_respond(request, 303, headers, tf_countof(headers) / 2, "", 0);
}

tf_httpd_user_app_t* tf_httpd_parse_user_app_from_path(const char* path, const char* expected_suffix)
{
	if (!path || path[0] != '/' || path[1] != '~')
	{
		return NULL;
	}

	size_t length = strlen(path);
	size_t suffix_length = expected_suffix ? strlen(expected_suffix) : 0;
	if (length < suffix_length || strcmp(path + length - suffix_length, expected_suffix) != 0)
	{
		return NULL;
	}

	const char* slash = strchr(path + 2, '/');
	if (!slash)
	{
		return NULL;
	}

	const char* user = path + 2;
	size_t user_length = (size_t)(slash - user);
	const char* app = slash + 1;
	size_t app_length = (size_t)(length - suffix_length - user_length - 3);
	tf_httpd_user_app_t* result = tf_malloc(sizeof(tf_httpd_user_app_t) + user_length + 1 + app_length + 1);

	*result = (tf_httpd_user_app_t) {
		.user = (char*)(result + 1),
		.app = (char*)(result + 1) + user_length + 1,
	};
	memcpy((char*)result->user, user, user_length);
	((char*)result->user)[user_length] = '\0';
	memcpy((char*)result->app, app, app_length);
	((char*)result->app)[app_length] = '\0';

	if (!tf_httpd_is_name_valid(result->user) || !tf_httpd_is_name_valid(result->app))
	{
		tf_free(result);
		result = NULL;
	}

	return result;
}

static void _httpd_endpoint_root_callback(const char* path, void* user_data)
{
	tf_http_request_t* request = user_data;
	const char* headers[] = {
		"Location",
		path ? path : "/~core/apps/",
	};
	tf_http_respond(request, 303, headers, tf_countof(headers) / 2, NULL, 0);
	tf_http_request_unref(request);
}

static void _httpd_endpoint_root(tf_http_request_t* request)
{
	const char* host = tf_http_request_get_header(request, "x-forwarded-host");
	if (!host)
	{
		host = tf_http_request_get_header(request, "host");
	}
	tf_task_t* task = request->user_data;
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	tf_http_request_ref(request);
	tf_ssb_db_resolve_index_async(ssb, host, _httpd_endpoint_root_callback, request);
}

static void _httpd_endpoint_robots_txt(tf_http_request_t* request)
{
	if (tf_httpd_redirect(request))
	{
		return;
	}
	char* response = "User-Agent: *\nDisallow: /*/*/edit\nAllow: /\n";
	const char* headers[] = { "Content-Type", "text/plain; charset=utf-8" };
	tf_http_respond(request, 200, headers, tf_countof(headers) / 2, response, response ? strlen(response) : 0);
}

static void _httpd_endpoint_debug(tf_http_request_t* request)
{
	if (tf_httpd_redirect(request))
	{
		return;
	}

	tf_task_t* task = request->user_data;
	char* response = tf_task_get_debug(task);
	const char* headers[] = {
		"Content-Type",
		"application/json; charset=utf-8",
		"Access-Control-Allow-Origin",
		"*",
	};
	tf_http_respond(request, 200, headers, tf_countof(headers) / 2, response, response ? strlen(response) : 0);
	tf_free(response);
}

const char** tf_httpd_form_data_decode(const char* data, int length)
{
	int key_max = 1;
	for (int i = 0; i < length; i++)
	{
		if (data[i] == '&')
		{
			key_max++;
		}
	}

	int write_length = length + 1;
	char** result = tf_malloc(sizeof(const char*) * (key_max + 1) * 2 + write_length);
	char* result_buffer = ((char*)result) + sizeof(const char*) * (key_max + 1) * 2;

	char* write_pos = result_buffer;
	int count = 0;
	int i = 0;
	while (i < length)
	{
		result[count++] = write_pos;
		while (i < length)
		{
			if (data[i] == '+')
			{
				*write_pos++ = ' ';
				i++;
			}
			else if (data[i] == '%' && i + 2 < length)
			{
				*write_pos++ = (char)strtoul((const char[]) { data[i + 1], data[i + 2], 0 }, NULL, 16);
				i += 3;
			}
			else if (data[i] == '=')
			{
				if (count % 2 == 0)
				{
					result[count++] = "";
				}
				i++;
				break;
			}
			else if (data[i] == '&')
			{
				if (count % 2 != 0)
				{
					result[count++] = "";
				}
				i++;
				break;
			}
			else
			{
				*write_pos++ = data[i++];
			}
		}
		*write_pos++ = '\0';
	}

	result[count++] = NULL;
	result[count++] = NULL;

	return (const char**)result;
}

const char* tf_httpd_form_data_get(const char** form_data, const char* key)
{
	for (int i = 0; form_data[i]; i += 2)
	{
		if (form_data[i] && strcmp(form_data[i], key) == 0)
		{
			return form_data[i + 1];
		}
	}
	return NULL;
}

static bool _string_property_equals(JSContext* context, JSValue object, const char* name, const char* value)
{
	JSValue object_value = JS_GetPropertyStr(context, object, name);
	const char* object_value_string = JS_ToCString(context, object_value);
	bool equals = object_value_string && strcmp(object_value_string, value) == 0;
	JS_FreeCString(context, object_value_string);
	JS_FreeValue(context, object_value);
	return equals;
}

JSValue tf_httpd_authenticate_jwt(tf_ssb_t* ssb, JSContext* context, const char* jwt)
{
	if (!jwt)
	{
		return JS_UNDEFINED;
	}

	int dot[2] = { 0 };
	int dot_count = 0;
	for (int i = 0; jwt[i]; i++)
	{
		if (jwt[i] == '.')
		{
			if (dot_count >= tf_countof(dot))
			{
				return JS_UNDEFINED;
			}
			dot[dot_count++] = i;
		}
	}
	if (dot_count != 2)
	{
		return JS_UNDEFINED;
	}

	uint8_t header[256] = { 0 };
	size_t actual_length = 0;
	if (sodium_base642bin(header, sizeof(header), jwt, dot[0], NULL, &actual_length, NULL, sodium_base64_VARIANT_URLSAFE_NO_PADDING) != 0 || actual_length >= sizeof(header))
	{
		return JS_UNDEFINED;
	}

	header[actual_length] = '\0';
	JSValue header_value = JS_ParseJSON(context, (const char*)header, actual_length, NULL);
	bool header_valid = _string_property_equals(context, header_value, "typ", "JWT") && _string_property_equals(context, header_value, "alg", "HS256");
	JS_FreeValue(context, header_value);
	if (!header_valid)
	{
		return JS_UNDEFINED;
	}

	char public_key_b64[k_id_base64_len] = { 0 };
	tf_ssb_whoami(ssb, public_key_b64, sizeof(public_key_b64));

	const char* payload = jwt + dot[0] + 1;
	size_t payload_length = dot[1] - dot[0] - 1;
	if (!tf_ssb_hmacsha256_verify(public_key_b64, payload, payload_length, jwt + dot[1] + 1, true))
	{
		return JS_UNDEFINED;
	}

	uint8_t payload_bin[256];
	size_t actual_payload_length = 0;
	if (sodium_base642bin(payload_bin, sizeof(payload_bin), payload, payload_length, NULL, &actual_payload_length, NULL, sodium_base64_VARIANT_URLSAFE_NO_PADDING) != 0 ||
		actual_payload_length >= sizeof(payload_bin))
	{
		return JS_UNDEFINED;
	}

	payload_bin[actual_payload_length] = '\0';
	JSValue parsed = JS_ParseJSON(context, (const char*)payload_bin, actual_payload_length, NULL);
	JSValue exp = JS_GetPropertyStr(context, parsed, "exp");
	int64_t exp_value = 0;
	JS_ToInt64(context, &exp_value, exp);

	uv_timespec64_t now = { 0 };
	uv_clock_gettime(UV_CLOCK_REALTIME, &now);
	if (now.tv_sec * 1000 + now.tv_nsec / 1000000LL >= exp_value)
	{
		JS_FreeValue(context, parsed);
		return JS_UNDEFINED;
	}

	return parsed;
}

bool tf_httpd_is_name_valid(const char* name)
{
	if (!name || !((*name >= 'a' && *name <= 'z') || (*name >= 'A' && *name <= 'Z')))
	{
		return false;
	}
	for (const char* p = name; *p; p++)
	{
		bool in_range = (*p >= 'a' && *p <= 'z') || (*p >= 'A' && *p <= 'Z') || (*p >= '0' && *p <= '9');
		if (!in_range)
		{
			return false;
		}
	}
	return true;
}

void tf_httpd_register(JSContext* context)
{
	JS_NewClassID(&_httpd_request_class_id);
	JSClassDef request_def = {
		.class_name = "Request",
		.finalizer = &_httpd_request_finalizer,
	};
	if (JS_NewClass(JS_GetRuntime(context), _httpd_request_class_id, &request_def) != 0)
	{
		fprintf(stderr, "Failed to register Request.\n");
	}

	JSValue global = JS_GetGlobalObject(context);
	JSValue httpd = JS_NewObject(context);
	JS_SetPropertyStr(context, httpd, "auth_query", JS_NewCFunction(context, _httpd_auth_query, "auth_query", 1));
	JS_SetPropertyStr(context, global, "httpd", httpd);
	JS_FreeValue(context, global);
}

tf_http_t* tf_httpd_create(JSContext* context)
{
	tf_task_t* task = tf_task_get(context);
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	uv_loop_t* loop = tf_task_get_loop(task);
	tf_http_t* http = tf_http_create(loop);
	tf_http_set_trace(http, tf_task_get_trace(task));

	int64_t http_port = 0;
	char out_http_port_file[512] = "";
	bool local_only = false;
	sqlite3* db = tf_ssb_acquire_db_reader(ssb);
	tf_ssb_db_get_global_setting_int64(db, "http_port", &http_port);
	tf_ssb_db_get_global_setting_string(db, "out_http_port_file", out_http_port_file, sizeof(out_http_port_file));
	tf_ssb_db_get_global_setting_bool(db, "http_local_only", &local_only);
	tf_ssb_release_db_reader(ssb, db);

	tf_http_add_handler(http, "/", _httpd_endpoint_root, NULL, task);
	tf_http_add_handler(http, "/codemirror/*", tf_httpd_endpoint_static, NULL, task);
	tf_http_add_handler(http, "/lit/*", tf_httpd_endpoint_static, NULL, task);
	tf_http_add_handler(http, "/prettier/*", tf_httpd_endpoint_static, NULL, task);
	tf_http_add_handler(http, "/speedscope/*", tf_httpd_endpoint_static, NULL, task);
	tf_http_add_handler(http, "/static/*", tf_httpd_endpoint_static, NULL, task);
	tf_http_add_handler(http, "/.well-known/*", tf_httpd_endpoint_static, NULL, task);
	tf_http_add_handler(http, "/&*.sha256", _httpd_endpoint_add_slash, NULL, task);
	tf_http_add_handler(http, "/&*.sha256/", tf_httpd_endpoint_static, NULL, task);
	tf_http_add_handler(http, "/&*.sha256/view", tf_httpd_endpoint_view, NULL, task);
	tf_http_add_handler(http, "/&*.sha256/*", tf_httpd_endpoint_app, NULL, task);
	tf_http_add_handler(http, "/~{word}/{word}", _httpd_endpoint_add_slash, NULL, task);
	tf_http_add_handler(http, "/~{word}/{word}/", tf_httpd_endpoint_app_index, NULL, task);
	tf_http_add_handler(http, "/~{word}/{word}/save", tf_httpd_endpoint_save, NULL, task);
	tf_http_add_handler(http, "/~{word}/{word}/delete", tf_httpd_endpoint_delete, NULL, task);
	tf_http_add_handler(http, "/~{word}/{word}/view", tf_httpd_endpoint_view, NULL, task);
	tf_http_add_handler(http, "/~{word}/{word}/*", tf_httpd_endpoint_app, NULL, task);
	tf_http_add_handler(http, "/save", tf_httpd_endpoint_save, NULL, task);

	tf_http_add_handler(http, "/robots.txt", _httpd_endpoint_robots_txt, NULL, NULL);
	tf_http_add_handler(http, "/debug", _httpd_endpoint_debug, NULL, task);
	tf_http_add_handler(http, "/mem", _httpd_endpoint_mem, NULL, task);
	tf_http_add_handler(http, "/trace", _httpd_endpoint_trace, NULL, task);

	tf_http_add_handler(http, "/login/logout", tf_httpd_endpoint_logout, NULL, task);
	tf_http_add_handler(http, "/login/auto", tf_httpd_endpoint_login_auto, NULL, task);
	tf_http_add_handler(http, "/login", tf_httpd_endpoint_login, NULL, task);

	tf_http_add_handler(http, "/app/socket", tf_httpd_endpoint_app_socket, NULL, task);

	if (http_port > 0 || *out_http_port_file)
	{
		httpd_listener_t* listener = tf_malloc(sizeof(httpd_listener_t));
		*listener = (httpd_listener_t) { 0 };
		int assigned_port = tf_http_listen(http, http_port, local_only, _httpd_listener_cleanup, listener);
		tf_printf(CYAN "~😎 Tilde Friends" RESET " " YELLOW VERSION_NUMBER RESET " is now up at " MAGENTA "http://127.0.0.1:%d/" RESET ".\n", assigned_port);

		if (*out_http_port_file)
		{
			const char* actual_http_port_file = tf_task_get_path_with_root(task, out_http_port_file);
			FILE* file = fopen(actual_http_port_file, "wb");
			if (file)
			{
				fprintf(file, "%d", assigned_port);
				fclose(file);
				tf_printf("Wrote the port file: %s.\n", out_http_port_file);
			}
			else
			{
				tf_printf("Failed to open %s for write: %s.\n", out_http_port_file, strerror(errno));
			}
			tf_free((char*)actual_http_port_file);
		}
	}
	return http;
}

void tf_httpd_destroy(tf_http_t* http)
{
	tf_http_destroy(http);
}

const char* tf_httpd_make_session_jwt(JSContext* context, tf_ssb_t* ssb, const char* name)
{
	if (!name || !*name)
	{
		return NULL;
	}

	uv_timespec64_t now = { 0 };
	uv_clock_gettime(UV_CLOCK_REALTIME, &now);

	const char* header_json = "{\"alg\":\"HS256\",\"typ\":\"JWT\"}";
	char header_base64[256];
	sodium_bin2base64(header_base64, sizeof(header_base64), (uint8_t*)header_json, strlen(header_json), sodium_base64_VARIANT_URLSAFE_NO_PADDING);

	JSValue payload = JS_NewObject(context);
	JS_SetPropertyStr(context, payload, "name", JS_NewString(context, name));
	JS_SetPropertyStr(context, payload, "exp", JS_NewInt64(context, now.tv_sec * 1000 + now.tv_nsec / 1000000LL + k_httpd_auth_refresh_interval));
	JSValue payload_json = JS_JSONStringify(context, payload, JS_NULL, JS_NULL);
	size_t payload_length = 0;
	const char* payload_string = JS_ToCStringLen(context, &payload_length, payload_json);
	char payload_base64[256];
	sodium_bin2base64(payload_base64, sizeof(payload_base64), (uint8_t*)payload_string, payload_length, sodium_base64_VARIANT_URLSAFE_NO_PADDING);

	char* result = NULL;
	uint8_t signature[crypto_sign_BYTES];
	unsigned long long signature_length = 0;
	char signature_base64[256] = { 0 };

	uint8_t private_key[crypto_sign_SECRETKEYBYTES] = { 0 };
	tf_ssb_get_private_key(ssb, private_key, sizeof(private_key));

	if (crypto_sign_detached(signature, &signature_length, (const uint8_t*)payload_base64, strlen(payload_base64), private_key) == 0)
	{
		sodium_bin2base64(signature_base64, sizeof(signature_base64), signature, sizeof(signature), sodium_base64_VARIANT_URLSAFE_NO_PADDING);
		size_t size = strlen(header_base64) + 1 + strlen(payload_base64) + 1 + strlen(signature_base64) + 1;
		result = tf_malloc(size);
		snprintf(result, size, "%s.%s.%s", header_base64, payload_base64, signature_base64);
	}
	sodium_memzero(private_key, sizeof(private_key));

	JS_FreeCString(context, payload_string);
	JS_FreeValue(context, payload_json);
	JS_FreeValue(context, payload);

	return result;
}
