#include "api.js.h"

#include "log.h"
#include "mem.h"
#include "ssb.db.h"
#include "ssb.h"
#include "task.h"
#include "util.js.h"

#include <quickjs.h>

typedef struct _app_path_pair_t
{
	const char* app;
	const char* path;
} app_path_pair_t;

typedef struct _get_apps_t
{
	app_path_pair_t* apps;
	int count;
	JSContext* context;
	JSValue promise[2];
	char user[];
} get_apps_t;

static void _tf_api_core_apps_work(tf_ssb_t* ssb, void* user_data)
{
	get_apps_t* work = user_data;

	JSMallocFunctions funcs = { 0 };
	tf_get_js_malloc_functions(&funcs);
	JSRuntime* runtime = JS_NewRuntime2(&funcs, NULL);
	JSContext* context = JS_NewContext(runtime);

	const char* apps = tf_ssb_db_get_property(ssb, work->user, "apps");
	if (apps)
	{
		JSValue apps_array = JS_ParseJSON(context, apps, strlen(apps), NULL);
		if (JS_IsArray(context, apps_array))
		{
			int length = tf_util_get_length(context, apps_array);
			for (int i = 0; i < length; i++)
			{
				JSValue name = JS_GetPropertyUint32(context, apps_array, i);
				const char* name_string = JS_ToCString(context, name);
				if (name_string)
				{
					work->apps = tf_resize_vec(work->apps, sizeof(app_path_pair_t) * (work->count + 1));
					work->apps[work->count].app = tf_strdup(name_string);
					size_t size = strlen("path:") + strlen(name_string) + 1;
					char* path_key = tf_malloc(size);
					snprintf(path_key, size, "path:%s", name_string);
					work->apps[work->count].path = tf_ssb_db_get_property(ssb, work->user, path_key);
					tf_free(path_key);
					work->count++;
				}
				JS_FreeCString(context, name_string);
				JS_FreeValue(context, name);
			}
		}
		JS_FreeValue(context, apps_array);
	}
	tf_free((void*)apps);

	JS_FreeContext(context);
	JS_FreeRuntime(runtime);
}

static void _tf_api_core_apps_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	get_apps_t* work = user_data;
	JSContext* context = work->context;
	JSValue result = JS_NewObject(context);
	for (int i = 0; i < work->count; i++)
	{
		JS_SetPropertyStr(context, result, work->apps[i].app, JS_NewString(context, work->apps[i].path ? work->apps[i].path : ""));
	}
	JSValue error = JS_Call(context, work->promise[0], JS_UNDEFINED, 1, &result);
	tf_util_report_error(context, error);
	JS_FreeValue(context, error);
	JS_FreeValue(context, result);
	JS_FreeValue(context, work->promise[0]);
	JS_FreeValue(context, work->promise[1]);
	for (int i = 0; i < work->count; i++)
	{
		tf_free((void*)work->apps[i].app);
		tf_free((void*)work->apps[i].path);
	}
	tf_free(work->apps);
	tf_free(work);
}

static JSValue _tf_api_core_apps(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	JSValue result = JS_UNDEFINED;
	JSValue user = argv[0];
	JSValue process = data[0];
	const char* user_string = JS_IsString(user) ? JS_ToCString(context, user) : NULL;

	if (JS_IsObject(process))
	{
		JSValue credentials = JS_GetPropertyStr(context, process, "credentials");
		if (JS_IsObject(credentials))
		{
			JSValue session = JS_GetPropertyStr(context, credentials, "session");
			if (JS_IsObject(session))
			{
				JSValue session_name = JS_GetPropertyStr(context, session, "name");
				const char* session_name_string = JS_IsString(session_name) ? JS_ToCString(context, session_name) : NULL;
				if (user_string && session_name_string && strcmp(user_string, session_name_string) && strcmp(user_string, "core"))
				{
					JS_FreeCString(context, user_string);
					user_string = NULL;
				}
				else if (!user_string)
				{
					user_string = session_name_string;
					session_name_string = NULL;
				}
				JS_FreeCString(context, session_name_string);
				JS_FreeValue(context, session_name);
			}
			JS_FreeValue(context, session);
		}
		JS_FreeValue(context, credentials);
	}

	if (user_string)
	{
		get_apps_t* work = tf_malloc(sizeof(get_apps_t) + strlen(user_string) + 1);
		*work = (get_apps_t) {
			.context = context,
		};
		memcpy(work->user, user_string, strlen(user_string) + 1);
		result = JS_NewPromiseCapability(context, work->promise);

		tf_task_t* task = tf_task_get(context);
		tf_ssb_t* ssb = tf_task_get_ssb(task);
		tf_ssb_run_work(ssb, _tf_api_core_apps_work, _tf_api_core_apps_after_work, work);
	}
	else
	{
		result = JS_NewObject(context);
	}
	JS_FreeCString(context, user_string);
	return result;
}

typedef struct _users_t
{
	const char* users;
	JSContext* context;
	JSValue promise[2];
} users_t;

static void _tf_api_core_users_work(tf_ssb_t* ssb, void* user_data)
{
	users_t* work = user_data;
	work->users = tf_ssb_db_get_property(ssb, "auth", "users");
}

static void _tf_api_core_users_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	users_t* work = user_data;
	JSContext* context = work->context;
	JSValue result = JS_UNDEFINED;
	if (work->users)
	{
		result = JS_ParseJSON(context, work->users, strlen(work->users), NULL);
		tf_free((void*)work->users);
	}
	if (JS_IsUndefined(result))
	{
		result = JS_NewArray(context);
	}
	JSValue error = JS_Call(context, JS_IsArray(context, result) ? work->promise[0] : work->promise[1], JS_UNDEFINED, 1, &result);
	tf_util_report_error(context, error);
	JS_FreeValue(context, error);
	JS_FreeValue(context, result);
	JS_FreeValue(context, work->promise[0]);
	JS_FreeValue(context, work->promise[1]);
	tf_free(work);
}

static JSValue _tf_api_core_users(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	tf_task_t* task = tf_task_get(context);
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	users_t* work = tf_malloc(sizeof(users_t));
	*work = (users_t) {
		.context = context,
	};
	JSValue result = JS_NewPromiseCapability(context, work->promise);
	tf_ssb_run_work(ssb, _tf_api_core_users_work, _tf_api_core_users_after_work, work);
	return result;
}

static JSValue _tf_api_core_register(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	JSValue event_name = argv[0];
	JSValue handler = argv[1];
	JSValue process = data[0];
	JSValue event_handlers = JS_GetPropertyStr(context, process, "eventHandlers");
	JSAtom atom = JS_ValueToAtom(context, event_name);
	JSValue array = JS_GetProperty(context, event_handlers, atom);
	if (!JS_IsArray(context, array))
	{
		JS_FreeValue(context, array);
		array = JS_NewArray(context);
		JS_SetProperty(context, event_handlers, atom, JS_DupValue(context, array));
	}
	JS_SetPropertyUint32(context, array, tf_util_get_length(context, array), JS_DupValue(context, handler));
	JS_FreeValue(context, array);
	JS_FreeAtom(context, atom);
	JS_FreeValue(context, event_handlers);
	return JS_UNDEFINED;
}

static JSValue _tf_api_core_unregister(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	JSValue event_name = argv[0];
	JSValue handler = argv[1];
	JSValue process = data[0];
	JSValue event_handlers = JS_GetPropertyStr(context, process, "eventHandlers");
	JSAtom atom = JS_ValueToAtom(context, event_name);
	JSValue array = JS_GetProperty(context, event_handlers, atom);

	if (JS_IsArray(context, array))
	{
		JSValue index_of = JS_GetPropertyStr(context, array, "indexOf");

		JSValue index = JS_Call(context, index_of, array, 1, &handler);
		int int_index = -1;
		JS_ToInt32(context, &int_index, index);
		if (int_index != -1)
		{
			JSValue splice = JS_GetPropertyStr(context, array, "splice");
			JSValue splice_args[] = {
				index,
				JS_NewInt32(context, 1),
			};
			JSValue result = JS_Call(context, splice, array, 2, splice_args);
			JS_FreeValue(context, result);
			JS_FreeValue(context, splice);
		}
		JS_FreeValue(context, index);
		JS_FreeValue(context, index_of);

		if (tf_util_get_length(context, array) == 0)
		{
			JS_DeleteProperty(context, event_handlers, atom, 0);
		}
	}

	JS_FreeValue(context, array);
	JS_FreeAtom(context, atom);
	JS_FreeValue(context, event_handlers);
	return JS_UNDEFINED;
}

typedef struct _permissions_for_user_t
{
	const char* user;
	const char* settings;
	JSContext* context;
	JSValue promise[2];
} permissions_for_user_t;

static void _tf_api_core_permissions_for_user_work(tf_ssb_t* ssb, void* user_data)
{
	permissions_for_user_t* work = user_data;
	work->settings = tf_ssb_db_get_property(ssb, "core", "settings");
}

static void _tf_api_core_permissions_for_user_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	permissions_for_user_t* work = user_data;
	JSContext* context = work->context;
	JSValue result = JS_UNDEFINED;
	if (work->settings)
	{
		JSValue json = JS_ParseJSON(context, work->settings, strlen(work->settings), NULL);
		if (JS_IsObject(json))
		{
			JSValue permissions = JS_GetPropertyStr(context, json, "permissions");
			if (JS_IsObject(permissions))
			{
				result = JS_GetPropertyStr(context, permissions, work->user);
			}
			JS_FreeValue(context, permissions);
		}
		JS_FreeValue(context, json);
		tf_free((void*)work->settings);
	}
	if (JS_IsUndefined(result))
	{
		result = JS_NewArray(context);
	}
	JSValue error = JS_Call(context, JS_IsArray(context, result) ? work->promise[0] : work->promise[1], JS_UNDEFINED, 1, &result);
	tf_util_report_error(context, error);
	JS_FreeValue(context, error);
	JS_FreeValue(context, result);
	JS_FreeValue(context, work->promise[0]);
	JS_FreeValue(context, work->promise[1]);
	JS_FreeCString(context, work->user);
	tf_free(work);
}

static JSValue _tf_api_core_permissionsForUser(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	tf_task_t* task = tf_task_get(context);
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	permissions_for_user_t* work = tf_malloc(sizeof(permissions_for_user_t));
	*work = (permissions_for_user_t) {
		.context = context,
		.user = JS_ToCString(context, argv[0]),
	};
	JSValue result = JS_NewPromiseCapability(context, work->promise);
	tf_ssb_run_work(ssb, _tf_api_core_permissions_for_user_work, _tf_api_core_permissions_for_user_after_work, work);
	return result;
}

typedef struct _permissions_granted_t
{
	JSContext* context;
	const char* user;
	const char* package_owner;
	const char* package_name;
	const char* settings;
	JSValue promise[2];
} permissions_granted_t;

static void _tf_api_core_permissions_granted_work(tf_ssb_t* ssb, void* user_data)
{
	permissions_granted_t* work = user_data;
	work->settings = tf_ssb_db_get_property(ssb, "core", "settings");
}

static void _tf_api_core_permissions_granted_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	permissions_granted_t* work = user_data;
	JSContext* context = work->context;
	JSValue result = JS_UNDEFINED;
	if (work->settings)
	{
		JSValue json = JS_ParseJSON(context, work->settings, strlen(work->settings), NULL);
		if (JS_IsObject(json) && work->user && work->package_owner && work->package_name)
		{
			JSValue user_permissions = JS_GetPropertyStr(context, json, "userPermissions");
			if (JS_IsObject(user_permissions))
			{
				JSValue user = JS_GetPropertyStr(context, user_permissions, work->user);
				if (JS_IsObject(user))
				{
					JSValue package_owner = JS_GetPropertyStr(context, user, work->package_owner);
					if (JS_IsObject(package_owner))
					{
						result = JS_GetPropertyStr(context, package_owner, work->package_name);
					}
					JS_FreeValue(context, package_owner);
				}
				JS_FreeValue(context, user);
			}
			JS_FreeValue(context, user_permissions);
		}
		JS_FreeValue(context, json);
		tf_free((void*)work->settings);
	}

	JSValue error = JS_Call(context, work->promise[0], JS_UNDEFINED, 1, &result);
	tf_util_report_error(context, error);
	JS_FreeValue(context, error);
	JS_FreeValue(context, result);
	JS_FreeValue(context, work->promise[0]);
	JS_FreeValue(context, work->promise[1]);
	tf_free((void*)work->user);
	tf_free((void*)work->package_owner);
	tf_free((void*)work->package_name);
	tf_free(work);
}

static const char* _tf_ssb_get_process_credentials_session_name(JSContext* context, JSValue process)
{
	JSValue credentials = JS_IsObject(process) ? JS_GetPropertyStr(context, process, "credentials") : JS_UNDEFINED;
	JSValue session = JS_IsObject(credentials) ? JS_GetPropertyStr(context, credentials, "session") : JS_UNDEFINED;
	JSValue name_value = JS_IsObject(session) ? JS_GetPropertyStr(context, session, "name") : JS_UNDEFINED;
	const char* name = JS_IsString(name_value) ? JS_ToCString(context, name_value) : NULL;
	const char* result = tf_strdup(name);
	JS_FreeCString(context, name);
	JS_FreeValue(context, name_value);
	JS_FreeValue(context, session);
	JS_FreeValue(context, credentials);
	return result;
}

static JSValue _tf_api_core_permissionsGranted(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	tf_task_t* task = tf_task_get(context);
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	JSValue process = data[0];
	JSValue package_owner_value = JS_GetPropertyStr(context, process, "packageOwner");
	JSValue package_name_value = JS_GetPropertyStr(context, process, "packageName");
	const char* package_owner = JS_ToCString(context, package_owner_value);
	const char* package_name = JS_ToCString(context, package_name_value);
	permissions_granted_t* work = tf_malloc(sizeof(permissions_granted_t));
	*work = (permissions_granted_t) {
		.context = context,
		.user = _tf_ssb_get_process_credentials_session_name(context, process),
		.package_owner = tf_strdup(package_owner),
		.package_name = tf_strdup(package_name),
	};
	JS_FreeCString(context, package_owner);
	JS_FreeCString(context, package_name);
	JS_FreeValue(context, package_owner_value);
	JS_FreeValue(context, package_name_value);
	JSValue result = JS_NewPromiseCapability(context, work->promise);
	tf_ssb_run_work(ssb, _tf_api_core_permissions_granted_work, _tf_api_core_permissions_granted_after_work, work);
	return result;
}

typedef struct _active_identity_work_t
{
	JSContext* context;
	const char* name;
	const char* package_owner;
	const char* package_name;
	char identity[k_id_base64_len];
	int result;
	JSValue promise[2];
} active_identity_work_t;

static void _tf_ssb_getActiveIdentity_visit(const char* identity, void* user_data)
{
	active_identity_work_t* request = user_data;
	if (!*request->identity)
	{
		snprintf(request->identity, sizeof(request->identity), "@%s", identity);
	}
}

static void _tf_ssb_getActiveIdentity_work(tf_ssb_t* ssb, void* user_data)
{
	active_identity_work_t* request = user_data;
	sqlite3* db = tf_ssb_acquire_db_reader(ssb);
	tf_ssb_db_identity_get_active(db, request->name, request->package_owner, request->package_name, request->identity, sizeof(request->identity));
	tf_ssb_release_db_reader(ssb, db);

	if (!*request->identity)
	{
		tf_ssb_db_identity_visit(ssb, request->name, _tf_ssb_getActiveIdentity_visit, request);
	}

	if (!*request->identity && tf_ssb_db_user_has_permission(ssb, NULL, request->name, "administration"))
	{
		tf_ssb_whoami(ssb, request->identity, sizeof(request->identity));
	}
}

static void _tf_ssb_getActiveIdentity_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	active_identity_work_t* request = user_data;
	JSContext* context = request->context;
	if (request->result == 0)
	{
		JSValue identity = JS_NewString(context, request->identity);
		JSValue error = JS_Call(context, request->promise[0], JS_UNDEFINED, 1, &identity);
		JS_FreeValue(context, identity);
		tf_util_report_error(context, error);
		JS_FreeValue(context, error);
	}
	else
	{
		JSValue error = JS_Call(context, request->promise[1], JS_UNDEFINED, 0, NULL);
		tf_util_report_error(context, error);
		JS_FreeValue(context, error);
	}
	JS_FreeValue(context, request->promise[0]);
	JS_FreeValue(context, request->promise[1]);
	tf_free((void*)request->name);
	tf_free((void*)request->package_owner);
	tf_free((void*)request->package_name);
	tf_free(request);
}

static JSValue _tf_ssb_getActiveIdentity(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	tf_task_t* task = tf_task_get(context);
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	JSValue process = data[0];
	JSValue package_owner_value = JS_GetPropertyStr(context, process, "packageOwner");
	JSValue package_name_value = JS_GetPropertyStr(context, process, "packageName");

	const char* name = _tf_ssb_get_process_credentials_session_name(context, process);
	const char* package_owner = JS_ToCString(context, package_owner_value);
	const char* package_name = JS_ToCString(context, package_name_value);
	active_identity_work_t* work = tf_malloc(sizeof(active_identity_work_t));
	*work = (active_identity_work_t) {
		.context = context,
		.name = tf_strdup(name),
		.package_owner = tf_strdup(package_owner),
		.package_name = tf_strdup(package_name),
	};
	JSValue result = JS_NewPromiseCapability(context, work->promise);
	tf_free((void*)name);
	JS_FreeCString(context, package_owner);
	JS_FreeCString(context, package_name);

	JS_FreeValue(context, package_owner_value);
	JS_FreeValue(context, package_name_value);
	tf_ssb_run_work(ssb, _tf_ssb_getActiveIdentity_work, _tf_ssb_getActiveIdentity_after_work, work);
	return result;
}

typedef struct _identities_visit_t
{
	JSContext* context;
	JSValue promise[2];
	const char** identities;
	int count;
	char user[];
} identities_visit_t;

static void _tf_ssb_getIdentities_visit(const char* identity, void* user_data)
{
	identities_visit_t* work = user_data;
	work->identities = tf_resize_vec(work->identities, (work->count + 1) * sizeof(const char*));
	char id[k_id_base64_len];
	snprintf(id, sizeof(id), "@%s", identity);
	work->identities[work->count++] = tf_strdup(id);
}

static void _tf_ssb_get_all_identities_work(tf_ssb_t* ssb, void* user_data)
{
	tf_ssb_db_identity_visit_all(ssb, _tf_ssb_getIdentities_visit, user_data);
}

static void _tf_ssb_get_identities_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	identities_visit_t* work = user_data;
	JSContext* context = tf_ssb_get_context(ssb);
	JSValue result = JS_NewArray(context);
	for (int i = 0; i < work->count; i++)
	{
		JS_SetPropertyUint32(context, result, i, JS_NewString(context, work->identities[i]));
		tf_free((void*)work->identities[i]);
	}
	tf_free(work->identities);

	JSValue error = JS_Call(context, work->promise[0], JS_UNDEFINED, 1, &result);
	JS_FreeValue(context, result);
	tf_util_report_error(context, error);
	JS_FreeValue(context, error);
	JS_FreeValue(context, work->promise[0]);
	JS_FreeValue(context, work->promise[1]);
	tf_free(work);
}

static JSValue _tf_ssb_getAllIdentities(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	JSValue result = JS_UNDEFINED;
	tf_task_t* task = tf_task_get(context);
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	if (ssb)
	{
		identities_visit_t* work = tf_malloc(sizeof(identities_visit_t));
		*work = (identities_visit_t) {
			.context = context,
		};

		result = JS_NewPromiseCapability(context, work->promise);
		tf_ssb_run_work(ssb, _tf_ssb_get_all_identities_work, _tf_ssb_get_identities_after_work, work);
	}
	return result;
}

static void _tf_ssb_get_identities_work(tf_ssb_t* ssb, void* user_data)
{
	identities_visit_t* work = user_data;
	if (tf_ssb_db_user_has_permission(ssb, NULL, work->user, "administration"))
	{
		char id[k_id_base64_len] = "";
		if (tf_ssb_whoami(ssb, id, sizeof(id)))
		{
			_tf_ssb_getIdentities_visit(*id == '@' ? id + 1 : id, work);
		}
	}
	tf_ssb_db_identity_visit(ssb, work->user, _tf_ssb_getIdentities_visit, user_data);
}

static JSValue _tf_ssb_getIdentities(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	JSValue result = JS_UNDEFINED;
	tf_task_t* task = tf_task_get(context);
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	JSValue process = data[0];
	if (ssb)
	{
		const char* user = _tf_ssb_get_process_credentials_session_name(context, process);
		if (user)
		{
			size_t user_length = user ? strlen(user) : 0;
			identities_visit_t* work = tf_malloc(sizeof(identities_visit_t) + user_length + 1);
			*work = (identities_visit_t) {
				.context = context,
			};
			memcpy(work->user, user, user_length + 1);
			tf_free((void*)user);

			result = JS_NewPromiseCapability(context, work->promise);
			tf_ssb_run_work(ssb, _tf_ssb_get_identities_work, _tf_ssb_get_identities_after_work, work);
		}
	}
	return result;
}

static JSValue _tf_ssb_getOwnerIdentities(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	JSValue result = JS_UNDEFINED;
	tf_task_t* task = tf_task_get(context);
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	JSValue process = data[0];
	if (ssb)
	{
		JSValue value = JS_GetPropertyStr(context, process, "packageOwner");
		const char* user = JS_ToCString(context, value);
		size_t user_length = user ? strlen(user) : 0;
		identities_visit_t* work = tf_malloc(sizeof(identities_visit_t) + user_length + 1);
		*work = (identities_visit_t) {
			.context = context,
		};
		memcpy(work->user, user, user_length + 1);
		JS_FreeCString(context, user);
		JS_FreeValue(context, value);

		result = JS_NewPromiseCapability(context, work->promise);
		tf_ssb_run_work(ssb, _tf_ssb_get_identities_work, _tf_ssb_get_identities_after_work, work);
	}
	return result;
}

typedef struct _settings_descriptions_get_t
{
	const char* settings;
	JSContext* context;
	JSValue promise[2];
} settings_descriptions_get_t;

static void _tf_ssb_get_settings_descriptions_work(tf_ssb_t* ssb, void* user_data)
{
	settings_descriptions_get_t* work = user_data;
	work->settings = tf_ssb_db_get_property(ssb, "core", "settings");
}

static void _tf_ssb_get_settings_descriptions_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	settings_descriptions_get_t* work = user_data;
	JSContext* context = work->context;
	JSValue result = JS_NewObject(context);
	JSValue settings = JS_ParseJSON(context, work->settings ? work->settings : "", work->settings ? strlen(work->settings) : 0, NULL);
	const char* name;
	const char* type;
	tf_setting_kind_t kind;
	const char* description;
	for (int i = 0; tf_util_get_global_setting_by_index(i, &name, &type, &kind, &description); i++)
	{
		JSValue entry = JS_NewObject(context);
		JS_SetPropertyStr(context, entry, "type", JS_NewString(context, type));
		JS_SetPropertyStr(context, entry, "description", JS_NewString(context, description));
		switch (kind)
		{
		case k_kind_unknown:
			break;
		case k_kind_bool:
			JS_SetPropertyStr(context, entry, "default_value", JS_NewBool(context, tf_util_get_default_global_setting_bool(name)));
			break;
		case k_kind_int:
			JS_SetPropertyStr(context, entry, "default_value", JS_NewInt32(context, tf_util_get_default_global_setting_int(name)));
			break;
		case k_kind_string:
			JS_SetPropertyStr(context, entry, "default_value", JS_NewString(context, tf_util_get_default_global_setting_string(name)));
			break;
		}
		if (JS_IsObject(settings))
		{
			JS_SetPropertyStr(context, entry, "value", JS_GetPropertyStr(context, settings, name));
		}
		JS_SetPropertyStr(context, result, name, entry);
	}
	JS_FreeValue(context, settings);
	JSValue error = JS_Call(context, work->promise[0], JS_UNDEFINED, 1, &result);
	tf_util_report_error(context, error);
	JS_FreeValue(context, error);
	JS_FreeValue(context, result);
	JS_FreeValue(context, work->promise[0]);
	JS_FreeValue(context, work->promise[1]);
	tf_free((void*)work->settings);
	tf_free(work);
}

static JSValue _tf_ssb_globalSettingsDescriptions(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	tf_task_t* task = tf_task_get(context);
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	settings_descriptions_get_t* work = tf_malloc(sizeof(settings_descriptions_get_t));
	*work = (settings_descriptions_get_t) { .context = context };
	JSValue result = JS_NewPromiseCapability(context, work->promise);
	tf_ssb_run_work(ssb, _tf_ssb_get_settings_descriptions_work, _tf_ssb_get_settings_descriptions_after_work, work);
	return result;
}

typedef struct _settings_get_t
{
	const char* key;
	tf_setting_kind_t kind;
	void* value;
	JSContext* context;
	JSValue promise[2];
} settings_get_t;

static void _tf_ssb_settings_get_work(tf_ssb_t* ssb, void* user_data)
{
	settings_get_t* work = user_data;
	work->kind = tf_util_get_global_setting_kind(work->key);
	switch (work->kind)
	{
	case k_kind_unknown:
		break;
	case k_kind_bool:
		{
			sqlite3* db = tf_ssb_acquire_db_reader(ssb);
			bool value = false;
			tf_ssb_db_get_global_setting_bool(db, work->key, &value);
			work->value = (void*)(intptr_t)value;
			tf_ssb_release_db_reader(ssb, db);
		}
		break;
	case k_kind_int:
		{
			sqlite3* db = tf_ssb_acquire_db_reader(ssb);
			int64_t value = 0;
			tf_ssb_db_get_global_setting_int64(db, work->key, &value);
			work->value = (void*)(intptr_t)value;
			tf_ssb_release_db_reader(ssb, db);
		}
		break;
	case k_kind_string:
		{
			sqlite3* db = tf_ssb_acquire_db_reader(ssb);
			work->value = (void*)tf_ssb_db_get_global_setting_string_alloc(db, work->key);
			tf_ssb_release_db_reader(ssb, db);
		}
		break;
	}
}

static void _tf_ssb_settings_get_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	settings_get_t* work = user_data;
	JSContext* context = tf_ssb_get_context(ssb);
	JSValue result = JS_UNDEFINED;
	switch (work->kind)
	{
	case k_kind_unknown:
		break;
	case k_kind_bool:
		result = work->value ? JS_TRUE : JS_FALSE;
		break;
	case k_kind_int:
		result = JS_NewInt64(context, (int64_t)(intptr_t)work->value);
		break;
	case k_kind_string:
		result = JS_NewString(context, (const char*)work->value);
		tf_free(work->value);
		break;
	}

	JSValue error = JS_Call(context, work->promise[0], JS_UNDEFINED, 1, &result);
	tf_util_report_error(context, error);
	JS_FreeValue(context, error);
	JS_FreeValue(context, result);
	JS_FreeValue(context, work->promise[0]);
	JS_FreeValue(context, work->promise[1]);
	JS_FreeCString(context, work->key);
	tf_free(work);
}

static JSValue _tf_ssb_globalSettingsGet(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	tf_task_t* task = tf_task_get(context);
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	settings_get_t* work = tf_malloc(sizeof(settings_get_t));
	*work = (settings_get_t) { .context = context, .key = JS_ToCString(context, argv[0]) };
	JSValue result = JS_NewPromiseCapability(context, work->promise);
	tf_ssb_run_work(ssb, _tf_ssb_settings_get_work, _tf_ssb_settings_get_after_work, work);
	return result;
}

static JSValue _tf_api_register_imports(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	JSValue imports = argv[0];
	JSValue process = argv[1];
	JSValue core = JS_GetPropertyStr(context, imports, "core");
	JS_SetPropertyStr(context, core, "apps", JS_NewCFunctionData(context, _tf_api_core_apps, 1, 0, 1, &process));
	JS_SetPropertyStr(context, core, "register", JS_NewCFunctionData(context, _tf_api_core_register, 2, 0, 1, &process));
	JS_SetPropertyStr(context, core, "unregister", JS_NewCFunctionData(context, _tf_api_core_unregister, 2, 0, 1, &process));

	JS_SetPropertyStr(context, core, "users", JS_NewCFunctionData(context, _tf_api_core_users, 0, 0, 1, &process));

	JS_SetPropertyStr(context, core, "permissionsForUser", JS_NewCFunctionData(context, _tf_api_core_permissionsForUser, 1, 0, 1, &process));
	JS_SetPropertyStr(context, core, "permissionsGranted", JS_NewCFunctionData(context, _tf_api_core_permissionsGranted, 0, 0, 1, &process));

	JSValue app = JS_NewObject(context);
	JS_SetPropertyStr(context, app, "owner", JS_GetPropertyStr(context, process, "packageOwner"));
	JS_SetPropertyStr(context, app, "name", JS_GetPropertyStr(context, process, "packageName"));
	JS_SetPropertyStr(context, core, "app", app);

	JS_SetPropertyStr(context, core, "url", JS_GetPropertyStr(context, process, "url"));

	JSValue ssb = JS_GetPropertyStr(context, imports, "ssb");
	JS_SetPropertyStr(context, ssb, "getAllIdentities", JS_NewCFunction(context, _tf_ssb_getAllIdentities, "getAllIdentities", 0));
	JS_SetPropertyStr(context, ssb, "getActiveIdentity", JS_NewCFunctionData(context, _tf_ssb_getActiveIdentity, 0, 0, 1, &process));
	JS_SetPropertyStr(context, ssb, "getIdentities", JS_NewCFunctionData(context, _tf_ssb_getIdentities, 0, 0, 1, &process));
	JS_SetPropertyStr(context, ssb, "getOwnerIdentities", JS_NewCFunctionData(context, _tf_ssb_getOwnerIdentities, 0, 0, 1, &process));
	JS_FreeValue(context, ssb);

	JSValue credentials = JS_GetPropertyStr(context, process, "credentials");
	JSValue permissions = JS_IsObject(credentials) ? JS_GetPropertyStr(context, credentials, "permissions") : JS_UNDEFINED;
	JSValue administration = JS_IsObject(permissions) ? JS_GetPropertyStr(context, permissions, "administration") : JS_UNDEFINED;
	if (JS_ToBool(context, administration) > 0)
	{
		JS_SetPropertyStr(context, core, "globalSettingsDescriptions", JS_NewCFunction(context, _tf_ssb_globalSettingsDescriptions, "globalSettingsDescriptions", 0));
		JS_SetPropertyStr(context, core, "globalSettingsGet", JS_NewCFunction(context, _tf_ssb_globalSettingsGet, "globalSettingsGet", 1));
	}
	JS_FreeValue(context, administration);
	JS_FreeValue(context, permissions);
	JS_FreeValue(context, credentials);

	JS_FreeValue(context, core);
	return JS_UNDEFINED;
}

void tf_api_register(JSContext* context)
{
	JSValue global = JS_GetGlobalObject(context);
	JSValue ssb = JS_GetPropertyStr(context, global, "ssb");
	JS_SetPropertyStr(context, ssb, "registerImports", JS_NewCFunction(context, _tf_api_register_imports, "registerImports", 2));
	JS_FreeValue(context, ssb);
	JS_FreeValue(context, global);
}
