import {LitElement, html, keyed, live} from './lit-all.min.js';
import * as tfrpc from '/static/tfrpc.js';

class TfJournalAppElement extends LitElement {
	static get properties() {
		return {
			ids: {type: Array},
			owner_ids: {type: Array},
			whoami: {type: String},
			journals: {type: Object},
		};
	}

	constructor() {
		super();
		this.ids = [];
		this.owner_ids = [];
		this.journals = {};
		this.load();
	}

	async load() {
		this.ids = await tfrpc.rpc.getIdentities();
		this.whoami = await tfrpc.rpc.localStorageGet('journal_whoami');
		await this.read_journals();
	}

	async read_journals() {
		let max_rowid;
		let journals;
		while (true) {
			[max_rowid, journals] = await tfrpc.rpc.collection(
				[this.whoami],
				'journal-entry',
				undefined,
				max_rowid,
				journals
			);
			this.journals = Object.assign({}, journals);
			console.log('JOURNALS', this.journals);
		}
	}

	async on_whoami_changed(event) {
		let new_id = event.srcElement.selected;
		await tfrpc.rpc.localStorageSet('journal_whoami', new_id);
		this.whoami = new_id;
	}

	async on_journal_publish(event) {
		let key = event.detail.key;
		let text = event.detail.text;
		let message = {
			type: 'journal-entry',
			key: key,
			text: text,
		};
		message.recps = [this.whoami];
		print(message);
		message = await tfrpc.rpc.encrypt(
			this.whoami,
			message.recps,
			JSON.stringify(message)
		);
		print(message);
		await tfrpc.rpc.appendMessage(this.whoami, message);
	}

	render() {
		console.log('RENDER APP', this.journals);
		let self = this;
		return html`
			<div>
				<tf-id-picker
					.ids=${this.ids}
					selected=${this.whoami}
					@change=${this.on_whoami_changed}
				></tf-id-picker>
			</div>
			<tf-journal-entry
				whoami=${this.whoami}
				.journals=${this.journals}
				@publish=${this.on_journal_publish}
			></tf-journal-entry>
		`;
	}
}

customElements.define('tf-journal-app', TfJournalAppElement);
