import * as linkify from './commonmark-linkify.js';

var reUnsafeProtocol = /^javascript:|vbscript:|file:|data:/i;
var reSafeDataProtocol = /^data:image\/(?:png|gif|jpeg|webp)/i;
var potentiallyUnsafe = function (url) {
	return reUnsafeProtocol.test(url) && !reSafeDataProtocol.test(url);
};

function image(node, entering) {
	if (
		node.firstChild?.type === 'text' &&
		node.firstChild.literal.startsWith('video:')
	) {
		if (entering) {
			this.lit(
				'<video style="max-width: 100%; max-height: 480px" title="' +
					this.esc(node.firstChild?.literal) +
					'" controls>'
			);
			this.lit('<source src="' + this.esc(node.destination) + '"></source>');
			this.disableTags += 1;
		} else {
			this.disableTags -= 1;
			this.lit('</video>');
		}
	} else if (
		node.firstChild?.type === 'text' &&
		node.firstChild.literal.startsWith('audio:')
	) {
		if (entering) {
			this.lit(
				'<audio style="height: 32px; max-width: 100%" title="' +
					this.esc(node.firstChild?.literal) +
					'" controls>'
			);
			this.lit('<source src="' + this.esc(node.destination) + '"></source>');
			this.disableTags += 1;
		} else {
			this.disableTags -= 1;
			this.lit('</audio>');
		}
	} else {
		if (entering) {
			if (this.disableTags === 0) {
				this.lit(
					'<div class="img_caption">' +
						this.esc(node.firstChild?.literal || node.destination) +
						'</div>'
				);
				if (this.options.safe && potentiallyUnsafe(node.destination)) {
					this.lit('<img src="" alt="');
				} else {
					this.lit('<img src="' + this.esc(node.destination) + '" alt="');
				}
			}
			this.disableTags += 1;
		} else {
			this.disableTags -= 1;
			if (this.disableTags === 0) {
				if (node.title) {
					this.lit('" title="' + this.esc(node.title));
				}
				this.lit('" />');
			}
		}
	}
}

export function markdown(md) {
	var reader = new commonmark.Parser();
	var writer = new commonmark.HtmlRenderer({safe: true});
	writer.image = image;
	var parsed = reader.parse(md || '');
	parsed = linkify.transform(parsed);
	var walker = parsed.walker();
	var event, node;
	while ((event = walker.next())) {
		node = event.node;
		if (event.entering) {
			if (node.type == 'link') {
				if (
					node.destination.startsWith('@') &&
					node.destination.endsWith('.ed25519')
				) {
					node.destination = '#' + node.destination;
				} else if (
					node.destination.startsWith('%') &&
					node.destination.endsWith('.sha256')
				) {
					node.destination = '#' + node.destination;
				} else if (
					node.destination.startsWith('&') &&
					node.destination.endsWith('.sha256')
				) {
					node.destination = '/' + node.destination + '/view';
				}
			} else if (node.type == 'image') {
				if (node.destination.startsWith('&')) {
					node.destination = '/' + node.destination + '/view';
				}
			}
		}
	}
	return writer.render(parsed);
}

export function human_readable_size(bytes) {
	let v = bytes;
	let u = 'B';
	for (let unit of ['kB', 'MB', 'GB']) {
		if (v > 1024) {
			v /= 1024;
			u = unit;
		} else {
			break;
		}
	}
	return `${Math.round(v * 10) / 10} ${u}`;
}
