import 'dart:async';

import 'package:chopper/chopper.dart';
import 'package:http/http.dart' as http;
import 'package:http/testing.dart';
import 'package:test/test.dart';

import 'test_service.dart';

void main() {
  group('Interceptors', () {
    final requestClient = MockClient(
      (request) async {
        expect(
          request.url.toString(),
          equals('/test/get/1234/intercept'),
        );

        return http.Response('', 200);
      },
    );

    final responseClient = MockClient(
      (request) async => http.Response('body', 200),
    );

    tearDown(() {
      requestClient.close();
      responseClient.close();
    });

    test('RequestInterceptor', () async {
      final chopper = ChopperClient(
        interceptors: [RequestIntercept()],
        services: [
          HttpTestService.create(),
        ],
        client: requestClient,
      );

      await chopper.getService<HttpTestService>().getTest(
            '1234',
            dynamicHeader: '',
          );
    });

    test('RequestInterceptorFunc', () async {
      final chopper = ChopperClient(
        interceptors: [
          (Request request) => request.copyWith(
                uri: request.uri.replace(path: '${request.uri.path}/intercept'),
              ),
        ],
        services: [
          HttpTestService.create(),
        ],
        client: requestClient,
      );

      await chopper.getService<HttpTestService>().getTest(
            '1234',
            dynamicHeader: '',
          );
    });

    test('ResponseInterceptor', () async {
      final chopper = ChopperClient(
        interceptors: [ResponseIntercept()],
        services: [
          HttpTestService.create(),
        ],
        client: responseClient,
      );

      await chopper.getService<HttpTestService>().getTest(
            '1234',
            dynamicHeader: '',
          );

      expect(ResponseIntercept.intercepted, isA<_Intercepted>());
    });

    test('ResponseInterceptorFunc', () async {
      dynamic intercepted;

      final chopper = ChopperClient(
        interceptors: [
          (Response response) {
            intercepted = _Intercepted(response.body);

            return response;
          },
        ],
        services: [
          HttpTestService.create(),
        ],
        client: responseClient,
      );

      await chopper.getService<HttpTestService>().getTest(
            '1234',
            dynamicHeader: '',
          );

      expect(intercepted, isA<_Intercepted<dynamic>>());
    });

    test('TypedResponseInterceptorFunc1', () async {
      dynamic intercepted;

      final chopper = ChopperClient(
        interceptors: [
          <BodyType>(Response<BodyType> response) {
            intercepted = _Intercepted(response.body);

            return response;
          },
        ],
        services: [
          HttpTestService.create(),
        ],
        client: responseClient,
      );

      await chopper.getService<HttpTestService>().getTest(
            '1234',
            dynamicHeader: '',
          );

      expect(intercepted, isA<_Intercepted<String?>>());
    });

    test('TypedResponseInterceptorFunc2', () async {
      final client = MockClient((http.Request req) async {
        return http.Response('["1","2"]', 200);
      });

      dynamic intercepted;

      final chopper = ChopperClient(
        client: client,
        converter: JsonConverter(),
        interceptors: [
          <BodyType, InnerType>(Response<BodyType> response) {
            expect(isTypeOf<String, InnerType>(), isTrue);
            expect(isTypeOf<BodyType, List<String>>(), isTrue);
            intercepted = _Intercepted<BodyType>(response.body as BodyType);

            return response;
          },
        ],
        services: [
          HttpTestService.create(),
        ],
      );

      await chopper.getService<HttpTestService>().listString();

      expect(intercepted, isA<_Intercepted<List<String>>>());
    });

    test('headers', () async {
      final client = MockClient((http.Request req) async {
        expect(req.headers.containsKey('foo'), isTrue);
        expect(req.headers['foo'], equals('bar'));

        return http.Response('', 200);
      });

      final chopper = ChopperClient(
        interceptors: [
          HeadersInterceptor({'foo': 'bar'}),
        ],
        services: [
          HttpTestService.create(),
        ],
        client: client,
      );

      await chopper.getService<HttpTestService>().getTest(
            '1234',
            dynamicHeader: '',
          );
    });

    final fakeRequest = Request(
      'POST',
      Uri.parse('/'),
      Uri.parse('base'),
      body: 'test',
      headers: {'foo': 'bar'},
    );

    test('Curl interceptors', () async {
      final curl = CurlInterceptor();
      var log = '';
      chopperLogger.onRecord.listen((r) => log = r.message);
      await curl.onRequest(fakeRequest);

      expect(
        log,
        equals(
          "curl -v -X POST -H 'foo: bar' -H 'content-type: text/plain; charset=utf-8' -d 'test' \"base/\"",
        ),
      );
    });

    final fakeRequestMultipart = Request(
      'POST',
      Uri.parse('/'),
      Uri.parse('base'),
      headers: {'foo': 'bar'},
      parts: [
        PartValue<int>('p1', 123),
        PartValueFile<http.MultipartFile>(
          'p2',
          http.MultipartFile.fromBytes('file', [0], filename: 'filename'),
        ),
      ],
      multipart: true,
    );

    test('Curl interceptors Multipart', () async {
      final curl = CurlInterceptor();
      var log = '';
      chopperLogger.onRecord.listen((r) => log = r.message);
      await curl.onRequest(fakeRequestMultipart);

      expect(
        log,
        equals(
          "curl -v -X POST -H 'foo: bar' -f 'p1: 123' -f 'file: filename' \"base/\"",
        ),
      );
    });
  });
}

class ResponseIntercept implements ResponseInterceptor {
  static dynamic intercepted;

  @override
  FutureOr<Response> onResponse(Response response) {
    intercepted = _Intercepted(response.body);

    return response;
  }
}

class RequestIntercept implements RequestInterceptor {
  @override
  FutureOr<Request> onRequest(Request request) => request.copyWith(
        uri: request.uri.replace(path: '${request.uri}/intercept'),
      );
}

class _Intercepted<BodyType> {
  final BodyType body;

  _Intercepted(this.body);
}
