package com.unciv.ui.components.extensions

import com.badlogic.gdx.graphics.Color
import com.badlogic.gdx.scenes.scene2d.ui.Table
import com.unciv.models.UncivSound
import com.unciv.models.metadata.GameSettings
import com.unciv.models.translations.tr
import com.unciv.ui.audio.MusicController
import com.unciv.ui.audio.MusicTrackChooserFlags
import com.unciv.ui.components.input.onClick
import com.unciv.ui.components.widgets.UncivSlider
import com.unciv.ui.components.widgets.WrappableLabel
import com.unciv.ui.popups.Popup
import kotlin.math.floor

/**
 *  Library of extensions adding sound controls to a Table.
 *
 *  Used in [OptionsPopup][com.unciv.ui.popups.options.OptionsPopup].[SoundTab][com.unciv.ui.popups.options.SoundTab] and [WorldScreenMusicPopup][com.unciv.ui.screens.worldscreen.mainmenu.WorldScreenMusicPopup].
 *
 *  @see [addMusicControls]
 */
interface MusicControls {
    fun Table.addVolumeSlider(text: String, initial: Float, silent: Boolean = false, onChange: (Float) -> Unit) {
        add(text.tr()).left().fillX()

        val volumeSlider = UncivSlider(
            0f, 1.0f, 0.05f,
            initial = initial,
            sound = if (silent) UncivSound.Silent else UncivSound.Slider,
            getTipText = UncivSlider::formatPercent,
            onChange = onChange
        )
        add(volumeSlider).pad(5f).row()
    }

    fun Table.addSoundEffectsVolumeSlider(settings: GameSettings) =
        addVolumeSlider("Sound effects volume", settings.soundEffectsVolume) {
            settings.soundEffectsVolume = it
        }

    fun Table.addCitySoundsVolumeSlider(settings: GameSettings) =
        addVolumeSlider("City ambient sound volume", settings.citySoundsVolume) {
            settings.citySoundsVolume = it
        }

    fun Table.addVoicesVolumeSlider(settings: GameSettings) =
        addVolumeSlider("Leader voices volume", settings.voicesVolume) {
            settings.voicesVolume = it
        }

    fun Table.addMusicVolumeSlider(settings: GameSettings, music: MusicController) =
        addVolumeSlider("Music volume".tr(), settings.musicVolume, true) {
            settings.musicVolume = it

            music.setVolume(it)
            if (!music.isPlaying())
                music.chooseTrack(flags = MusicTrackChooserFlags.setPlayDefault)
        }

    fun Table.addMusicPauseSlider(settings: GameSettings, music: MusicController) {
        // map to/from 0-1-2..10-12-14..30-35-40..60-75-90-105-120
        fun posToLength(pos: Float): Float = when (pos) {
            in 0f..10f -> pos
            in 11f..20f -> pos * 2f - 10f
            in 21f..26f -> pos * 5f - 70f
            else -> pos * 15f - 330f
        }

        fun lengthToPos(length: Float): Float = floor(
            when (length) {
                in 0f..10f -> length
                in 11f..30f -> (length + 10f) / 2f
                in 31f..60f -> (length + 10f) / 5f
                else -> (length + 330f) / 15f
            }
        )

        val getTipText: (Float) -> String = {
            "%.0f".format(posToLength(it))
        }

        add("Pause between tracks".tr()).left().fillX()

        val pauseLengthSlider = UncivSlider(
            0f, 30f, 1f,
            initial = lengthToPos(music.silenceLength),
            sound = UncivSound.Silent,
            getTipText = getTipText
        ) {
            music.silenceLength = posToLength(it)
            settings.pauseBetweenTracks = music.silenceLength.toInt()
        }
        add(pauseLengthSlider).pad(5f).row()
    }

    fun Table.addMusicCurrentlyPlaying(music: MusicController) {
        val label = WrappableLabel("", width - 10f, Color(0xffd0afff.toInt()), 16)
        label.wrap = true
        add(label).padTop(20f).colspan(2).fillX().row()
        music.onChange {
            label.setText("Currently playing: [$it]".tr())
        }
        firstAscendant(Popup::class.java)?.run {
            closeListeners.add { music.onChange(null) }
        }
    }

    fun Table.addSimplePlayerControls(music: MusicController) {
        fun String.toImageButton(overColor: Color) = toImageButton(30f, 30f, Color.CLEAR, overColor)
        add(Table().apply {
            defaults().space(25f)
            add("OtherIcons/Pause".toImageButton(Color.GOLD).onClick { music.pause(0.5f) })
            add("OtherIcons/ForwardArrow".toImageButton(Color.LIME).onClick { music.resume(0.5f) })
            add("OtherIcons/Loading".toImageButton(Color.VIOLET).onClick { music.chooseTrack(flags = MusicTrackChooserFlags.none) })
        }).colspan(2).center().row()
    }

    /** Adds music volume/pause sliders, currently playing label and player controls to a [Table] */
    fun Table.addMusicControls(settings: GameSettings, music: MusicController) {
        addMusicVolumeSlider(settings, music)
        addMusicPauseSlider(settings, music)
        addMusicCurrentlyPlaying(music)
        addSimplePlayerControls(music)
    }
}
