package com.theskillapp.skillapp.shared.picker

import android.os.Bundle
import android.view.View
import android.widget.NumberPicker
import com.theskillapp.skillapp.R
import com.theskillapp.skillapp.domain.model.Goal
import com.theskillapp.skillapp.domain.model.MeasurementUnit
import com.theskillapp.skillapp.model.UiGoal
import com.theskillapp.skillapp.model.UiGoal.Type.Companion.mapToUI
import com.theskillapp.skillapp.model.UiMeasurementUnit
import com.theskillapp.skillapp.model.UiMeasurementUnit.Companion.mapToUI
import com.theskillapp.skillapp.shared.extensions.disableKeyboardInput
import com.theskillapp.skillapp.shared.extensions.setValues
import com.theskillapp.skillapp.shared.hardware.hideKeyboard

abstract class ValuePicker<T>(protected val unit: MeasurementUnit<T>) : PickerDialog(), NumberPicker.OnValueChangeListener {
    abstract val value: T
    val count get() = unit.toLong(value)

    val maxCount: Long
        get() {
            val goalType = if (firstPickerEnabled) goalType else UiGoal.Type.Daily
            return goalType?.getMaximumCount(unit) ?: 0
        }

    val maxValue: T get() = unit.toType(maxCount)

    abstract fun getPickerValuesForValue(value: T): Pair<Int, Int>

    val goalType get() = goalTypeValues[firstPicker.value]
    val goal get() = goalType?.let { goalType -> Goal(unit.toLong(value), goalType.domainCounterpart) }

    override val numberOfFirstPickerValues get() = goalTypeValues.size
    final override val numberOfSecondPickerValues get() = getPickerValuesForValue(maxValue).first + 1
    final override val numberOfThirdPickerValues get() = getPickerValuesForValue(maxValue).second + 1

    private val noValueFormatter = NumberPicker.Formatter { requireContext().getString(R.string.plan_no_time) }

    override fun formatFirstPickerValue(value: Int): String {
        return requireContext().getString(goalTypeValues[value]?.goalResId ?: R.string.goal_type_none)
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        if (firstPickerEnabled) setupFirstPicker()
    }

    private fun setupFirstPicker() {
        firstPicker.disableKeyboardInput()
        firstPicker.wrapSelectorWheel = false
        firstPicker.setOnValueChangedListener(this)
    }

    override fun onSaveInstanceState(bundle: Bundle) {
        bundle.putInt(FIRST_PICKER_VALUE, firstPicker.value)
        bundle.putLong(COUNT, count)
    }

    override fun getPickerValues(bundle: Bundle): Triple<Int, Int, Int> {
        val count = bundle.getLong(COUNT, 0)
        val pickerValues = getPickerValuesForValue(unit.toType(count))

        return Triple(
            first = bundle.getInt(FIRST_PICKER_VALUE, 0),
            second = pickerValues.first,
            third = pickerValues.second,
        )
    }

    override fun configureSecondPickerValues() {
        secondPicker.configureValues(numberOfSecondPickerValues, this::formatSecondPickerValue)
    }

    override fun configureThirdPickerValues() {
        thirdPicker.configureValues(numberOfThirdPickerValues, this::formatThirdPickerValue)
    }

    private fun NumberPicker.configureValues(numberOfValues: Int, formatter: NumberPicker.Formatter) {
        setValues(
            numberOfValues,
            if (numberOfValues > 1) formatter else noValueFormatter,
        )

        isEnabled = numberOfValues > 1
    }

    override fun onValueChange(picker: NumberPicker, oldVal: Int, newVal: Int) {
        secondPicker.clearFocus()
        thirdPicker.clearFocus()
        configureSecondPickerValues()
        configureThirdPickerValues()

        if (maxCount == 0L) {
            picker.hideKeyboard()
        }
    }

    data class Configuration(
        val value: Value,
        val isInEditMode: Boolean = false,
        val overrideThemeResId: Int = 0,
    ) {
        fun getArguments() = Bundle().apply {
            putLong(COUNT, value.count)
            putInt(FIRST_PICKER_VALUE, goalTypeValues.indexOf((value.goalType ?: Goal.Type.Daily).mapToUI()))
            putBoolean(ENABLE_FIRST_PICKER, value.mode.goalPickerEnabled)
            putInt(TITLE_RES_EXTRA, value.mode.getTitleTextResId(value.unit.mapToUI(), isInEditMode))
        }
    }

    sealed class Value {
        abstract val count: Long
        abstract val goalType: Goal.Type?
        abstract val unit: MeasurementUnit<*>
        abstract val mode: Mode

        data class RegularValue(
            override val count: Long,
            override val unit: MeasurementUnit<*>,
        ) : Value() {
            override val goalType get() = null
            override val mode get() = Mode.ValuePicker
        }

        data class GoalValue(
            val goal: Goal?,
            override val unit: MeasurementUnit<*>,
        ): Value() {
            override val count get() = goal?.count ?: 0
            override val goalType get() = goal?.type
            override val mode get() = Mode.GoalPicker
        }
    }

    sealed class Mode {
        object ValuePicker: Mode() {
            override fun getViewModeTitleTextResId(unit: UiMeasurementUnit) = unit.addRecordDialogTitleResId
            override fun getEditModeTitleTextResId(unit: UiMeasurementUnit) = unit.changeCountResId
            override val goalPickerEnabled = false
        }

        object GoalPicker: Mode() {
            override fun getViewModeTitleTextResId(unit: UiMeasurementUnit) = R.string.select_goal
            override val goalPickerEnabled = true
        }

        abstract val goalPickerEnabled: Boolean

        abstract fun getViewModeTitleTextResId(unit: UiMeasurementUnit): Int
        open fun getEditModeTitleTextResId(unit: UiMeasurementUnit) = getViewModeTitleTextResId(unit)

        fun getTitleTextResId(unit: UiMeasurementUnit, isInEditMode: Boolean): Int {
            if (isInEditMode) return getViewModeTitleTextResId(unit)
            else return getEditModeTitleTextResId(unit)
        }
    }

    companion object {
        const val COUNT = "COUNT"

        val goalTypeValues = arrayOf(null, *UiGoal.Type.values())
    }
}
