import { isObservable } from 'rxjs';
import { AbstractControl } from '@angular/forms';
import { ChangeDetectorRef, ComponentRef, TemplateRef, Type, VERSION, ɵNoopNgZone } from '@angular/core';
export function disableTreeValidityCall(form, callback) {
    const _updateTreeValidity = form._updateTreeValidity.bind(form);
    form._updateTreeValidity = () => { };
    callback();
    form._updateTreeValidity = _updateTreeValidity;
}
export function getFieldId(formId, field, index) {
    if (field.id) {
        return field.id;
    }
    let type = field.type;
    if (!type && field.template) {
        type = 'template';
    }
    if (type instanceof Type) {
        type = type.prototype.constructor.name;
    }
    return [formId, type, field.key, index].join('_');
}
export function hasKey(field) {
    return !isNil(field.key) && field.key !== '' && (!Array.isArray(field.key) || field.key.length > 0);
}
export function getKeyPath(field) {
    if (!hasKey(field)) {
        return [];
    }
    /* We store the keyPath in the field for performance reasons. This function will be called frequently. */
    if (field._keyPath?.key !== field.key) {
        let path = [];
        if (typeof field.key === 'string') {
            const key = field.key.indexOf('[') === -1 ? field.key : field.key.replace(/\[(\w+)\]/g, '.$1');
            path = key.indexOf('.') !== -1 ? key.split('.') : [key];
        }
        else if (Array.isArray(field.key)) {
            path = field.key.slice(0);
        }
        else {
            path = [`${field.key}`];
        }
        defineHiddenProp(field, '_keyPath', { key: field.key, path });
    }
    return field._keyPath.path.slice(0);
}
export const FORMLY_VALIDATORS = ['required', 'pattern', 'minLength', 'maxLength', 'min', 'max'];
export function assignFieldValue(field, value) {
    let paths = getKeyPath(field);
    if (paths.length === 0) {
        return;
    }
    let root = field;
    while (root.parent) {
        root = root.parent;
        paths = [...getKeyPath(root), ...paths];
    }
    if (value === undefined && field.resetOnHide) {
        const k = paths.pop();
        const m = paths.reduce((model, path) => model[path] || {}, root.model);
        delete m[k];
        return;
    }
    assignModelValue(root.model, paths, value);
}
export function assignModelValue(model, paths, value) {
    for (let i = 0; i < paths.length - 1; i++) {
        const path = paths[i];
        if (!model[path] || !isObject(model[path])) {
            model[path] = /^\d+$/.test(paths[i + 1]) ? [] : {};
        }
        model = model[path];
    }
    model[paths[paths.length - 1]] = clone(value);
}
export function getFieldValue(field) {
    let model = field.parent ? field.parent.model : field.model;
    for (const path of getKeyPath(field)) {
        if (!model) {
            return model;
        }
        model = model[path];
    }
    return model;
}
export function reverseDeepMerge(dest, ...args) {
    args.forEach((src) => {
        for (const srcArg in src) {
            if (isNil(dest[srcArg]) || isBlankString(dest[srcArg])) {
                dest[srcArg] = clone(src[srcArg]);
            }
            else if (objAndSameType(dest[srcArg], src[srcArg])) {
                reverseDeepMerge(dest[srcArg], src[srcArg]);
            }
        }
    });
    return dest;
}
// check a value is null or undefined
export function isNil(value) {
    return value == null;
}
export function isUndefined(value) {
    return value === undefined;
}
export function isBlankString(value) {
    return value === '';
}
export function isFunction(value) {
    return typeof value === 'function';
}
export function objAndSameType(obj1, obj2) {
    return (isObject(obj1) &&
        isObject(obj2) &&
        Object.getPrototypeOf(obj1) === Object.getPrototypeOf(obj2) &&
        !(Array.isArray(obj1) || Array.isArray(obj2)));
}
export function isObject(x) {
    return x != null && typeof x === 'object';
}
export function isPromise(obj) {
    return !!obj && typeof obj.then === 'function';
}
export function clone(value) {
    if (!isObject(value) ||
        isObservable(value) ||
        value instanceof TemplateRef ||
        /* instanceof SafeHtmlImpl */ value.changingThisBreaksApplicationSecurity ||
        ['RegExp', 'FileList', 'File', 'Blob'].indexOf(value.constructor?.name) !== -1) {
        return value;
    }
    if (value instanceof Set) {
        return new Set(value);
    }
    if (value instanceof Map) {
        return new Map(value);
    }
    if (value instanceof Uint8Array) {
        return new Uint8Array(value);
    }
    if (value instanceof Uint16Array) {
        return new Uint16Array(value);
    }
    if (value instanceof Uint32Array) {
        return new Uint32Array(value);
    }
    // https://github.com/moment/moment/blob/master/moment.js#L252
    if (value._isAMomentObject && isFunction(value.clone)) {
        return value.clone();
    }
    if (value instanceof AbstractControl) {
        return null;
    }
    if (value instanceof Date) {
        return new Date(value.getTime());
    }
    if (Array.isArray(value)) {
        return value.slice(0).map((v) => clone(v));
    }
    // best way to clone a js object maybe
    // https://stackoverflow.com/questions/41474986/how-to-clone-a-javascript-es6-class-instance
    const proto = Object.getPrototypeOf(value);
    let c = Object.create(proto);
    c = Object.setPrototypeOf(c, proto);
    // need to make a deep copy so we dont use Object.assign
    // also Object.assign wont copy property descriptor exactly
    return Object.keys(value).reduce((newVal, prop) => {
        const propDesc = Object.getOwnPropertyDescriptor(value, prop);
        if (propDesc.get) {
            Object.defineProperty(newVal, prop, propDesc);
        }
        else {
            newVal[prop] = clone(value[prop]);
        }
        return newVal;
    }, c);
}
export function defineHiddenProp(field, prop, defaultValue) {
    Object.defineProperty(field, prop, { enumerable: false, writable: true, configurable: true });
    field[prop] = defaultValue;
}
export function observeDeep(source, paths, setFn) {
    let observers = [];
    const unsubscribe = () => {
        observers.forEach((observer) => observer());
        observers = [];
    };
    const observer = observe(source, paths, ({ firstChange, currentValue }) => {
        !firstChange && setFn();
        unsubscribe();
        if (isObject(currentValue) && currentValue.constructor.name === 'Object') {
            Object.keys(currentValue).forEach((prop) => {
                observers.push(observeDeep(source, [...paths, prop], setFn));
            });
        }
    });
    return () => {
        observer.unsubscribe();
        unsubscribe();
    };
}
export function observe(o, paths, setFn) {
    if (!o._observers) {
        defineHiddenProp(o, '_observers', {});
    }
    let target = o;
    for (let i = 0; i < paths.length - 1; i++) {
        if (!target[paths[i]] || !isObject(target[paths[i]])) {
            target[paths[i]] = /^\d+$/.test(paths[i + 1]) ? [] : {};
        }
        target = target[paths[i]];
    }
    const key = paths[paths.length - 1];
    const prop = paths.join('.');
    if (!o._observers[prop]) {
        o._observers[prop] = { value: target[key], onChange: [] };
    }
    const state = o._observers[prop];
    if (target[key] !== state.value) {
        state.value = target[key];
    }
    if (setFn && state.onChange.indexOf(setFn) === -1) {
        state.onChange.push(setFn);
        setFn({ currentValue: state.value, firstChange: true });
        if (state.onChange.length >= 1 && isObject(target)) {
            const { enumerable } = Object.getOwnPropertyDescriptor(target, key) || { enumerable: true };
            Object.defineProperty(target, key, {
                enumerable,
                configurable: true,
                get: () => state.value,
                set: (currentValue) => {
                    if (currentValue !== state.value) {
                        const previousValue = state.value;
                        state.value = currentValue;
                        state.onChange.forEach((changeFn) => changeFn({ previousValue, currentValue, firstChange: false }));
                    }
                },
            });
        }
    }
    return {
        setValue(currentValue, emitEvent = true) {
            if (currentValue === state.value) {
                return;
            }
            const previousValue = state.value;
            state.value = currentValue;
            state.onChange.forEach((changeFn) => {
                if (changeFn !== setFn && emitEvent) {
                    changeFn({ previousValue, currentValue, firstChange: false });
                }
            });
        },
        unsubscribe() {
            state.onChange = state.onChange.filter((changeFn) => changeFn !== setFn);
            if (state.onChange.length === 0) {
                delete o._observers[prop];
            }
        },
    };
}
export function getField(f, key) {
    key = (Array.isArray(key) ? key.join('.') : key);
    if (!f.fieldGroup) {
        return undefined;
    }
    for (let i = 0, len = f.fieldGroup.length; i < len; i++) {
        const c = f.fieldGroup[i];
        const k = (Array.isArray(c.key) ? c.key.join('.') : c.key);
        if (k === key) {
            return c;
        }
        if (c.fieldGroup && (isNil(k) || key.indexOf(`${k}.`) === 0)) {
            const field = getField(c, isNil(k) ? key : key.slice(k.length + 1));
            if (field) {
                return field;
            }
        }
    }
    return undefined;
}
export function markFieldForCheck(field) {
    field._componentRefs?.forEach((ref) => {
        // NOTE: we cannot use ref.changeDetectorRef, see https://github.com/ngx-formly/ngx-formly/issues/2191
        if (ref instanceof ComponentRef) {
            const changeDetectorRef = ref.injector.get(ChangeDetectorRef);
            changeDetectorRef.markForCheck();
        }
        else {
            ref.markForCheck();
        }
    });
}
export function isNoopNgZone(ngZone) {
    return ngZone instanceof ɵNoopNgZone;
}
export function isHiddenField(field) {
    const isHidden = (f) => f.hide || f.expressions?.hide || f.hideExpression;
    let setDefaultValue = !field.resetOnHide || !isHidden(field);
    if (!isHidden(field) && field.resetOnHide) {
        let parent = field.parent;
        while (parent && !isHidden(parent)) {
            parent = parent.parent;
        }
        setDefaultValue = !parent || !isHidden(parent);
    }
    return !setDefaultValue;
}
export function isSignalRequired() {
    return +VERSION.major > 18 || (+VERSION.major >= 18 && +VERSION.minor >= 1);
}
//# sourceMappingURL=data:application/json;base64,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