import { Injectable, Type } from '@angular/core';
import { reverseDeepMerge } from './../utils';
import * as i0 from "@angular/core";
/**
 * Maintains list of formly config options. This can be used to register new field type.
 */
export class FormlyConfig {
    constructor() {
        this.types = {};
        this.validators = {};
        this.wrappers = {};
        this.messages = {};
        this.extras = {
            checkExpressionOn: 'modelChange',
            lazyRender: true,
            resetFieldOnHide: true,
            renderFormlyFieldElement: true,
            showError(field) {
                return (field.formControl?.invalid &&
                    (field.formControl?.touched || field.options.parentForm?.submitted || !!field.field.validation?.show));
            },
        };
        this.extensions = {};
        this.presets = {};
        this.extensionsByPriority = {};
        this.componentRefs = {};
    }
    addConfig(config) {
        if (Array.isArray(config)) {
            config.forEach((c) => this.addConfig(c));
            return;
        }
        if (config.types) {
            config.types.forEach((type) => this.setType(type));
        }
        if (config.validators) {
            config.validators.forEach((validator) => this.setValidator(validator));
        }
        if (config.wrappers) {
            config.wrappers.forEach((wrapper) => this.setWrapper(wrapper));
        }
        if (config.validationMessages) {
            config.validationMessages.forEach((validation) => this.addValidatorMessage(validation.name, validation.message));
        }
        if (config.extensions) {
            this.setSortedExtensions(config.extensions);
        }
        if (config.extras) {
            this.extras = { ...this.extras, ...config.extras };
        }
        if (config.presets) {
            this.presets = {
                ...this.presets,
                ...config.presets.reduce((acc, curr) => ({ ...acc, [curr.name]: curr.config }), {}),
            };
        }
    }
    /**
     * Allows you to specify a custom type which you can use in your field configuration.
     * You can pass an object of options, or an array of objects of options.
     */
    setType(options) {
        if (Array.isArray(options)) {
            options.forEach((option) => this.setType(option));
        }
        else {
            if (!this.types[options.name]) {
                this.types[options.name] = { name: options.name };
            }
            ['component', 'extends', 'defaultOptions', 'wrappers'].forEach((prop) => {
                if (options.hasOwnProperty(prop)) {
                    this.types[options.name][prop] = options[prop];
                }
            });
        }
    }
    getType(name, throwIfNotFound = false) {
        if (name instanceof Type) {
            return { component: name, name: name.prototype.constructor.name };
        }
        if (!this.types[name]) {
            if (throwIfNotFound) {
                throw new Error(`[Formly Error] The type "${name}" could not be found. Please make sure that is registered through the FormlyModule declaration.`);
            }
            return null;
        }
        this.mergeExtendedType(name);
        return this.types[name];
    }
    /** @ignore */
    getMergedField(field = {}) {
        const type = this.getType(field.type);
        if (!type) {
            return;
        }
        if (type.defaultOptions) {
            reverseDeepMerge(field, type.defaultOptions);
        }
        const extendDefaults = type.extends && this.getType(type.extends).defaultOptions;
        if (extendDefaults) {
            reverseDeepMerge(field, extendDefaults);
        }
        if (field?.optionsTypes) {
            field.optionsTypes.forEach((option) => {
                const defaultOptions = this.getType(option).defaultOptions;
                if (defaultOptions) {
                    reverseDeepMerge(field, defaultOptions);
                }
            });
        }
        const componentRef = this.resolveFieldTypeRef(field);
        if (componentRef?.instance?.defaultOptions) {
            reverseDeepMerge(field, componentRef.instance.defaultOptions);
        }
        if (!field.wrappers && type.wrappers) {
            field.wrappers = [...type.wrappers];
        }
    }
    /** @ignore @internal */
    resolveFieldTypeRef(field = {}) {
        const type = this.getType(field.type);
        if (!type) {
            return null;
        }
        if (!type.component) {
            return null;
        }
        if (!this.componentRefs[type.name]) {
            const { _viewContainerRef, _injector } = field.options;
            if (!_viewContainerRef || !_injector) {
                return null;
            }
            const componentRef = _viewContainerRef.createComponent(type.component, { injector: _injector });
            this.componentRefs[type.name] = componentRef;
            try {
                componentRef.destroy();
            }
            catch (e) {
                console.error(`An error occurred while destroying the Formly component type "${field.type}"`, e);
            }
        }
        return this.componentRefs[type.name];
    }
    /** @ignore @internal */
    clearRefs() {
        this.componentRefs = {};
    }
    setWrapper(options) {
        this.wrappers[options.name] = options;
        if (options.types) {
            options.types.forEach((type) => {
                this.setTypeWrapper(type, options.name);
            });
        }
    }
    getWrapper(name) {
        if (name instanceof Type) {
            return { component: name, name: name.prototype.constructor.name };
        }
        if (!this.wrappers[name]) {
            throw new Error(`[Formly Error] The wrapper "${name}" could not be found. Please make sure that is registered through the FormlyModule declaration.`);
        }
        return this.wrappers[name];
    }
    /** @ignore */
    setTypeWrapper(type, name) {
        if (!this.types[type]) {
            this.types[type] = {};
        }
        if (!this.types[type].wrappers) {
            this.types[type].wrappers = [];
        }
        if (this.types[type].wrappers.indexOf(name) === -1) {
            this.types[type].wrappers.push(name);
        }
    }
    setValidator(options) {
        this.validators[options.name] = options;
    }
    getValidator(name) {
        if (!this.validators[name]) {
            throw new Error(`[Formly Error] The validator "${name}" could not be found. Please make sure that is registered through the FormlyModule declaration.`);
        }
        return this.validators[name];
    }
    addValidatorMessage(name, message) {
        this.messages[name] = message;
        if (typeof ngDevMode === 'undefined' || ngDevMode) {
            const deprecated = { minlength: 'minLength', maxlength: 'maxLength' };
            if (deprecated[name]) {
                console.warn(`Formly deprecation: passing validation messages key '${name}' is deprecated since v6.0, use '${deprecated[name]}' instead.`);
                this.messages[deprecated[name]] = message;
            }
        }
    }
    getValidatorMessage(name) {
        return this.messages[name];
    }
    setSortedExtensions(extensionOptions) {
        // insert new extensions, grouped by priority
        extensionOptions.forEach((extensionOption) => {
            const priority = extensionOption.priority ?? 1;
            this.extensionsByPriority[priority] = {
                ...this.extensionsByPriority[priority],
                [extensionOption.name]: extensionOption.extension,
            };
        });
        // flatten extensions object with sorted keys
        this.extensions = Object.keys(this.extensionsByPriority)
            .map(Number)
            .sort((a, b) => a - b)
            .reduce((acc, prio) => ({
            ...acc,
            ...this.extensionsByPriority[prio],
        }), {});
    }
    mergeExtendedType(name) {
        if (!this.types[name].extends) {
            return;
        }
        const extendedType = this.getType(this.types[name].extends);
        if (!this.types[name].component) {
            this.types[name].component = extendedType.component;
        }
        if (!this.types[name].wrappers) {
            this.types[name].wrappers = extendedType.wrappers;
        }
    }
    static { this.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "18.2.13", ngImport: i0, type: FormlyConfig, deps: [], target: i0.ɵɵFactoryTarget.Injectable }); }
    static { this.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "18.2.13", ngImport: i0, type: FormlyConfig, providedIn: 'root' }); }
}
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "18.2.13", ngImport: i0, type: FormlyConfig, decorators: [{
            type: Injectable,
            args: [{ providedIn: 'root' }]
        }] });
//# sourceMappingURL=data:application/json;base64,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