import { Validators } from '@angular/forms';
import { FORMLY_VALIDATORS, defineHiddenProp, isPromise, observe, clone, isObject, hasKey } from '../../utils';
import { updateValidity } from '../field-form/utils';
import { isObservable } from 'rxjs';
import { map } from 'rxjs/operators';
export class FieldValidationExtension {
    constructor(config) {
        this.config = config;
    }
    onPopulate(field) {
        this.initFieldValidation(field, 'validators');
        this.initFieldValidation(field, 'asyncValidators');
    }
    initFieldValidation(field, type) {
        const validators = [];
        if (type === 'validators' && !(field.hasOwnProperty('fieldGroup') && !hasKey(field))) {
            validators.push(this.getPredefinedFieldValidation(field));
        }
        if (field[type]) {
            for (const validatorName of Object.keys(field[type])) {
                validatorName === 'validation'
                    ? validators.push(...field[type].validation.map((v) => this.wrapNgValidatorFn(field, v)))
                    : validators.push(this.wrapNgValidatorFn(field, field[type][validatorName], validatorName));
            }
        }
        defineHiddenProp(field, '_' + type, validators);
    }
    getPredefinedFieldValidation(field) {
        let VALIDATORS = [];
        FORMLY_VALIDATORS.forEach((opt) => observe(field, ['props', opt], ({ currentValue, firstChange }) => {
            VALIDATORS = VALIDATORS.filter((o) => o !== opt);
            if (opt === 'required' && currentValue != null && typeof currentValue !== 'boolean') {
                console.warn(`Formly: Invalid prop 'required' of type '${typeof currentValue}', expected 'boolean' (Field:${field.key}).`);
            }
            if (currentValue != null && currentValue !== false) {
                VALIDATORS.push(opt);
            }
            if (!firstChange && field.formControl) {
                updateValidity(field.formControl);
            }
        }));
        return (control) => {
            if (VALIDATORS.length === 0) {
                return null;
            }
            return Validators.compose(VALIDATORS.map((opt) => () => {
                const value = field.props[opt];
                switch (opt) {
                    case 'required':
                        return Validators.required(control);
                    case 'pattern':
                        return Validators.pattern(value)(control);
                    case 'minLength':
                        const minLengthResult = Validators.minLength(value)(control);
                        const minLengthKey = this.config.getValidatorMessage('minlength') || field.validation?.messages?.minlength
                            ? 'minlength'
                            : 'minLength';
                        return minLengthResult ? { [minLengthKey]: minLengthResult.minlength } : null;
                    case 'maxLength':
                        const maxLengthResult = Validators.maxLength(value)(control);
                        const maxLengthKey = this.config.getValidatorMessage('maxlength') || field.validation?.messages?.maxlength
                            ? 'maxlength'
                            : 'maxLength';
                        return maxLengthResult ? { [maxLengthKey]: maxLengthResult.maxlength } : null;
                    case 'min':
                        return Validators.min(value)(control);
                    case 'max':
                        return Validators.max(value)(control);
                    default:
                        return null;
                }
            }))(control);
        };
    }
    wrapNgValidatorFn(field, validator, validatorName) {
        let validatorOption;
        if (typeof validator === 'string') {
            validatorOption = clone(this.config.getValidator(validator));
        }
        if (typeof validator === 'object' && validator.name) {
            validatorOption = clone(this.config.getValidator(validator.name));
            if (validator.options) {
                validatorOption.options = validator.options;
            }
        }
        if (typeof validator === 'object' && validator.expression) {
            const { expression, ...options } = validator;
            validatorOption = {
                name: validatorName,
                validation: expression,
                options: Object.keys(options).length > 0 ? options : null,
            };
        }
        if (typeof validator === 'function') {
            validatorOption = {
                name: validatorName,
                validation: validator,
            };
        }
        return (control) => {
            const errors = validatorOption.validation(control, field, validatorOption.options);
            if (isPromise(errors)) {
                return errors.then((v) => this.handleResult(field, validatorName ? !!v : v, validatorOption));
            }
            if (isObservable(errors)) {
                return errors.pipe(map((v) => this.handleResult(field, validatorName ? !!v : v, validatorOption)));
            }
            return this.handleResult(field, validatorName ? !!errors : errors, validatorOption);
        };
    }
    handleResult(field, errors, { name, options }) {
        if (typeof errors === 'boolean') {
            errors = errors ? null : { [name]: options ? options : true };
        }
        const ctrl = field.formControl;
        ctrl?._childrenErrors?.[name]?.();
        if (isObject(errors)) {
            Object.keys(errors).forEach((name) => {
                const errorPath = errors[name].errorPath ? errors[name].errorPath : options?.errorPath;
                const childCtrl = errorPath ? field.formControl.get(errorPath) : null;
                if (childCtrl) {
                    const { errorPath: _errorPath, ...opts } = errors[name];
                    childCtrl.setErrors({ ...(childCtrl.errors || {}), [name]: opts });
                    !ctrl._childrenErrors && defineHiddenProp(ctrl, '_childrenErrors', {});
                    ctrl._childrenErrors[name] = () => {
                        const { [name]: _toDelete, ...childErrors } = childCtrl.errors || {};
                        childCtrl.setErrors(Object.keys(childErrors).length === 0 ? null : childErrors);
                    };
                }
            });
        }
        return errors;
    }
}
//# sourceMappingURL=data:application/json;base64,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