import { UntypedFormArray, UntypedFormGroup, FormControl } from '@angular/forms';
import { getKeyPath, getFieldValue, isNil, defineHiddenProp, observe, hasKey } from '../../utils';
export function unregisterControl(field, emitEvent = false) {
    const control = field.formControl;
    const fieldIndex = control._fields ? control._fields.indexOf(field) : -1;
    if (fieldIndex !== -1) {
        control._fields.splice(fieldIndex, 1);
    }
    const form = control.parent;
    if (!form) {
        return;
    }
    const opts = { emitEvent };
    if (form instanceof UntypedFormArray) {
        const key = form.controls.findIndex((c) => c === control);
        if (key !== -1) {
            form.removeAt(key, opts);
        }
    }
    else if (form instanceof UntypedFormGroup) {
        const paths = getKeyPath(field);
        const key = paths[paths.length - 1];
        if (form.get([key]) === control) {
            form.removeControl(key, opts);
        }
    }
    control.setParent(null);
}
export function findControl(field) {
    if (field.formControl) {
        return field.formControl;
    }
    if (field.shareFormControl === false) {
        return null;
    }
    return field.form?.get(getKeyPath(field));
}
export function registerControl(field, control, emitEvent = false) {
    control = control || field.formControl;
    if (!control._fields) {
        defineHiddenProp(control, '_fields', []);
    }
    if (control._fields.indexOf(field) === -1) {
        control._fields.push(field);
    }
    if (!field.formControl && control) {
        defineHiddenProp(field, 'formControl', control);
        control.setValidators(null);
        control.setAsyncValidators(null);
        field.props.disabled = !!field.props.disabled;
        const disabledObserver = observe(field, ['props', 'disabled'], ({ firstChange, currentValue }) => {
            if (!firstChange) {
                currentValue ? field.formControl.disable() : field.formControl.enable();
            }
        });
        if (control instanceof FormControl) {
            control.registerOnDisabledChange(disabledObserver.setValue);
        }
    }
    if (!field.form || !hasKey(field)) {
        return;
    }
    let form = field.form;
    const paths = getKeyPath(field);
    const value = getFieldValue(field);
    if (!(isNil(control.value) && isNil(value)) && control.value !== value && control instanceof FormControl) {
        control.patchValue(value);
    }
    for (let i = 0; i < paths.length - 1; i++) {
        const path = paths[i];
        if (!form.get([path])) {
            form.setControl(path, new UntypedFormGroup({}), { emitEvent });
        }
        form = form.get([path]);
    }
    const key = paths[paths.length - 1];
    if (!field._hide && form.get([key]) !== control) {
        form.setControl(key, control, { emitEvent });
    }
}
export function updateValidity(c, onlySelf = false) {
    const status = c.status;
    const value = c.value;
    c.updateValueAndValidity({ emitEvent: false, onlySelf });
    if (status !== c.status) {
        c.statusChanges.emit(c.status);
    }
    if (value !== c.value) {
        c.valueChanges.emit(c.value);
    }
}
export function clearControl(form) {
    delete form?._fields;
    form.setValidators(null);
    form.setAsyncValidators(null);
    if (form instanceof UntypedFormGroup || form instanceof UntypedFormArray) {
        Object.values(form.controls).forEach((c) => clearControl(c));
    }
}
//# sourceMappingURL=data:application/json;base64,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