import { UntypedFormGroup, UntypedFormControl, Validators, FormControl, } from '@angular/forms';
import { getFieldValue, defineHiddenProp, hasKey, getKeyPath } from '../../utils';
import { registerControl, findControl, updateValidity } from './utils';
import { of } from 'rxjs';
export class FieldFormExtension {
    prePopulate(field) {
        if (!this.root) {
            this.root = field;
        }
        if (field.parent) {
            Object.defineProperty(field, 'form', {
                get: () => field.parent.formControl,
                configurable: true,
            });
        }
    }
    onPopulate(field) {
        if (field.hasOwnProperty('fieldGroup') && !hasKey(field)) {
            defineHiddenProp(field, 'formControl', field.form);
        }
        else {
            this.addFormControl(field);
        }
    }
    postPopulate(field) {
        if (this.root !== field) {
            return;
        }
        this.root = null;
        const markForCheck = this.setValidators(field);
        if (markForCheck && field.parent) {
            let parent = field.parent;
            while (parent) {
                if (hasKey(parent) || !parent.parent) {
                    updateValidity(parent.formControl, true);
                }
                parent = parent.parent;
            }
        }
    }
    addFormControl(field) {
        let control = findControl(field);
        if (field.fieldArray) {
            return;
        }
        if (!control) {
            const controlOptions = { updateOn: field.modelOptions.updateOn };
            if (field.fieldGroup) {
                control = new UntypedFormGroup({}, controlOptions);
            }
            else {
                const value = hasKey(field) ? getFieldValue(field) : field.defaultValue;
                control = new UntypedFormControl({ value, disabled: !!field.props.disabled }, { ...controlOptions, initialValueIsDefault: true });
            }
        }
        else {
            if (control instanceof FormControl) {
                const value = hasKey(field) ? getFieldValue(field) : field.defaultValue;
                control.defaultValue = value;
            }
        }
        registerControl(field, control);
    }
    setValidators(field, disabled = false) {
        if (disabled === false && hasKey(field) && field.props?.disabled) {
            disabled = true;
        }
        let markForCheck = false;
        field.fieldGroup?.forEach((f) => f && this.setValidators(f, disabled) && (markForCheck = true));
        if (hasKey(field) || !field.parent || (!hasKey(field) && !field.fieldGroup)) {
            const { formControl: c } = field;
            if (c) {
                if (hasKey(field) && c instanceof FormControl) {
                    if (disabled && c.enabled) {
                        c.disable({ emitEvent: false, onlySelf: true });
                        markForCheck = true;
                    }
                    if (!disabled && c.disabled) {
                        c.enable({ emitEvent: false, onlySelf: true });
                        markForCheck = true;
                    }
                }
                if (null === c.validator && this.hasValidators(field, '_validators')) {
                    c.setValidators(() => {
                        const v = Validators.compose(this.mergeValidators(field, '_validators'));
                        return v ? v(c) : null;
                    });
                    markForCheck = true;
                }
                if (null === c.asyncValidator && this.hasValidators(field, '_asyncValidators')) {
                    c.setAsyncValidators(() => {
                        const v = Validators.composeAsync(this.mergeValidators(field, '_asyncValidators'));
                        return v ? v(c) : of(null);
                    });
                    markForCheck = true;
                }
                if (markForCheck) {
                    updateValidity(c, true);
                    // update validity of `FormGroup` instance created by field with nested key.
                    let parent = c.parent;
                    for (let i = 1; i < getKeyPath(field).length; i++) {
                        if (parent) {
                            updateValidity(parent, true);
                            parent = parent.parent;
                        }
                    }
                }
            }
        }
        return markForCheck;
    }
    hasValidators(field, type) {
        const c = field.formControl;
        if (c?._fields?.length > 1 && c._fields.some((f) => f[type].length > 0)) {
            return true;
        }
        else if (field[type].length > 0) {
            return true;
        }
        return field.fieldGroup?.some((f) => f?.fieldGroup && !hasKey(f) && this.hasValidators(f, type));
    }
    mergeValidators(field, type) {
        const validators = [];
        const c = field.formControl;
        if (c?._fields?.length > 1) {
            c._fields
                .filter((f) => !f._hide)
                .forEach((f) => validators.push(...f[type]));
        }
        else if (field[type]) {
            validators.push(...field[type]);
        }
        if (field.fieldGroup) {
            field.fieldGroup
                .filter((f) => f?.fieldGroup && !hasKey(f))
                .forEach((f) => validators.push(...this.mergeValidators(f, type)));
        }
        return validators;
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZmllbGQtZm9ybS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uLy4uLy4uLy4uLy4uL3NyYy9jb3JlL3NyYy9saWIvZXh0ZW5zaW9ucy9maWVsZC1mb3JtL2ZpZWxkLWZvcm0udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQ0EsT0FBTyxFQUNMLGdCQUFnQixFQUNoQixrQkFBa0IsRUFFbEIsVUFBVSxFQUdWLFdBQVcsR0FDWixNQUFNLGdCQUFnQixDQUFDO0FBQ3hCLE9BQU8sRUFBRSxhQUFhLEVBQUUsZ0JBQWdCLEVBQUUsTUFBTSxFQUFFLFVBQVUsRUFBRSxNQUFNLGFBQWEsQ0FBQztBQUNsRixPQUFPLEVBQUUsZUFBZSxFQUFFLFdBQVcsRUFBRSxjQUFjLEVBQUUsTUFBTSxTQUFTLENBQUM7QUFDdkUsT0FBTyxFQUFFLEVBQUUsRUFBRSxNQUFNLE1BQU0sQ0FBQztBQUUxQixNQUFNLE9BQU8sa0JBQWtCO0lBRTdCLFdBQVcsQ0FBQyxLQUE2QjtRQUN2QyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1lBQ2YsSUFBSSxDQUFDLElBQUksR0FBRyxLQUFLLENBQUM7UUFDcEIsQ0FBQztRQUVELElBQUksS0FBSyxDQUFDLE1BQU0sRUFBRSxDQUFDO1lBQ2pCLE1BQU0sQ0FBQyxjQUFjLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDbkMsR0FBRyxFQUFFLEdBQUcsRUFBRSxDQUFDLEtBQUssQ0FBQyxNQUFPLENBQUMsV0FBVztnQkFDcEMsWUFBWSxFQUFFLElBQUk7YUFDbkIsQ0FBQyxDQUFDO1FBQ0wsQ0FBQztJQUNILENBQUM7SUFFRCxVQUFVLENBQUMsS0FBNkI7UUFDdEMsSUFBSSxLQUFLLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUM7WUFDekQsZ0JBQWdCLENBQUMsS0FBSyxFQUFFLGFBQWEsRUFBRSxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDckQsQ0FBQzthQUFNLENBQUM7WUFDTixJQUFJLENBQUMsY0FBYyxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQzdCLENBQUM7SUFDSCxDQUFDO0lBRUQsWUFBWSxDQUFDLEtBQTZCO1FBQ3hDLElBQUksSUFBSSxDQUFDLElBQUksS0FBSyxLQUFLLEVBQUUsQ0FBQztZQUN4QixPQUFPO1FBQ1QsQ0FBQztRQUVELElBQUksQ0FBQyxJQUFJLEdBQUcsSUFBSSxDQUFDO1FBQ2pCLE1BQU0sWUFBWSxHQUFHLElBQUksQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDL0MsSUFBSSxZQUFZLElBQUksS0FBSyxDQUFDLE1BQU0sRUFBRSxDQUFDO1lBQ2pDLElBQUksTUFBTSxHQUFHLEtBQUssQ0FBQyxNQUFNLENBQUM7WUFDMUIsT0FBTyxNQUFNLEVBQUUsQ0FBQztnQkFDZCxJQUFJLE1BQU0sQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsQ0FBQztvQkFDckMsY0FBYyxDQUFDLE1BQU0sQ0FBQyxXQUFXLEVBQUUsSUFBSSxDQUFDLENBQUM7Z0JBQzNDLENBQUM7Z0JBQ0QsTUFBTSxHQUFHLE1BQU0sQ0FBQyxNQUFNLENBQUM7WUFDekIsQ0FBQztRQUNILENBQUM7SUFDSCxDQUFDO0lBRU8sY0FBYyxDQUFDLEtBQTZCO1FBQ2xELElBQUksT0FBTyxHQUFHLFdBQVcsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUNqQyxJQUFJLEtBQUssQ0FBQyxVQUFVLEVBQUUsQ0FBQztZQUNyQixPQUFPO1FBQ1QsQ0FBQztRQUVELElBQUksQ0FBQyxPQUFPLEVBQUUsQ0FBQztZQUNiLE1BQU0sY0FBYyxHQUEyQixFQUFFLFFBQVEsRUFBRSxLQUFLLENBQUMsWUFBWSxDQUFDLFFBQVEsRUFBRSxDQUFDO1lBRXpGLElBQUksS0FBSyxDQUFDLFVBQVUsRUFBRSxDQUFDO2dCQUNyQixPQUFPLEdBQUcsSUFBSSxnQkFBZ0IsQ0FBQyxFQUFFLEVBQUUsY0FBYyxDQUFDLENBQUM7WUFDckQsQ0FBQztpQkFBTSxDQUFDO2dCQUNOLE1BQU0sS0FBSyxHQUFHLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDO2dCQUN4RSxPQUFPLEdBQUcsSUFBSSxrQkFBa0IsQ0FDOUIsRUFBRSxLQUFLLEVBQUUsUUFBUSxFQUFFLENBQUMsQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLFFBQVEsRUFBRSxFQUMzQyxFQUFFLEdBQUcsY0FBYyxFQUFFLHFCQUFxQixFQUFFLElBQUksRUFBRSxDQUNuRCxDQUFDO1lBQ0osQ0FBQztRQUNILENBQUM7YUFBTSxDQUFDO1lBQ04sSUFBSSxPQUFPLFlBQVksV0FBVyxFQUFFLENBQUM7Z0JBQ25DLE1BQU0sS0FBSyxHQUFHLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDO2dCQUN2RSxPQUFlLENBQUMsWUFBWSxHQUFHLEtBQUssQ0FBQztZQUN4QyxDQUFDO1FBQ0gsQ0FBQztRQUVELGVBQWUsQ0FBQyxLQUFLLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVPLGFBQWEsQ0FBQyxLQUE2QixFQUFFLFFBQVEsR0FBRyxLQUFLO1FBQ25FLElBQUksUUFBUSxLQUFLLEtBQUssSUFBSSxNQUFNLENBQUMsS0FBSyxDQUFDLElBQUksS0FBSyxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUUsQ0FBQztZQUNqRSxRQUFRLEdBQUcsSUFBSSxDQUFDO1FBQ2xCLENBQUM7UUFFRCxJQUFJLFlBQVksR0FBRyxLQUFLLENBQUM7UUFDekIsS0FBSyxDQUFDLFVBQVUsRUFBRSxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsSUFBSSxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsRUFBRSxRQUFRLENBQUMsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLENBQUMsQ0FBQyxDQUFDO1FBQ2hHLElBQUksTUFBTSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sSUFBSSxDQUFDLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxFQUFFLENBQUM7WUFDNUUsTUFBTSxFQUFFLFdBQVcsRUFBRSxDQUFDLEVBQUUsR0FBRyxLQUFLLENBQUM7WUFDakMsSUFBSSxDQUFDLEVBQUUsQ0FBQztnQkFDTixJQUFJLE1BQU0sQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFlBQVksV0FBVyxFQUFFLENBQUM7b0JBQzlDLElBQUksUUFBUSxJQUFJLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQzt3QkFDMUIsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7d0JBQ2hELFlBQVksR0FBRyxJQUFJLENBQUM7b0JBQ3RCLENBQUM7b0JBRUQsSUFBSSxDQUFDLFFBQVEsSUFBSSxDQUFDLENBQUMsUUFBUSxFQUFFLENBQUM7d0JBQzVCLENBQUMsQ0FBQyxNQUFNLENBQUMsRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLFFBQVEsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO3dCQUMvQyxZQUFZLEdBQUcsSUFBSSxDQUFDO29CQUN0QixDQUFDO2dCQUNILENBQUM7Z0JBRUQsSUFBSSxJQUFJLEtBQUssQ0FBQyxDQUFDLFNBQVMsSUFBSSxJQUFJLENBQUMsYUFBYSxDQUFDLEtBQUssRUFBRSxhQUFhLENBQUMsRUFBRSxDQUFDO29CQUNyRSxDQUFDLENBQUMsYUFBYSxDQUFDLEdBQUcsRUFBRTt3QkFDbkIsTUFBTSxDQUFDLEdBQUcsVUFBVSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFjLEtBQUssRUFBRSxhQUFhLENBQUMsQ0FBQyxDQUFDO3dCQUV0RixPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUM7b0JBQ3pCLENBQUMsQ0FBQyxDQUFDO29CQUNILFlBQVksR0FBRyxJQUFJLENBQUM7Z0JBQ3RCLENBQUM7Z0JBRUQsSUFBSSxJQUFJLEtBQUssQ0FBQyxDQUFDLGNBQWMsSUFBSSxJQUFJLENBQUMsYUFBYSxDQUFDLEtBQUssRUFBRSxrQkFBa0IsQ0FBQyxFQUFFLENBQUM7b0JBQy9FLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyxHQUFHLEVBQUU7d0JBQ3hCLE1BQU0sQ0FBQyxHQUFHLFVBQVUsQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBbUIsS0FBSyxFQUFFLGtCQUFrQixDQUFDLENBQUMsQ0FBQzt3QkFDckcsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDO29CQUM3QixDQUFDLENBQUMsQ0FBQztvQkFDSCxZQUFZLEdBQUcsSUFBSSxDQUFDO2dCQUN0QixDQUFDO2dCQUVELElBQUksWUFBWSxFQUFFLENBQUM7b0JBQ2pCLGNBQWMsQ0FBQyxDQUFDLEVBQUUsSUFBSSxDQUFDLENBQUM7b0JBRXhCLDRFQUE0RTtvQkFDNUUsSUFBSSxNQUFNLEdBQUcsQ0FBQyxDQUFDLE1BQU0sQ0FBQztvQkFDdEIsS0FBSyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLFVBQVUsQ0FBQyxLQUFLLENBQUMsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQzt3QkFDbEQsSUFBSSxNQUFNLEVBQUUsQ0FBQzs0QkFDWCxjQUFjLENBQUMsTUFBTSxFQUFFLElBQUksQ0FBQyxDQUFDOzRCQUM3QixNQUFNLEdBQUcsTUFBTSxDQUFDLE1BQU0sQ0FBQzt3QkFDekIsQ0FBQztvQkFDSCxDQUFDO2dCQUNILENBQUM7WUFDSCxDQUFDO1FBQ0gsQ0FBQztRQUVELE9BQU8sWUFBWSxDQUFDO0lBQ3RCLENBQUM7SUFFTyxhQUFhLENBQUMsS0FBNkIsRUFBRSxJQUF3QztRQUMzRixNQUFNLENBQUMsR0FBRyxLQUFLLENBQUMsV0FBVyxDQUFDO1FBQzVCLElBQUksQ0FBQyxFQUFFLE9BQU8sRUFBRSxNQUFNLEdBQUcsQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxFQUFFLENBQUM7WUFDeEUsT0FBTyxJQUFJLENBQUM7UUFDZCxDQUFDO2FBQU0sSUFBSSxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRSxDQUFDO1lBQ2xDLE9BQU8sSUFBSSxDQUFDO1FBQ2QsQ0FBQztRQUVELE9BQU8sS0FBSyxDQUFDLFVBQVUsRUFBRSxJQUFJLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsRUFBRSxVQUFVLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLElBQUksSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FBQztJQUNuRyxDQUFDO0lBRU8sZUFBZSxDQUFJLEtBQTZCLEVBQUUsSUFBd0M7UUFDaEcsTUFBTSxVQUFVLEdBQVEsRUFBRSxDQUFDO1FBQzNCLE1BQU0sQ0FBQyxHQUFHLEtBQUssQ0FBQyxXQUFXLENBQUM7UUFDNUIsSUFBSSxDQUFDLEVBQUUsT0FBTyxFQUFFLE1BQU0sR0FBRyxDQUFDLEVBQUUsQ0FBQztZQUMzQixDQUFDLENBQUMsT0FBTztpQkFDTixNQUFNLENBQUMsQ0FBQyxDQUF5QixFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUM7aUJBQy9DLE9BQU8sQ0FBQyxDQUFDLENBQXlCLEVBQUUsRUFBRSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3pFLENBQUM7YUFBTSxJQUFJLEtBQUssQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDO1lBQ3ZCLFVBQVUsQ0FBQyxJQUFJLENBQUMsR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQztRQUNsQyxDQUFDO1FBRUQsSUFBSSxLQUFLLENBQUMsVUFBVSxFQUFFLENBQUM7WUFDckIsS0FBSyxDQUFDLFVBQVU7aUJBQ2IsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLEVBQUUsVUFBVSxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDO2lCQUMxQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsR0FBRyxJQUFJLENBQUMsZUFBZSxDQUFDLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDdkUsQ0FBQztRQUVELE9BQU8sVUFBVSxDQUFDO0lBQ3BCLENBQUM7Q0FDRiIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEZvcm1seUV4dGVuc2lvbiwgRm9ybWx5RmllbGRDb25maWdDYWNoZSB9IGZyb20gJy4uLy4uL21vZGVscyc7XG5pbXBvcnQge1xuICBVbnR5cGVkRm9ybUdyb3VwLFxuICBVbnR5cGVkRm9ybUNvbnRyb2wsXG4gIEFic3RyYWN0Q29udHJvbE9wdGlvbnMsXG4gIFZhbGlkYXRvcnMsXG4gIFZhbGlkYXRvckZuLFxuICBBc3luY1ZhbGlkYXRvckZuLFxuICBGb3JtQ29udHJvbCxcbn0gZnJvbSAnQGFuZ3VsYXIvZm9ybXMnO1xuaW1wb3J0IHsgZ2V0RmllbGRWYWx1ZSwgZGVmaW5lSGlkZGVuUHJvcCwgaGFzS2V5LCBnZXRLZXlQYXRoIH0gZnJvbSAnLi4vLi4vdXRpbHMnO1xuaW1wb3J0IHsgcmVnaXN0ZXJDb250cm9sLCBmaW5kQ29udHJvbCwgdXBkYXRlVmFsaWRpdHkgfSBmcm9tICcuL3V0aWxzJztcbmltcG9ydCB7IG9mIH0gZnJvbSAncnhqcyc7XG5cbmV4cG9ydCBjbGFzcyBGaWVsZEZvcm1FeHRlbnNpb24gaW1wbGVtZW50cyBGb3JtbHlFeHRlbnNpb24ge1xuICBwcml2YXRlIHJvb3Q6IEZvcm1seUZpZWxkQ29uZmlnQ2FjaGU7XG4gIHByZVBvcHVsYXRlKGZpZWxkOiBGb3JtbHlGaWVsZENvbmZpZ0NhY2hlKSB7XG4gICAgaWYgKCF0aGlzLnJvb3QpIHtcbiAgICAgIHRoaXMucm9vdCA9IGZpZWxkO1xuICAgIH1cblxuICAgIGlmIChmaWVsZC5wYXJlbnQpIHtcbiAgICAgIE9iamVjdC5kZWZpbmVQcm9wZXJ0eShmaWVsZCwgJ2Zvcm0nLCB7XG4gICAgICAgIGdldDogKCkgPT4gZmllbGQucGFyZW50IS5mb3JtQ29udHJvbCxcbiAgICAgICAgY29uZmlndXJhYmxlOiB0cnVlLFxuICAgICAgfSk7XG4gICAgfVxuICB9XG5cbiAgb25Qb3B1bGF0ZShmaWVsZDogRm9ybWx5RmllbGRDb25maWdDYWNoZSkge1xuICAgIGlmIChmaWVsZC5oYXNPd25Qcm9wZXJ0eSgnZmllbGRHcm91cCcpICYmICFoYXNLZXkoZmllbGQpKSB7XG4gICAgICBkZWZpbmVIaWRkZW5Qcm9wKGZpZWxkLCAnZm9ybUNvbnRyb2wnLCBmaWVsZC5mb3JtKTtcbiAgICB9IGVsc2Uge1xuICAgICAgdGhpcy5hZGRGb3JtQ29udHJvbChmaWVsZCk7XG4gICAgfVxuICB9XG5cbiAgcG9zdFBvcHVsYXRlKGZpZWxkOiBGb3JtbHlGaWVsZENvbmZpZ0NhY2hlKSB7XG4gICAgaWYgKHRoaXMucm9vdCAhPT0gZmllbGQpIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICB0aGlzLnJvb3QgPSBudWxsO1xuICAgIGNvbnN0IG1hcmtGb3JDaGVjayA9IHRoaXMuc2V0VmFsaWRhdG9ycyhmaWVsZCk7XG4gICAgaWYgKG1hcmtGb3JDaGVjayAmJiBmaWVsZC5wYXJlbnQpIHtcbiAgICAgIGxldCBwYXJlbnQgPSBmaWVsZC5wYXJlbnQ7XG4gICAgICB3aGlsZSAocGFyZW50KSB7XG4gICAgICAgIGlmIChoYXNLZXkocGFyZW50KSB8fCAhcGFyZW50LnBhcmVudCkge1xuICAgICAgICAgIHVwZGF0ZVZhbGlkaXR5KHBhcmVudC5mb3JtQ29udHJvbCwgdHJ1ZSk7XG4gICAgICAgIH1cbiAgICAgICAgcGFyZW50ID0gcGFyZW50LnBhcmVudDtcbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICBwcml2YXRlIGFkZEZvcm1Db250cm9sKGZpZWxkOiBGb3JtbHlGaWVsZENvbmZpZ0NhY2hlKSB7XG4gICAgbGV0IGNvbnRyb2wgPSBmaW5kQ29udHJvbChmaWVsZCk7XG4gICAgaWYgKGZpZWxkLmZpZWxkQXJyYXkpIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICBpZiAoIWNvbnRyb2wpIHtcbiAgICAgIGNvbnN0IGNvbnRyb2xPcHRpb25zOiBBYnN0cmFjdENvbnRyb2xPcHRpb25zID0geyB1cGRhdGVPbjogZmllbGQubW9kZWxPcHRpb25zLnVwZGF0ZU9uIH07XG5cbiAgICAgIGlmIChmaWVsZC5maWVsZEdyb3VwKSB7XG4gICAgICAgIGNvbnRyb2wgPSBuZXcgVW50eXBlZEZvcm1Hcm91cCh7fSwgY29udHJvbE9wdGlvbnMpO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgY29uc3QgdmFsdWUgPSBoYXNLZXkoZmllbGQpID8gZ2V0RmllbGRWYWx1ZShmaWVsZCkgOiBmaWVsZC5kZWZhdWx0VmFsdWU7XG4gICAgICAgIGNvbnRyb2wgPSBuZXcgVW50eXBlZEZvcm1Db250cm9sKFxuICAgICAgICAgIHsgdmFsdWUsIGRpc2FibGVkOiAhIWZpZWxkLnByb3BzLmRpc2FibGVkIH0sXG4gICAgICAgICAgeyAuLi5jb250cm9sT3B0aW9ucywgaW5pdGlhbFZhbHVlSXNEZWZhdWx0OiB0cnVlIH0sXG4gICAgICAgICk7XG4gICAgICB9XG4gICAgfSBlbHNlIHtcbiAgICAgIGlmIChjb250cm9sIGluc3RhbmNlb2YgRm9ybUNvbnRyb2wpIHtcbiAgICAgICAgY29uc3QgdmFsdWUgPSBoYXNLZXkoZmllbGQpID8gZ2V0RmllbGRWYWx1ZShmaWVsZCkgOiBmaWVsZC5kZWZhdWx0VmFsdWU7XG4gICAgICAgIChjb250cm9sIGFzIGFueSkuZGVmYXVsdFZhbHVlID0gdmFsdWU7XG4gICAgICB9XG4gICAgfVxuXG4gICAgcmVnaXN0ZXJDb250cm9sKGZpZWxkLCBjb250cm9sKTtcbiAgfVxuXG4gIHByaXZhdGUgc2V0VmFsaWRhdG9ycyhmaWVsZDogRm9ybWx5RmllbGRDb25maWdDYWNoZSwgZGlzYWJsZWQgPSBmYWxzZSkge1xuICAgIGlmIChkaXNhYmxlZCA9PT0gZmFsc2UgJiYgaGFzS2V5KGZpZWxkKSAmJiBmaWVsZC5wcm9wcz8uZGlzYWJsZWQpIHtcbiAgICAgIGRpc2FibGVkID0gdHJ1ZTtcbiAgICB9XG5cbiAgICBsZXQgbWFya0ZvckNoZWNrID0gZmFsc2U7XG4gICAgZmllbGQuZmllbGRHcm91cD8uZm9yRWFjaCgoZikgPT4gZiAmJiB0aGlzLnNldFZhbGlkYXRvcnMoZiwgZGlzYWJsZWQpICYmIChtYXJrRm9yQ2hlY2sgPSB0cnVlKSk7XG4gICAgaWYgKGhhc0tleShmaWVsZCkgfHwgIWZpZWxkLnBhcmVudCB8fCAoIWhhc0tleShmaWVsZCkgJiYgIWZpZWxkLmZpZWxkR3JvdXApKSB7XG4gICAgICBjb25zdCB7IGZvcm1Db250cm9sOiBjIH0gPSBmaWVsZDtcbiAgICAgIGlmIChjKSB7XG4gICAgICAgIGlmIChoYXNLZXkoZmllbGQpICYmIGMgaW5zdGFuY2VvZiBGb3JtQ29udHJvbCkge1xuICAgICAgICAgIGlmIChkaXNhYmxlZCAmJiBjLmVuYWJsZWQpIHtcbiAgICAgICAgICAgIGMuZGlzYWJsZSh7IGVtaXRFdmVudDogZmFsc2UsIG9ubHlTZWxmOiB0cnVlIH0pO1xuICAgICAgICAgICAgbWFya0ZvckNoZWNrID0gdHJ1ZTtcbiAgICAgICAgICB9XG5cbiAgICAgICAgICBpZiAoIWRpc2FibGVkICYmIGMuZGlzYWJsZWQpIHtcbiAgICAgICAgICAgIGMuZW5hYmxlKHsgZW1pdEV2ZW50OiBmYWxzZSwgb25seVNlbGY6IHRydWUgfSk7XG4gICAgICAgICAgICBtYXJrRm9yQ2hlY2sgPSB0cnVlO1xuICAgICAgICAgIH1cbiAgICAgICAgfVxuXG4gICAgICAgIGlmIChudWxsID09PSBjLnZhbGlkYXRvciAmJiB0aGlzLmhhc1ZhbGlkYXRvcnMoZmllbGQsICdfdmFsaWRhdG9ycycpKSB7XG4gICAgICAgICAgYy5zZXRWYWxpZGF0b3JzKCgpID0+IHtcbiAgICAgICAgICAgIGNvbnN0IHYgPSBWYWxpZGF0b3JzLmNvbXBvc2UodGhpcy5tZXJnZVZhbGlkYXRvcnM8VmFsaWRhdG9yRm4+KGZpZWxkLCAnX3ZhbGlkYXRvcnMnKSk7XG5cbiAgICAgICAgICAgIHJldHVybiB2ID8gdihjKSA6IG51bGw7XG4gICAgICAgICAgfSk7XG4gICAgICAgICAgbWFya0ZvckNoZWNrID0gdHJ1ZTtcbiAgICAgICAgfVxuXG4gICAgICAgIGlmIChudWxsID09PSBjLmFzeW5jVmFsaWRhdG9yICYmIHRoaXMuaGFzVmFsaWRhdG9ycyhmaWVsZCwgJ19hc3luY1ZhbGlkYXRvcnMnKSkge1xuICAgICAgICAgIGMuc2V0QXN5bmNWYWxpZGF0b3JzKCgpID0+IHtcbiAgICAgICAgICAgIGNvbnN0IHYgPSBWYWxpZGF0b3JzLmNvbXBvc2VBc3luYyh0aGlzLm1lcmdlVmFsaWRhdG9yczxBc3luY1ZhbGlkYXRvckZuPihmaWVsZCwgJ19hc3luY1ZhbGlkYXRvcnMnKSk7XG4gICAgICAgICAgICByZXR1cm4gdiA/IHYoYykgOiBvZihudWxsKTtcbiAgICAgICAgICB9KTtcbiAgICAgICAgICBtYXJrRm9yQ2hlY2sgPSB0cnVlO1xuICAgICAgICB9XG5cbiAgICAgICAgaWYgKG1hcmtGb3JDaGVjaykge1xuICAgICAgICAgIHVwZGF0ZVZhbGlkaXR5KGMsIHRydWUpO1xuXG4gICAgICAgICAgLy8gdXBkYXRlIHZhbGlkaXR5IG9mIGBGb3JtR3JvdXBgIGluc3RhbmNlIGNyZWF0ZWQgYnkgZmllbGQgd2l0aCBuZXN0ZWQga2V5LlxuICAgICAgICAgIGxldCBwYXJlbnQgPSBjLnBhcmVudDtcbiAgICAgICAgICBmb3IgKGxldCBpID0gMTsgaSA8IGdldEtleVBhdGgoZmllbGQpLmxlbmd0aDsgaSsrKSB7XG4gICAgICAgICAgICBpZiAocGFyZW50KSB7XG4gICAgICAgICAgICAgIHVwZGF0ZVZhbGlkaXR5KHBhcmVudCwgdHJ1ZSk7XG4gICAgICAgICAgICAgIHBhcmVudCA9IHBhcmVudC5wYXJlbnQ7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfVxuXG4gICAgcmV0dXJuIG1hcmtGb3JDaGVjaztcbiAgfVxuXG4gIHByaXZhdGUgaGFzVmFsaWRhdG9ycyhmaWVsZDogRm9ybWx5RmllbGRDb25maWdDYWNoZSwgdHlwZTogJ192YWxpZGF0b3JzJyB8ICdfYXN5bmNWYWxpZGF0b3JzJyk6IGJvb2xlYW4ge1xuICAgIGNvbnN0IGMgPSBmaWVsZC5mb3JtQ29udHJvbDtcbiAgICBpZiAoYz8uX2ZpZWxkcz8ubGVuZ3RoID4gMSAmJiBjLl9maWVsZHMuc29tZSgoZikgPT4gZlt0eXBlXS5sZW5ndGggPiAwKSkge1xuICAgICAgcmV0dXJuIHRydWU7XG4gICAgfSBlbHNlIGlmIChmaWVsZFt0eXBlXS5sZW5ndGggPiAwKSB7XG4gICAgICByZXR1cm4gdHJ1ZTtcbiAgICB9XG5cbiAgICByZXR1cm4gZmllbGQuZmllbGRHcm91cD8uc29tZSgoZikgPT4gZj8uZmllbGRHcm91cCAmJiAhaGFzS2V5KGYpICYmIHRoaXMuaGFzVmFsaWRhdG9ycyhmLCB0eXBlKSk7XG4gIH1cblxuICBwcml2YXRlIG1lcmdlVmFsaWRhdG9yczxUPihmaWVsZDogRm9ybWx5RmllbGRDb25maWdDYWNoZSwgdHlwZTogJ192YWxpZGF0b3JzJyB8ICdfYXN5bmNWYWxpZGF0b3JzJyk6IFRbXSB7XG4gICAgY29uc3QgdmFsaWRhdG9yczogYW55ID0gW107XG4gICAgY29uc3QgYyA9IGZpZWxkLmZvcm1Db250cm9sO1xuICAgIGlmIChjPy5fZmllbGRzPy5sZW5ndGggPiAxKSB7XG4gICAgICBjLl9maWVsZHNcbiAgICAgICAgLmZpbHRlcigoZjogRm9ybWx5RmllbGRDb25maWdDYWNoZSkgPT4gIWYuX2hpZGUpXG4gICAgICAgIC5mb3JFYWNoKChmOiBGb3JtbHlGaWVsZENvbmZpZ0NhY2hlKSA9PiB2YWxpZGF0b3JzLnB1c2goLi4uZlt0eXBlXSkpO1xuICAgIH0gZWxzZSBpZiAoZmllbGRbdHlwZV0pIHtcbiAgICAgIHZhbGlkYXRvcnMucHVzaCguLi5maWVsZFt0eXBlXSk7XG4gICAgfVxuXG4gICAgaWYgKGZpZWxkLmZpZWxkR3JvdXApIHtcbiAgICAgIGZpZWxkLmZpZWxkR3JvdXBcbiAgICAgICAgLmZpbHRlcigoZikgPT4gZj8uZmllbGRHcm91cCAmJiAhaGFzS2V5KGYpKVxuICAgICAgICAuZm9yRWFjaCgoZikgPT4gdmFsaWRhdG9ycy5wdXNoKC4uLnRoaXMubWVyZ2VWYWxpZGF0b3JzKGYsIHR5cGUpKSk7XG4gICAgfVxuXG4gICAgcmV0dXJuIHZhbGlkYXRvcnM7XG4gIH1cbn1cbiJdfQ==