import { isObject, isNil, isUndefined, isFunction, defineHiddenProp, observe, getFieldValue, assignFieldValue, hasKey, } from '../../utils';
import { evalExpression, evalStringExpression } from './utils';
import { isObservable, Observable } from 'rxjs';
import { tap } from 'rxjs/operators';
import { unregisterControl, registerControl, updateValidity } from '../field-form/utils';
import { UntypedFormArray } from '@angular/forms';
export class FieldExpressionExtension {
    onPopulate(field) {
        if (field._expressions) {
            return;
        }
        // cache built expression
        defineHiddenProp(field, '_expressions', {});
        observe(field, ['hide'], ({ currentValue, firstChange }) => {
            defineHiddenProp(field, '_hide', !!currentValue);
            if (!firstChange || (firstChange && currentValue === true)) {
                field.props.hidden = currentValue;
                field.options._hiddenFieldsForCheck.push({ field });
            }
        });
        if (field.hideExpression) {
            observe(field, ['hideExpression'], ({ currentValue: expr }) => {
                field._expressions.hide = this.parseExpressions(field, 'hide', typeof expr === 'boolean' ? () => expr : expr);
            });
        }
        const evalExpr = (key, expr) => {
            if (typeof expr === 'string' || isFunction(expr)) {
                field._expressions[key] = this.parseExpressions(field, key, expr);
            }
            else if (expr instanceof Observable) {
                field._expressions[key] = {
                    value$: expr.pipe(tap((v) => {
                        this.evalExpr(field, key, v);
                        field.options._detectChanges(field);
                    })),
                };
            }
        };
        field.expressions = field.expressions || {};
        for (const key of Object.keys(field.expressions)) {
            observe(field, ['expressions', key], ({ currentValue: expr }) => {
                evalExpr(key, isFunction(expr) ? (...args) => expr(field, args[3]) : expr);
            });
        }
        field.expressionProperties = field.expressionProperties || {};
        for (const key of Object.keys(field.expressionProperties)) {
            observe(field, ['expressionProperties', key], ({ currentValue }) => evalExpr(key, currentValue));
        }
    }
    postPopulate(field) {
        if (field.parent) {
            return;
        }
        if (!field.options.checkExpressions) {
            let checkLocked = false;
            field.options.checkExpressions = (f, ignoreCache) => {
                if (checkLocked) {
                    return;
                }
                checkLocked = true;
                const fieldChanged = this.checkExpressions(f, ignoreCache);
                const options = field.options;
                options._hiddenFieldsForCheck
                    .sort((f) => (f.field.hide ? -1 : 1))
                    .forEach((f) => this.changeHideState(f.field, f.field.hide ?? f.default, !ignoreCache));
                options._hiddenFieldsForCheck = [];
                if (fieldChanged) {
                    this.checkExpressions(field);
                }
                checkLocked = false;
            };
        }
    }
    parseExpressions(field, path, expr) {
        let parentExpression;
        if (field.parent && ['hide', 'props.disabled'].includes(path)) {
            const rootValue = (f) => {
                return path === 'hide' ? f.hide : f.props.disabled;
            };
            parentExpression = () => {
                let root = field.parent;
                while (root.parent && !rootValue(root)) {
                    root = root.parent;
                }
                return rootValue(root);
            };
        }
        expr = expr || (() => false);
        if (typeof expr === 'string') {
            expr = evalStringExpression(expr, ['model', 'formState', 'field']);
        }
        let currentValue;
        return {
            callback: (ignoreCache) => {
                try {
                    const exprValue = evalExpression(parentExpression ? (...args) => parentExpression(field) || expr(...args) : expr, { field }, [field.model, field.options.formState, field, ignoreCache]);
                    if (ignoreCache ||
                        (currentValue !== exprValue &&
                            (!isObject(exprValue) ||
                                isObservable(exprValue) ||
                                JSON.stringify(exprValue) !== JSON.stringify(currentValue)))) {
                        currentValue = exprValue;
                        this.evalExpr(field, path, exprValue);
                        return true;
                    }
                    return false;
                }
                catch (error) {
                    error.message = `[Formly Error] [Expression "${path}"] ${error.message}`;
                    throw error;
                }
            },
        };
    }
    checkExpressions(field, ignoreCache = false) {
        if (!field) {
            return false;
        }
        let fieldChanged = false;
        if (field._expressions) {
            for (const key of Object.keys(field._expressions)) {
                field._expressions[key].callback?.(ignoreCache) && (fieldChanged = true);
            }
        }
        field.fieldGroup?.forEach((f) => this.checkExpressions(f, ignoreCache) && (fieldChanged = true));
        return fieldChanged;
    }
    changeDisabledState(field, value) {
        if (field.fieldGroup) {
            field.fieldGroup
                .filter((f) => !f._expressions.hasOwnProperty('props.disabled'))
                .forEach((f) => this.changeDisabledState(f, value));
        }
        if (hasKey(field) && field.props.disabled !== value) {
            field.props.disabled = value;
        }
    }
    changeHideState(field, hide, resetOnHide) {
        if (field.fieldGroup) {
            field.fieldGroup
                .filter((f) => f && !f._expressions.hide)
                .forEach((f) => this.changeHideState(f, hide, resetOnHide));
        }
        if (field.formControl && hasKey(field)) {
            defineHiddenProp(field, '_hide', !!(hide || field.hide));
            const c = field.formControl;
            if (c._fields?.length > 1) {
                updateValidity(c);
            }
            if (hide === true && (!c._fields || c._fields.every((f) => !!f._hide))) {
                unregisterControl(field, true);
                if (resetOnHide && field.resetOnHide) {
                    assignFieldValue(field, undefined);
                    field.formControl.reset({ value: undefined, disabled: field.formControl.disabled });
                    field.options.fieldChanges.next({ value: undefined, field, type: 'valueChanges' });
                    if (field.fieldGroup && field.formControl instanceof UntypedFormArray) {
                        field.fieldGroup.length = 0;
                    }
                }
            }
            else if (hide === false) {
                if (field.resetOnHide && !isUndefined(field.defaultValue) && isUndefined(getFieldValue(field))) {
                    assignFieldValue(field, field.defaultValue);
                }
                registerControl(field, undefined, true);
                if (field.resetOnHide && field.fieldArray && field.fieldGroup?.length !== field.model?.length) {
                    field.options.build(field);
                }
            }
        }
        if (field.options.fieldChanges) {
            field.options.fieldChanges.next({ field, type: 'hidden', value: hide });
        }
    }
    evalExpr(field, prop, value) {
        if (prop.indexOf('model.') === 0) {
            const key = prop.replace(/^model\./, ''), parent = field.fieldGroup ? field : field.parent;
            let control = field?.key === key ? field.formControl : field.form.get(key);
            if (!control && field.get(key)) {
                control = field.get(key).formControl;
            }
            assignFieldValue({ key, parent, model: field.model }, value);
            if (control && !(isNil(control.value) && isNil(value)) && control.value !== value) {
                control.patchValue(value);
            }
        }
        else {
            try {
                let target = field;
                const paths = this._evalExpressionPath(field, prop);
                const lastIndex = paths.length - 1;
                for (let i = 0; i < lastIndex; i++) {
                    target = target[paths[i]];
                }
                target[paths[lastIndex]] = value;
            }
            catch (error) {
                error.message = `[Formly Error] [Expression "${prop}"] ${error.message}`;
                throw error;
            }
            if (['templateOptions.disabled', 'props.disabled'].includes(prop) && hasKey(field)) {
                this.changeDisabledState(field, value);
            }
        }
        this.emitExpressionChanges(field, prop, value);
    }
    emitExpressionChanges(field, property, value) {
        if (!field.options.fieldChanges) {
            return;
        }
        field.options.fieldChanges.next({
            field,
            type: 'expressionChanges',
            property,
            value,
        });
    }
    _evalExpressionPath(field, prop) {
        if (field._expressions[prop] && field._expressions[prop].paths) {
            return field._expressions[prop].paths;
        }
        let paths = [];
        if (prop.indexOf('[') === -1) {
            paths = prop.split('.');
        }
        else {
            prop
                .split(/[[\]]{1,2}/) // https://stackoverflow.com/a/20198206
                .filter((p) => p)
                .forEach((path) => {
                const arrayPath = path.match(/['|"](.*?)['|"]/);
                if (arrayPath) {
                    paths.push(arrayPath[1]);
                }
                else {
                    paths.push(...path.split('.').filter((p) => p));
                }
            });
        }
        if (field._expressions[prop]) {
            field._expressions[prop].paths = paths;
        }
        return paths;
    }
}
//# sourceMappingURL=data:application/json;base64,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