import { ComponentRef } from '@angular/core';
import { getFieldId, assignFieldValue, isUndefined, getFieldValue, reverseDeepMerge, defineHiddenProp, clone, getField, markFieldForCheck, hasKey, observe, isHiddenField, isSignalRequired, } from '../../utils';
import { Subject } from 'rxjs';
export class CoreExtension {
    constructor(config) {
        this.config = config;
        this.formId = 0;
    }
    prePopulate(field) {
        const root = field.parent;
        this.initRootOptions(field);
        this.initFieldProps(field);
        if (root) {
            Object.defineProperty(field, 'options', { get: () => root.options, configurable: true });
            Object.defineProperty(field, 'model', {
                get: () => (hasKey(field) && field.fieldGroup ? getFieldValue(field) : root.model),
                configurable: true,
            });
        }
        Object.defineProperty(field, 'get', {
            value: (key) => getField(field, key),
            configurable: true,
        });
        this.getFieldComponentInstance(field).prePopulate?.(field);
    }
    onPopulate(field) {
        this.initFieldOptions(field);
        this.getFieldComponentInstance(field).onPopulate?.(field);
        if (field.fieldGroup) {
            field.fieldGroup.forEach((f, index) => {
                if (f) {
                    Object.defineProperty(f, 'parent', { get: () => field, configurable: true });
                    Object.defineProperty(f, 'index', { get: () => index, configurable: true });
                }
                this.formId++;
            });
        }
    }
    postPopulate(field) {
        this.getFieldComponentInstance(field).postPopulate?.(field);
    }
    initFieldProps(field) {
        field.props ??= field.templateOptions;
        Object.defineProperty(field, 'templateOptions', {
            get: () => field.props,
            set: (props) => (field.props = props),
            configurable: true,
        });
    }
    initRootOptions(field) {
        if (field.parent) {
            return;
        }
        const options = field.options;
        field.options.formState = field.options.formState || {};
        if (!options.showError) {
            options.showError = this.config.extras.showError;
        }
        if (!options.fieldChanges) {
            defineHiddenProp(options, 'fieldChanges', new Subject());
        }
        if (!options._hiddenFieldsForCheck) {
            options._hiddenFieldsForCheck = [];
        }
        options._detectChanges = (f) => {
            if (f._componentRefs) {
                markFieldForCheck(f);
            }
            f.fieldGroup?.forEach((f) => f && options._detectChanges(f));
        };
        options.detectChanges = (f) => {
            f.options.checkExpressions?.(f);
            options._detectChanges(f);
        };
        options.resetModel = (model) => {
            model = clone(model ?? options._initialModel);
            if (field.model) {
                Object.keys(field.model).forEach((k) => delete field.model[k]);
                Object.assign(field.model, model || {});
            }
            if (!isSignalRequired()) {
                observe(options, ['parentForm', 'submitted']).setValue(false, false);
            }
            options.build(field);
            field.form.reset(field.model);
        };
        options.updateInitialValue = (model) => (options._initialModel = clone(model ?? field.model));
        field.options.updateInitialValue();
    }
    initFieldOptions(field) {
        reverseDeepMerge(field, {
            id: getFieldId(`formly_${this.formId}`, field, field.index),
            hooks: {},
            modelOptions: {},
            validation: { messages: {} },
            props: !field.type || !hasKey(field)
                ? {}
                : {
                    label: '',
                    placeholder: '',
                    disabled: false,
                },
        });
        if (this.config.extras.resetFieldOnHide && field.resetOnHide !== false) {
            field.resetOnHide = true;
        }
        if (field.type !== 'formly-template' &&
            (field.template || field.expressions?.template || field.expressionProperties?.template)) {
            field.type = 'formly-template';
        }
        if (!field.type && field.fieldGroup) {
            field.type = 'formly-group';
        }
        if (field.type) {
            this.config.getMergedField(field);
        }
        if (hasKey(field) &&
            !isUndefined(field.defaultValue) &&
            isUndefined(getFieldValue(field)) &&
            !isHiddenField(field)) {
            assignFieldValue(field, field.defaultValue);
        }
        field.wrappers = field.wrappers || [];
    }
    getFieldComponentInstance(field) {
        const componentRefInstance = () => {
            let componentRef = this.config.resolveFieldTypeRef(field);
            const fieldComponentRef = field._componentRefs?.slice(-1)[0];
            if (fieldComponentRef instanceof ComponentRef &&
                fieldComponentRef?.componentType === componentRef?.componentType) {
                componentRef = fieldComponentRef;
            }
            return componentRef?.instance;
        };
        if (!field._proxyInstance) {
            defineHiddenProp(field, '_proxyInstance', new Proxy({}, {
                get: (_, prop) => componentRefInstance()?.[prop],
                set: (_, prop, value) => (componentRefInstance()[prop] = value),
            }));
        }
        return field._proxyInstance;
    }
}
//# sourceMappingURL=data:application/json;base64,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