import { Component, Input, ViewContainerRef, ViewChild, ComponentRef, Optional, } from '@angular/core';
import { FormControl } from '@angular/forms';
import { defineHiddenProp, observe, observeDeep, getFieldValue, assignFieldValue, isObject, markFieldForCheck, hasKey, isSignalRequired, } from '../utils';
import { isObservable } from 'rxjs';
import { debounceTime, distinctUntilChanged, map, startWith } from 'rxjs/operators';
import * as i0 from "@angular/core";
import * as i1 from "../services/formly.config";
import * as i2 from "./formly.template";
/**
 * The `<formly-field>` component is used to render the UI widget (layout + type) of a given `field`.
 */
export class FormlyField {
    get containerRef() {
        return this.config.extras.renderFormlyFieldElement ? this.viewContainerRef : this.hostContainerRef;
    }
    get elementRef() {
        if (this.config.extras.renderFormlyFieldElement) {
            return this._elementRef;
        }
        if (this.componentRefs?.[0] instanceof ComponentRef) {
            return this.componentRefs[0].location;
        }
        return null;
    }
    constructor(config, renderer, _elementRef, hostContainerRef, form) {
        this.config = config;
        this.renderer = renderer;
        this._elementRef = _elementRef;
        this.hostContainerRef = hostContainerRef;
        this.form = form;
        this.hostObservers = [];
        this.componentRefs = [];
        this.hooksObservers = [];
        this.detectFieldBuild = false;
        this.valueChangesUnsubscribe = () => { };
    }
    ngAfterContentInit() {
        this.triggerHook('afterContentInit');
    }
    ngAfterViewInit() {
        this.triggerHook('afterViewInit');
    }
    ngDoCheck() {
        if (this.detectFieldBuild && this.field && this.field.options) {
            this.render();
        }
    }
    ngOnInit() {
        this.triggerHook('onInit');
    }
    ngOnChanges(changes) {
        this.triggerHook('onChanges', changes);
    }
    ngOnDestroy() {
        this.resetRefs(this.field);
        this.hostObservers.forEach((hostObserver) => hostObserver.unsubscribe());
        this.hooksObservers.forEach((unsubscribe) => unsubscribe());
        this.valueChangesUnsubscribe();
        this.triggerHook('onDestroy');
    }
    renderField(containerRef, f, wrappers = []) {
        if (this.containerRef === containerRef) {
            this.resetRefs(this.field);
            this.containerRef.clear();
            wrappers = this.field?.wrappers;
        }
        if (wrappers?.length > 0) {
            const [wrapper, ...wps] = wrappers;
            const { component } = this.config.getWrapper(wrapper);
            const ref = containerRef.createComponent(component);
            this.attachComponentRef(ref, f);
            observe(ref.instance, ['fieldComponent'], ({ currentValue, previousValue, firstChange }) => {
                if (currentValue) {
                    if (previousValue && previousValue._lContainer === currentValue._lContainer) {
                        return;
                    }
                    const viewRef = previousValue ? previousValue.detach() : null;
                    if (viewRef && !viewRef.destroyed) {
                        currentValue.insert(viewRef);
                    }
                    else {
                        this.renderField(currentValue, f, wps);
                    }
                    !firstChange && ref.changeDetectorRef.detectChanges();
                }
            });
        }
        else if (f?.type) {
            const inlineType = this.form?.templates?.find((ref) => ref.name === f.type);
            let ref;
            if (inlineType) {
                ref = containerRef.createEmbeddedView(inlineType.ref, { $implicit: f });
            }
            else {
                const { component } = this.config.getType(f.type, true);
                ref = containerRef.createComponent(component);
            }
            this.attachComponentRef(ref, f);
        }
    }
    triggerHook(name, changes) {
        if (name === 'onInit' || (name === 'onChanges' && changes.field && !changes.field.firstChange)) {
            this.valueChangesUnsubscribe();
            this.valueChangesUnsubscribe = this.fieldChanges(this.field);
        }
        if (this.field?.hooks?.[name]) {
            if (!changes || changes.field) {
                const r = this.field.hooks[name](this.field);
                if (isObservable(r) && ['onInit', 'afterContentInit', 'afterViewInit'].indexOf(name) !== -1) {
                    const sub = r.subscribe();
                    this.hooksObservers.push(() => sub.unsubscribe());
                }
            }
        }
        if (name === 'onChanges' && changes.field) {
            this.resetRefs(changes.field.previousValue);
            this.render();
        }
    }
    attachComponentRef(ref, field) {
        this.componentRefs.push(ref);
        field._componentRefs.push(ref);
        if (ref instanceof ComponentRef) {
            Object.assign(ref.instance, { field });
        }
    }
    render() {
        if (!this.field) {
            return;
        }
        // require Formly build
        if (!this.field.options) {
            this.detectFieldBuild = true;
            return;
        }
        this.detectFieldBuild = false;
        this.hostObservers.forEach((hostObserver) => hostObserver.unsubscribe());
        this.hostObservers = [
            observe(this.field, ['hide'], ({ firstChange, currentValue }) => {
                const containerRef = this.containerRef;
                if (this.config.extras.lazyRender === false) {
                    firstChange && this.renderField(containerRef, this.field);
                    if (!firstChange || (firstChange && currentValue)) {
                        this.elementRef &&
                            this.renderer.setStyle(this.elementRef.nativeElement, 'display', currentValue ? 'none' : '');
                    }
                }
                else {
                    if (currentValue) {
                        containerRef.clear();
                        if (this.field.className) {
                            this.renderer.removeAttribute(this.elementRef.nativeElement, 'class');
                        }
                    }
                    else {
                        this.renderField(containerRef, this.field);
                        if (this.field.className) {
                            this.renderer.setAttribute(this.elementRef.nativeElement, 'class', this.field.className);
                        }
                    }
                }
                !firstChange && this.field.options.detectChanges(this.field);
            }),
            observe(this.field, ['className'], ({ firstChange, currentValue }) => {
                if ((!firstChange || (firstChange && currentValue)) &&
                    (!this.config.extras.lazyRender || this.field.hide !== true)) {
                    this.elementRef && this.renderer.setAttribute(this.elementRef.nativeElement, 'class', currentValue);
                }
            }),
        ];
        if (!isSignalRequired()) {
            ['touched', 'pristine', 'status'].forEach((prop) => this.hostObservers.push(observe(this.field, ['formControl', prop], ({ firstChange }) => !firstChange && markFieldForCheck(this.field))));
        }
        else if (this.field.formControl) {
            const events = this.field.formControl.events.subscribe(() => markFieldForCheck(this.field));
            this.hostObservers.push(events);
        }
    }
    resetRefs(field) {
        if (field) {
            if (field._localFields) {
                field._localFields = [];
            }
            else {
                defineHiddenProp(this.field, '_localFields', []);
            }
            if (field._componentRefs) {
                field._componentRefs = field._componentRefs.filter((ref) => this.componentRefs.indexOf(ref) === -1);
            }
            else {
                defineHiddenProp(this.field, '_componentRefs', []);
            }
        }
        this.componentRefs = [];
    }
    fieldChanges(field) {
        if (!field) {
            return () => { };
        }
        const propsObserver = observeDeep(field, ['props'], () => field.options.detectChanges(field));
        const subscribes = [
            () => {
                propsObserver();
            },
        ];
        for (const key of Object.keys(field._expressions || {})) {
            const expressionObserver = observe(field, ['_expressions', key], ({ currentValue, previousValue }) => {
                if (previousValue?.subscription) {
                    previousValue.subscription.unsubscribe();
                    previousValue.subscription = null;
                }
                if (isObservable(currentValue.value$)) {
                    currentValue.subscription = currentValue.value$.subscribe();
                }
            });
            subscribes.push(() => {
                if (field._expressions[key]?.subscription) {
                    field._expressions[key].subscription.unsubscribe();
                }
                expressionObserver.unsubscribe();
            });
        }
        for (const path of [['focus'], ['template'], ['fieldGroupClassName'], ['validation', 'show']]) {
            const fieldObserver = observe(field, path, ({ firstChange }) => !firstChange && field.options.detectChanges(field));
            subscribes.push(() => fieldObserver.unsubscribe());
        }
        if (field.formControl && !field.fieldGroup) {
            const control = field.formControl;
            let valueChanges = control.valueChanges.pipe(map((value) => {
                field.parsers?.map((parserFn) => (value = parserFn(value, field)));
                if (!Object.is(value, field.formControl.value)) {
                    field.formControl.setValue(value);
                }
                return value;
            }), distinctUntilChanged((x, y) => {
                if (x !== y || Array.isArray(x) || isObject(x)) {
                    return false;
                }
                return true;
            }));
            if (control.value !== getFieldValue(field)) {
                valueChanges = valueChanges.pipe(startWith(control.value));
            }
            const { updateOn, debounce } = field.modelOptions;
            if ((!updateOn || updateOn === 'change') && debounce?.default > 0) {
                valueChanges = valueChanges.pipe(debounceTime(debounce.default));
            }
            const sub = valueChanges.subscribe((value) => {
                // workaround for https://github.com/angular/angular/issues/13792
                if (control._fields?.length > 1 && control instanceof FormControl) {
                    control.patchValue(value, { emitEvent: false, onlySelf: true });
                }
                if (hasKey(field)) {
                    assignFieldValue(field, value);
                }
                field.options.fieldChanges.next({ value, field, type: 'valueChanges' });
            });
            subscribes.push(() => sub.unsubscribe());
        }
        let templateFieldsSubs = [];
        observe(field, ['_localFields'], ({ currentValue }) => {
            templateFieldsSubs.forEach((unsubscribe) => unsubscribe());
            templateFieldsSubs = (currentValue || []).map((f) => this.fieldChanges(f));
        });
        return () => {
            subscribes.forEach((unsubscribe) => unsubscribe());
            templateFieldsSubs.forEach((unsubscribe) => unsubscribe());
        };
    }
    static { this.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "18.2.13", ngImport: i0, type: FormlyField, deps: [{ token: i1.FormlyConfig }, { token: i0.Renderer2 }, { token: i0.ElementRef }, { token: i0.ViewContainerRef }, { token: i2.FormlyFieldTemplates, optional: true }], target: i0.ɵɵFactoryTarget.Component }); }
    static { this.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "14.0.0", version: "18.2.13", type: FormlyField, isStandalone: true, selector: "formly-field", inputs: { field: "field" }, viewQueries: [{ propertyName: "viewContainerRef", first: true, predicate: ["container"], descendants: true, read: ViewContainerRef, static: true }], usesOnChanges: true, ngImport: i0, template: '<ng-template #container></ng-template>', isInline: true, styles: [":host:empty{display:none}\n"] }); }
}
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "18.2.13", ngImport: i0, type: FormlyField, decorators: [{
            type: Component,
            args: [{ selector: 'formly-field', template: '<ng-template #container></ng-template>', standalone: true, styles: [":host:empty{display:none}\n"] }]
        }], ctorParameters: () => [{ type: i1.FormlyConfig }, { type: i0.Renderer2 }, { type: i0.ElementRef }, { type: i0.ViewContainerRef }, { type: i2.FormlyFieldTemplates, decorators: [{
                    type: Optional
                }] }], propDecorators: { field: [{
                type: Input
            }], viewContainerRef: [{
                type: ViewChild,
                args: ['container', { read: ViewContainerRef, static: true }]
            }] } });
export class LegacyFormlyField extends FormlyField {
    static { this.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "18.2.13", ngImport: i0, type: LegacyFormlyField, deps: null, target: i0.ɵɵFactoryTarget.Component }); }
    static { this.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "14.0.0", version: "18.2.13", type: LegacyFormlyField, selector: "formly-field", usesInheritance: true, ngImport: i0, template: '<ng-template #container></ng-template>', isInline: true, styles: [":host:empty{display:none}\n"] }); }
}
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "18.2.13", ngImport: i0, type: LegacyFormlyField, decorators: [{
            type: Component,
            args: [{ selector: 'formly-field', template: '<ng-template #container></ng-template>', standalone: false, styles: [":host:empty{display:none}\n"] }]
        }] });
//# sourceMappingURL=data:application/json;base64,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