import { Injectable } from '@angular/core';
import { UntypedFormArray, UntypedFormGroup } from '@angular/forms';
import { ɵreverseDeepMerge as reverseDeepMerge, ɵgetFieldValue as getFieldValue, ɵclone as clone, ɵhasKey as hasKey, } from '@ngx-formly/core';
import { tap } from 'rxjs/operators';
import * as i0 from "@angular/core";
// https://stackoverflow.com/a/27865285
function decimalPlaces(a) {
    if (!isFinite(a)) {
        return 0;
    }
    let e = 1, p = 0;
    while (Math.round(a * e) / e !== a) {
        e *= 10;
        p++;
    }
    return p;
}
function isEmpty(v) {
    return v === '' || v == null;
}
function isObject(v) {
    return v != null && typeof v === 'object' && !Array.isArray(v);
}
function isInteger(value) {
    return Number.isInteger ? Number.isInteger(value) : typeof value === 'number' && Math.floor(value) === value;
}
function isConst(schema) {
    return typeof schema === 'object' && (schema.hasOwnProperty('const') || (schema.enum && schema.enum.length === 1));
}
function toNumber(value) {
    if (value === '' || value === undefined) {
        return undefined;
    }
    if (value === null) {
        return null;
    }
    if (typeof value === 'number') {
        return value;
    }
    const val = parseFloat(value);
    return !isNaN(val) ? val : value;
}
function totalMatchedFields(field) {
    if (!field.fieldGroup) {
        return hasKey(field) && getFieldValue(field) !== undefined ? 1 : 0;
    }
    const total = field.fieldGroup.reduce((s, f) => totalMatchedFields(f) + s, 0);
    if (total === 0 && hasKey(field)) {
        const value = getFieldValue(field);
        if (value === null ||
            (value !== undefined && ((field.fieldArray && Array.isArray(value)) || (!field.fieldArray && isObject(value))))) {
            return 1;
        }
    }
    return total;
}
export class FormlyJsonschema {
    toFieldConfig(schema, options) {
        schema = clone(schema);
        return this._toFieldConfig(schema, { schema, ...(options || {}) });
    }
    _toFieldConfig(schema, { key, isOptional, ...options }) {
        schema = this.resolveSchema(schema, options);
        const types = this.guessSchemaType(schema);
        let field = {
            type: types[0],
            defaultValue: schema.default,
            props: {
                label: schema.title,
                readonly: schema.readOnly,
                description: schema.description,
            },
        };
        if (key != null) {
            field.key = key;
        }
        if (!options.ignoreDefault && (schema.readOnly || options.readOnly)) {
            field.props.disabled = true;
            options = { ...options, readOnly: true };
        }
        if (options.resetOnHide) {
            field.resetOnHide = true;
        }
        if (options.shareFormControl === false) {
            field.shareFormControl = false;
        }
        if (field.defaultValue === undefined && types.length === 1 && isOptional === false) {
            switch (types[0]) {
                case 'null': {
                    field.defaultValue = null;
                    break;
                }
                case 'string': {
                    field.defaultValue = '';
                    break;
                }
                case 'object': {
                    field.defaultValue = {};
                    break;
                }
                case 'array': {
                    field.defaultValue = schema.minItems > 0 ? Array.from(new Array(schema.minItems)) : [];
                    break;
                }
            }
        }
        if (options.ignoreDefault) {
            delete field.defaultValue;
        }
        this.addValidator(field, 'type', {
            schemaType: types,
            expression: ({ value }) => {
                if (value === undefined) {
                    return true;
                }
                if (value === null && types.indexOf('null') !== -1) {
                    return true;
                }
                switch (types[0]) {
                    case 'null': {
                        return typeof value === null;
                    }
                    case 'string': {
                        return typeof value === 'string';
                    }
                    case 'integer': {
                        return isInteger(value);
                    }
                    case 'number': {
                        return typeof value === 'number';
                    }
                    case 'object': {
                        return isObject(value);
                    }
                    case 'array': {
                        return Array.isArray(value);
                    }
                }
                return true;
            },
        });
        switch (field.type) {
            case 'number':
            case 'integer': {
                field.parsers = [
                    (v, f) => {
                        v = toNumber(v);
                        if (v === null && f) {
                            const input = typeof document !== 'undefined' && f.id
                                ? document.querySelector(`#${f.id}`)
                                : undefined;
                            if (input && !input.validity.badInput) {
                                v = undefined;
                            }
                            if (v !== f.formControl.value) {
                                f.formControl.setValue(v, { emitModelToViewChange: false });
                            }
                        }
                        return v;
                    },
                ];
                if (schema.hasOwnProperty('minimum')) {
                    field.props.min = schema.minimum;
                }
                if (schema.hasOwnProperty('maximum')) {
                    field.props.max = schema.maximum;
                }
                if (schema.hasOwnProperty('exclusiveMinimum')) {
                    field.props.exclusiveMinimum = schema.exclusiveMinimum;
                    this.addValidator(field, 'exclusiveMinimum', ({ value }) => isEmpty(value) || value > schema.exclusiveMinimum);
                }
                if (schema.hasOwnProperty('exclusiveMaximum')) {
                    field.props.exclusiveMaximum = schema.exclusiveMaximum;
                    this.addValidator(field, 'exclusiveMaximum', ({ value }) => isEmpty(value) || value < schema.exclusiveMaximum);
                }
                if (schema.hasOwnProperty('multipleOf')) {
                    field.props.step = schema.multipleOf;
                    this.addValidator(field, 'multipleOf', ({ value }) => {
                        if (isEmpty(value) || typeof value !== 'number' || value === 0 || schema.multipleOf <= 0) {
                            return true;
                        }
                        // https://github.com/ajv-validator/ajv/issues/652#issue-283610859
                        const multiplier = Math.pow(10, decimalPlaces(schema.multipleOf));
                        return Math.round(value * multiplier) % Math.round(schema.multipleOf * multiplier) === 0;
                    });
                }
                break;
            }
            case 'string': {
                field.parsers = [
                    (v, f) => {
                        if (types.indexOf('null') !== -1) {
                            v = isEmpty(v) ? null : v;
                        }
                        else if (f && !f.props.required) {
                            v = v === '' ? undefined : v;
                        }
                        return v;
                    },
                ];
                ['minLength', 'maxLength', 'pattern'].forEach((prop) => {
                    if (schema.hasOwnProperty(prop)) {
                        field.props[prop] = schema[prop];
                    }
                });
                break;
            }
            case 'object': {
                if (!field.fieldGroup) {
                    field.fieldGroup = [];
                }
                const { propDeps, schemaDeps } = this.resolveDependencies(schema);
                Object.keys(schema.properties || {}).forEach((property) => {
                    const isRequired = Array.isArray(schema.required) && schema.required.indexOf(property) !== -1;
                    const f = this._toFieldConfig(schema.properties[property], {
                        ...options,
                        key: property,
                        isOptional: isOptional || !isRequired,
                    });
                    field.fieldGroup.push(f);
                    if (isRequired || propDeps[property]) {
                        f.expressions = {
                            ...(f.expressions || {}),
                            'props.required': (f) => {
                                let parent = f.parent;
                                const model = f.fieldGroup && f.key != null ? parent.model : f.model;
                                while (parent.key == null && parent.parent) {
                                    parent = parent.parent;
                                }
                                const required = parent && parent.props ? parent.props.required : false;
                                if (!model && !required) {
                                    return false;
                                }
                                if (Array.isArray(schema.required) && schema.required.indexOf(property) !== -1) {
                                    return true;
                                }
                                return propDeps[property] && f.model && propDeps[property].some((k) => !isEmpty(f.model[k]));
                            },
                        };
                    }
                    if (schemaDeps[property]) {
                        const getConstValue = (s) => {
                            return s.hasOwnProperty('const') ? s.const : s.enum[0];
                        };
                        const oneOfSchema = schemaDeps[property].oneOf;
                        if (oneOfSchema &&
                            oneOfSchema.every((o) => o.properties && o.properties[property] && isConst(o.properties[property]))) {
                            oneOfSchema.forEach((oneOfSchemaItem) => {
                                const { [property]: constSchema, ...properties } = oneOfSchemaItem.properties;
                                field.fieldGroup.push({
                                    ...this._toFieldConfig({ ...oneOfSchemaItem, properties }, { ...options, shareFormControl: false, resetOnHide: true }),
                                    expressions: {
                                        hide: (f) => !f.model || getConstValue(constSchema) !== f.model[property],
                                    },
                                });
                            });
                        }
                        else {
                            field.fieldGroup.push({
                                ...this._toFieldConfig(schemaDeps[property], options),
                                expressions: {
                                    hide: (f) => !f.model || isEmpty(f.model[property]),
                                },
                            });
                        }
                    }
                });
                if (schema.oneOf) {
                    field.fieldGroup.push(this.resolveMultiSchema('oneOf', schema.oneOf, { ...options, shareFormControl: false }));
                }
                if (schema.anyOf) {
                    field.fieldGroup.push(this.resolveMultiSchema('anyOf', schema.anyOf, options));
                }
                break;
            }
            case 'array': {
                if (schema.hasOwnProperty('minItems')) {
                    field.props.minItems = schema.minItems;
                    this.addValidator(field, 'minItems', ({ value }) => {
                        return isEmpty(value) || value.length >= schema.minItems;
                    });
                    if (!isOptional && schema.minItems > 0 && field.defaultValue === undefined) {
                        field.defaultValue = Array.from(new Array(schema.minItems));
                    }
                }
                if (schema.hasOwnProperty('maxItems')) {
                    field.props.maxItems = schema.maxItems;
                    this.addValidator(field, 'maxItems', ({ value }) => {
                        return isEmpty(value) || value.length <= schema.maxItems;
                    });
                }
                if (schema.hasOwnProperty('uniqueItems')) {
                    field.props.uniqueItems = schema.uniqueItems;
                    this.addValidator(field, 'uniqueItems', ({ value }) => {
                        if (isEmpty(value) || !schema.uniqueItems) {
                            return true;
                        }
                        const uniqueItems = Array.from(new Set(value.map((v) => JSON.stringify(v, (k, o) => {
                            if (isObject(o)) {
                                return Object.keys(o)
                                    .sort()
                                    .reduce((obj, key) => {
                                    obj[key] = o[key];
                                    return obj;
                                }, {});
                            }
                            return o;
                        }))));
                        return uniqueItems.length === value.length;
                    });
                }
                // resolve items schema needed for isEnum check
                if (schema.items && !Array.isArray(schema.items)) {
                    schema.items = this.resolveSchema(schema.items, options);
                }
                // TODO: remove isEnum check once adding an option to skip extension
                if (!this.isEnum(schema)) {
                    field.fieldArray = (root) => {
                        const length = root.fieldGroup ? root.fieldGroup.length : 0;
                        const items = schema.items;
                        if (!Array.isArray(items)) {
                            if (!items) {
                                return {};
                            }
                            const isMultiSchema = items.oneOf || items.anyOf;
                            // When items is a single schema, the additionalItems keyword is meaningless, and it should not be used.
                            const f = this._toFieldConfig(items, isMultiSchema ? { ...options, key: `${length}`, isOptional: false } : { ...options, isOptional: false });
                            if (isMultiSchema && !hasKey(f)) {
                                f.key = null;
                            }
                            return f;
                        }
                        const itemSchema = items[length] ? items[length] : schema.additionalItems;
                        const f = itemSchema ? this._toFieldConfig(itemSchema, options) : {};
                        if (f.props) {
                            f.props.required = true;
                        }
                        if (items[length]) {
                            f.props.removable = false;
                        }
                        return f;
                    };
                }
                break;
            }
        }
        if (schema.hasOwnProperty('const')) {
            field.props.const = schema.const;
            this.addValidator(field, 'const', ({ value }) => value === schema.const);
            if (!field.type) {
                field.defaultValue = schema.const;
            }
        }
        if (this.isEnum(schema)) {
            const enumOptions = this.toEnumOptions(schema);
            const multiple = field.type === 'array';
            field.type = 'enum';
            field.props.multiple = multiple;
            field.props.options = enumOptions;
            const enumValues = enumOptions.map((o) => o.value);
            this.addValidator(field, 'enum', ({ value }) => {
                if (value === undefined) {
                    return true;
                }
                if (multiple) {
                    return Array.isArray(value) ? value.every((o) => enumValues.includes(o)) : false;
                }
                return enumValues.includes(value);
            });
        }
        if (schema.oneOf && !field.type) {
            delete field.key;
            field.fieldGroup = [
                this.resolveMultiSchema('oneOf', schema.oneOf, { ...options, key, shareFormControl: false }),
            ];
        }
        if (schema.anyOf && !field.type) {
            delete field.key;
            field.fieldGroup = [
                this.resolveMultiSchema('oneOf', schema.anyOf, { ...options, key, shareFormControl: false }),
            ];
        }
        // map in possible formlyConfig options from the widget property
        if (schema.widget?.formlyConfig) {
            field = this.mergeFields(field, schema.widget.formlyConfig);
        }
        field.templateOptions = field.props;
        // if there is a map function passed in, use it to allow the user to
        // further customize how fields are being mapped
        return options.map ? options.map(field, schema) : field;
    }
    resolveSchema(schema, options) {
        if (schema && schema.$ref) {
            schema = this.resolveDefinition(schema, options);
        }
        if (schema && schema.allOf) {
            schema = this.resolveAllOf(schema, options);
        }
        return schema;
    }
    resolveAllOf({ allOf, ...baseSchema }, options) {
        if (!allOf.length) {
            throw Error(`allOf array can not be empty ${allOf}.`);
        }
        return allOf.reduce((base, schema) => {
            schema = this.resolveSchema(schema, options);
            if (base.required && schema.required) {
                base.required = [...base.required, ...schema.required];
            }
            if (schema.uniqueItems) {
                base.uniqueItems = schema.uniqueItems;
            }
            // resolve to min value
            ['maxLength', 'maximum', 'exclusiveMaximum', 'maxItems', 'maxProperties'].forEach((prop) => {
                if (!isEmpty(base[prop]) && !isEmpty(schema[prop])) {
                    base[prop] = base[prop] < schema[prop] ? base[prop] : schema[prop];
                }
            });
            // resolve to max value
            ['minLength', 'minimum', 'exclusiveMinimum', 'minItems', 'minProperties'].forEach((prop) => {
                if (!isEmpty(base[prop]) && !isEmpty(schema[prop])) {
                    base[prop] = base[prop] > schema[prop] ? base[prop] : schema[prop];
                }
            });
            return reverseDeepMerge(base, schema);
        }, baseSchema);
    }
    resolveMultiSchema(mode, schemas, options) {
        return {
            type: 'multischema',
            fieldGroup: [
                {
                    type: 'enum',
                    defaultValue: -1,
                    props: {
                        multiple: mode === 'anyOf',
                        options: schemas.map((s, i) => ({ label: s.title, value: i, disabled: s.readOnly })),
                    },
                    hooks: {
                        onInit: (f) => f.formControl.valueChanges.pipe(tap(() => f.options.detectChanges(f.parent))),
                    },
                },
                {
                    fieldGroup: schemas.map((s, i) => ({
                        ...this._toFieldConfig(s, { ...options, resetOnHide: true }),
                        expressions: {
                            hide: (f, forceUpdate) => {
                                const control = f.parent.parent.fieldGroup[0].formControl;
                                if (control.value === -1 || forceUpdate) {
                                    let value = f.parent.fieldGroup
                                        .map((f, i) => [f, i, this.isFieldValid(f, i, schemas, options)])
                                        .sort(([f1, , f1Valid], [f2, , f2Valid]) => {
                                        if (f1Valid !== f2Valid) {
                                            return f2Valid ? 1 : -1;
                                        }
                                        const matchedFields1 = totalMatchedFields(f1);
                                        const matchedFields2 = totalMatchedFields(f2);
                                        if (matchedFields1 === matchedFields2) {
                                            if (f1.props.disabled === f2.props.disabled) {
                                                return 0;
                                            }
                                            return matchedFields2 > matchedFields1 ? 1 : -1;
                                        }
                                        return matchedFields2 > matchedFields1 ? 1 : -1;
                                    })
                                        .map(([, i]) => i);
                                    if (mode === 'anyOf') {
                                        const definedValue = value.filter((i) => totalMatchedFields(f.parent.fieldGroup[i]));
                                        value = definedValue.length > 0 ? definedValue : [value[0] || 0];
                                    }
                                    value = value.length > 0 ? value : [0];
                                    control.setValue(mode === 'anyOf' ? value : value[0]);
                                }
                                return Array.isArray(control.value) ? control.value.indexOf(i) === -1 : control.value !== i;
                            },
                        },
                    })),
                },
            ],
        };
    }
    resolveDefinition(schema, options) {
        const [uri, pointer] = schema.$ref.split('#/');
        if (uri) {
            throw Error(`Remote schemas for ${schema.$ref} not supported yet.`);
        }
        const definition = !pointer
            ? null
            : pointer
                .split('/')
                .reduce((def, path) => (def?.hasOwnProperty(path) ? def[path] : null), options.schema);
        if (!definition) {
            throw Error(`Cannot find a definition for ${schema.$ref}.`);
        }
        if (definition.$ref) {
            return this.resolveDefinition(definition, options);
        }
        return {
            ...definition,
            ...['title', 'description', 'default', 'widget'].reduce((annotation, p) => {
                if (schema.hasOwnProperty(p)) {
                    annotation[p] = schema[p];
                }
                return annotation;
            }, {}),
        };
    }
    resolveDependencies(schema) {
        const propDeps = {};
        const schemaDeps = {};
        Object.keys(schema.dependencies || {}).forEach((prop) => {
            const dependency = schema.dependencies[prop];
            if (Array.isArray(dependency)) {
                // Property dependencies
                dependency.forEach((dep) => {
                    if (!propDeps[dep]) {
                        propDeps[dep] = [prop];
                    }
                    else {
                        propDeps[dep].push(prop);
                    }
                });
            }
            else {
                // schema dependencies
                schemaDeps[prop] = dependency;
            }
        });
        return { propDeps, schemaDeps };
    }
    guessSchemaType(schema) {
        const type = schema?.type;
        if (!type && schema?.properties) {
            return ['object'];
        }
        if (Array.isArray(type)) {
            if (type.length === 1) {
                return type;
            }
            if (type.length === 2 && type.indexOf('null') !== -1) {
                return type.sort((t1) => (t1 == 'null' ? 1 : -1));
            }
            return type;
        }
        return type ? [type] : [];
    }
    addValidator(field, name, validator) {
        field.validators = field.validators || {};
        field.validators[name] = validator;
    }
    isEnum(schema) {
        return (!!schema.enum ||
            (schema.anyOf && schema.anyOf.every(isConst)) ||
            (schema.oneOf && schema.oneOf.every(isConst)) ||
            (schema.uniqueItems && schema.items && !Array.isArray(schema.items) && this.isEnum(schema.items)));
    }
    toEnumOptions(schema) {
        if (schema.enum) {
            return schema.enum.map((value) => ({ value, label: value }));
        }
        const toEnum = (s) => {
            const value = s.hasOwnProperty('const') ? s.const : s.enum[0];
            const option = { value, label: s.title || value };
            if (s.readOnly) {
                option.disabled = true;
            }
            return option;
        };
        if (schema.anyOf) {
            return schema.anyOf.map(toEnum);
        }
        if (schema.oneOf) {
            return schema.oneOf.map(toEnum);
        }
        return this.toEnumOptions(schema.items);
    }
    isFieldValid(root, i, schemas, options) {
        const schema = schemas[i];
        if (!schema._field) {
            Object.defineProperty(schema, '_field', { enumerable: false, writable: true, configurable: true });
        }
        let field = schema._field;
        let model = root.model ? root.model : root.fieldArray ? [] : {};
        if (root.model && hasKey(root)) {
            model = { [Array.isArray(root.key) ? root.key.join('.') : root.key]: getFieldValue(root) };
        }
        model = clone(model);
        if (!field) {
            field = schema._field = root.options.build({
                form: Array.isArray(model) ? new UntypedFormArray([]) : new UntypedFormGroup({}),
                fieldGroup: [
                    this._toFieldConfig(schema, {
                        ...options,
                        resetOnHide: true,
                        ignoreDefault: true,
                        map: null,
                    }),
                ],
                model,
                options: {},
            });
        }
        else {
            field.model = model;
            root.options.build(field);
        }
        return field.form.valid;
    }
    mergeFields(f1, f2) {
        for (const prop in f2) {
            const f1Prop = prop === 'templateOptions' ? 'props' : prop;
            if (isObject(f1[f1Prop]) && isObject(f2[prop])) {
                f1[f1Prop] = this.mergeFields(f1[f1Prop], f2[prop]);
            }
            else if (f2[prop] != null) {
                f1[f1Prop] = f2[prop];
            }
        }
        return f1;
    }
    static { this.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "18.2.13", ngImport: i0, type: FormlyJsonschema, deps: [], target: i0.ɵɵFactoryTarget.Injectable }); }
    static { this.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "18.2.13", ngImport: i0, type: FormlyJsonschema, providedIn: 'root' }); }
}
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "18.2.13", ngImport: i0, type: FormlyJsonschema, decorators: [{
            type: Injectable,
            args: [{ providedIn: 'root' }]
        }] });
//# sourceMappingURL=data:application/json;base64,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