import { TestContext } from "../../common/testContext";
import { LoggerMessageComponent, LoggerProgress, ProgressOptions, LoggerDelegate, LogLevel, LabeledLogger } from "../../types/logger";
/**
 * Default logger delegate implementation using Winston
 * Provides clean, formatted output to the console
 */
export declare class DefaultLoggerDelegate implements LoggerDelegate {
    private readonly logger;
    constructor();
    /**
     * Log a message at the specified level
     * @param level The log level (info, warn, error, debug)
     * @param message The formatted message to log
     */
    log(level: LogLevel, message: string): void;
}
/**
 * Core Logger implementation with singleton pattern
 */
declare class Logger {
    private static instance;
    private delegate;
    private projectRootCache;
    private testContext;
    private readonly logLevels;
    private readonly colorMap;
    private readonly statusColors;
    /**
     * Private constructor to enforce singleton pattern
     */
    private constructor();
    /**
     * Get the singleton instance of the Logger
     * @returns The Logger instance
     */
    static getInstance(): Logger;
    /**
     * Set a custom logger delegate
     * @param delegate The delegate implementation to use for logging
     */
    setDelegate(delegate: LoggerDelegate): void;
    /**
     * Get the current logger delegate
     * @returns Current delegate implementation
     */
    getDelegate(): LoggerDelegate;
    /**
     * Set the test context for this logger instance
     * @param testContext The test context to use for logging
     */
    setTestContext(testContext: TestContext): void;
    /**
     * Convert message components to a colorized string
     * Applies colors and formatting to log components
     * @param components Array of message components to colorize
     * @returns Formatted string with ANSI color codes
     */
    private colorizeMessage;
    /**
     * Format a message for log file output (plain text without colors)
     * @param level Log level
     * @param components Message components
     * @returns Plain text formatted message for file output
     */
    /**
     * Core logging method that all other log methods use
     * Creates a formatted log component from a text string
     * @param text The message text
     * @param level Log level to determine text color
     * @param isBold Whether the text should be bold
     * @returns Formatted log component
     */
    private createComponent;
    /**
     * Creates components for the test file label if in a test environment
     * @returns Array of components for the test file label or empty array if not in a test
     */
    private createTestFileComponents;
    /**
     * Converts string messages to properly formatted log components
     * @param components Raw message components (strings or formatted components)
     * @param level Log level to apply styling
     * @returns Array of properly formatted log components
     */
    private normalizeComponents;
    /**
     * Formats and sends log message to the delegate
     * This is the core logging implementation used by all logging methods
     * @param level Log level
     * @param components Message components
     * @param prefix Optional prefix to prepend to the message
     */
    private formatAndSend;
    private log;
    /**
     * Log an informational message
     * @param components Message components
     */
    info(...components: LoggerMessageComponent[]): void;
    warn(...components: LoggerMessageComponent[]): void;
    error(...components: LoggerMessageComponent[]): void;
    debug(...components: LoggerMessageComponent[]): void;
    labeled(label: string): LabeledLogger;
    /**
     * Log a message with a formatted label prefix
     * @param level Log level
     * @param label The label to display
     * @param components Message components
     */
    private logWithLabel;
    /**
     * Create a styled background label
     * @param text The label text
     * @param bgColor Background color for the label
     * @returns Formatted label string with ANSI color codes
     */
    private createLabel;
    /**
     * Start a progress tracking operation
     * @param options Progress configuration options
     * @param components Initial message components
     * @returns Progress control object
     */
    startProgress(options: ProgressOptions, ...components: LoggerMessageComponent[]): LoggerProgress;
    /**
     * Log a progress message with a styled label
     * @param labelText Text for the progress label
     * @param labelColor Background color for the label
     * @param level Log level
     * @param components Message components
     */
    private logProgress;
    /**
     * Map operation result type to appropriate log level
     * @param result The operation result type
     * @returns The corresponding log level
     */
    private getLogMethodForResult;
    logCodeExec(code: string): void;
}
declare const logger: Logger;
export default logger;
