"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.CacheHandler = void 0;
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const os_1 = __importDefault(require("os"));
const logger_1 = __importDefault(require("../../common/logger"));
const CodeEvaluator_1 = require("../../common/CodeEvaluator");
/**
 * CacheHandler provides a unified caching solution for both StepPerformer and AutoPerformer.
 * It works with the SnapshotComparator to compare snapshots and find matching cache entries.
 */
class CacheHandler {
    static CACHE_DIRECTORY = "__pilot_cache__";
    static DEFAULT_CACHE_FILENAME = "global.json";
    static APP_NAME = "PilotAutomation";
    cache = new Map();
    temporaryCache = new Map();
    overrideCacheFilePath;
    cacheFilePath;
    cacheOptions;
    testContext;
    snapshotComparator;
    codeEvaluator;
    resolvedCacheFilePath;
    /**
     * Creates a new CacheHandler instance
     * @param snapshotComparator The snapshot comparator to use for hash generation and comparison
     * @param testContext Test context containing the current test file path function
     * @param cacheOptions Cache configuration options
     * @param cacheFilePath Optional explicit cache file path override
     */
    constructor(snapshotComparator, testContext, cacheOptions = {}, cacheFilePath) {
        this.overrideCacheFilePath = cacheFilePath;
        this.cacheOptions = this.createCacheOptionsWithDefaults(cacheOptions);
        this.snapshotComparator = snapshotComparator;
        this.testContext = testContext;
        this.cacheFilePath = this.determineCurrentCacheFilePath();
        this.codeEvaluator = new CodeEvaluator_1.CodeEvaluator();
    }
    createCacheOptionsWithDefaults(cacheOptions) {
        return {
            shouldUseCache: cacheOptions.shouldUseCache ?? true,
            shouldOverrideCache: cacheOptions.shouldOverrideCache ?? false,
        };
    }
    determineCurrentCacheFilePath() {
        return this.overrideCacheFilePath || this.getCacheFilePath();
    }
    /**
     * Generate hashes for a snapshot using all registered algorithms
     * @param screenCapture The screen capture result
     * @returns Object with hash values from each registered algorithm
     */
    async generateHashes(screenCapture) {
        return await this.snapshotComparator.generateHashes(screenCapture);
    }
    loadCacheFromFile() {
        this.cacheFilePath = this.determineCurrentCacheFilePath();
        try {
            const resolvedPath = this.cacheFilePath;
            if (fs_1.default.existsSync(resolvedPath)) {
                const data = fs_1.default.readFileSync(resolvedPath, "utf-8");
                const json = JSON.parse(data);
                this.cache = new Map(Object.entries(json));
            }
            else {
                this.cache.clear();
            }
        }
        catch (error) {
            logger_1.default.warn("Error loading cache from file:", {
                message: String(error),
                color: "yellow",
            });
            this.cache.clear();
        }
    }
    saveCacheToFile() {
        try {
            const resolvedPath = this.cacheFilePath;
            const dirPath = path_1.default.dirname(resolvedPath);
            if (!fs_1.default.existsSync(dirPath)) {
                fs_1.default.mkdirSync(dirPath, { recursive: true });
            }
            const json = Object.fromEntries(this.cache);
            fs_1.default.writeFileSync(resolvedPath, JSON.stringify(json, null, 2), {
                flag: "w+",
            });
            logger_1.default.info("Pilot cache saved successfully");
        }
        catch (error) {
            logger_1.default.error("Error saving cache to file:", {
                message: String(error),
                color: "red",
            });
        }
    }
    /**
     * Get cached values by key from the persistent cache
     * @param cacheKey The cache key string
     * @returns Array of cache values if found, undefined otherwise
     */
    getFromPersistentCache(cacheKey) {
        if (this.shouldOverrideCache()) {
            logger_1.default.info("Cache disabled, generating new response");
            return undefined;
        }
        return this.cache.get(cacheKey);
    }
    /**
     * Add value to temporary cache
     * @param cacheKey The cache key string
     * @param value The value to cache
     * @param snapshotHashes Hash values for the current snapshot
     */
    addToTemporaryCacheSnapshotBased(cacheKey, value, snapshotHashes) {
        logger_1.default.info("Saving result to cache for future use");
        const cacheValue = {
            value,
            snapshotHashes: snapshotHashes || {},
            creationTime: Date.now(),
        };
        const existingValues = this.temporaryCache.get(cacheKey) || [];
        this.temporaryCache.set(cacheKey, [
            ...existingValues,
            cacheValue,
        ]);
    }
    /**
     * Add value to temporary cache
     * @param cacheKey The cache key string
     * @param value The value to cache
     * @param validationMatcher a code line that validate the existence of the step's relevant element
     */
    addToTemporaryCacheValidationMatcherBased(cacheKey, value, validationMatcher) {
        logger_1.default.labeled("CACHE").info("Saving response to cache");
        const cacheValue = {
            value,
            validationMatcher: validationMatcher,
            creationTime: Date.now(),
        };
        const existingValues = this.temporaryCache.get(cacheKey) || [];
        this.temporaryCache.set(cacheKey, [
            ...existingValues,
            cacheValue,
        ]);
    }
    /**
     * Persist temporary cache to permanent cache and save to file
     */
    flushTemporaryCache() {
        if (this.temporaryCache.size === 0) {
            return;
        }
        this.temporaryCache.forEach((values, key) => {
            const existingValues = this.cache.get(key) || [];
            this.cache.set(key, [...existingValues, ...values]);
        });
        this.saveCacheToFile();
        this.clearTemporaryCache();
    }
    /**
     * Clear the temporary cache without persisting it
     */
    clearTemporaryCache() {
        this.temporaryCache.clear();
    }
    /**
     * Find matching cache entry by comparing snapshot hashes
     * @param cacheValues Array of cache values to search
     * @param currentHashes Current snapshot hashes (complete with all algorithms)
     * @returns Matching cache value if found, undefined otherwise
     */
    findMatchingCacheEntrySnapshotBased(cacheValues, currentHashes) {
        if (!cacheValues?.length || !currentHashes) {
            return undefined;
        }
        return cacheValues.find((entry) => {
            return this.snapshotComparator.compareSnapshot(currentHashes, entry.snapshotHashes);
        });
    }
    /**
     * Searches for a matching cache entry by evaluating its validation matcher code.
     * If the matcher evaluates to true, the corresponding cache entry is returned.
     * Also determines whether additional code should be run based on a comparison between the original code and the matcher.
     * @param cacheValues Array of cache values to search
     * @param context The context
     * @param sharedContext The shared context to use for evaluating the validation matcher
     * @returns Matching cache value if found, undefined otherwise
     */
    async findMatchingCacheEntryValidationMatcherBased(cacheValues, context, sharedContext) {
        for (const entry of cacheValues) {
            if (!entry.validationMatcher) {
                return entry;
            }
            const matcher = entry.validationMatcher;
            if (!matcher || typeof matcher !== "string")
                continue;
            try {
                const result = await this.codeEvaluator.evaluate(matcher, context, sharedContext);
                if (result) {
                    return entry;
                }
            }
            catch (error) {
                logger_1.default.debug("Error evaluating matcher:", matcher);
            }
        }
        return undefined;
    }
    /**
     * Generate a cache key from serializable data
     * @param keyData The data to use as a cache key (must be JSON serializable)
     * @returns Cache key string or undefined if cache is disabled
     * @example
     * // Generate a key for step performer
     * const key = cacheHandler.generateCacheKey({ step, previousSteps });
     *
     * // Generate a key for auto performer
     * const key = cacheHandler.generateCacheKey({ goal, previousSteps });
     */
    generateCacheKey(keyData) {
        if (!this.isCacheInUse()) {
            return undefined;
        }
        return JSON.stringify(keyData);
    }
    shouldOverrideCache() {
        return this.cacheOptions?.shouldOverrideCache;
    }
    isCacheInUse() {
        return this.cacheOptions?.shouldUseCache !== false;
    }
    /**
     * Gets the OS-specific user data directory path for the application
     * @returns The appropriate application data directory for the current OS
     */
    getUserDataDir() {
        const platform = process.platform;
        const appName = CacheHandler.APP_NAME;
        switch (platform) {
            case "darwin":
                return path_1.default.join(os_1.default.homedir(), "Library", "Application Support", appName);
            case "win32":
                return process.env.APPDATA
                    ? path_1.default.join(process.env.APPDATA, appName)
                    : path_1.default.join(os_1.default.homedir(), "AppData", "Roaming", appName);
            case "linux":
                return path_1.default.join(os_1.default.homedir(), ".config", appName);
            default:
                return path_1.default.join(os_1.default.homedir(), ".local", "share", appName);
        }
    }
    /**
     * Determines the appropriate cache file path based on the caller path
     * @returns The resolved cache file path
     */
    getCacheFilePath() {
        const callerPath = this.testContext.getCurrentTestFilePath();
        if (callerPath) {
            this.resolvedCacheFilePath = this.getCallerCacheFilePath(callerPath);
        }
        else if (this.resolvedCacheFilePath) {
            return this.resolvedCacheFilePath;
        }
        else {
            this.resolvedCacheFilePath = this.getDefaultCacheFilePath();
        }
        return this.resolvedCacheFilePath;
    }
    /**
     * Gets the cache file path for a specific caller file (typically a Jest test)
     * @param callerPath The path of the calling file
     * @returns The resolved cache file path specific to the caller
     */
    getCallerCacheFilePath(callerPath) {
        const testDir = path_1.default.dirname(callerPath);
        const testFilename = path_1.default.basename(callerPath, path_1.default.extname(callerPath));
        return path_1.default.join(testDir, CacheHandler.CACHE_DIRECTORY, `${testFilename}.json`);
    }
    /**
     * Gets the default global cache file path when no caller path is available
     * @returns The resolved default cache file path
     */
    getDefaultCacheFilePath() {
        return path_1.default.join(this.getUserDataDir(), CacheHandler.CACHE_DIRECTORY, CacheHandler.DEFAULT_CACHE_FILENAME);
    }
}
exports.CacheHandler = CacheHandler;
