import { Config, TestingFrameworkAPICatalogCategory, AutoReviewSectionConfig } from "./types";
import { AutoReport } from "./types/auto";
/**
 * The main Pilot class that provides AI-assisted testing capabilities for a given underlying testing framework.
 * Enables writing tests in natural language that are translated into precise testing actions.
 *
 * Usage:
 * ```typescript
 * const pilot = new Pilot({
 *   frameworkDriver: new PuppeteerFrameworkDriver(),
 *   promptHandler: new OpenAIHandler({ apiKey: process.env.OPENAI_API_KEY })
 * });
 *
 * pilot.start();
 * await pilot.perform('Navigate to the login page');
 * await pilot.perform('Enter "user@example.com" in the email field');
 * pilot.end();
 * ```
 */
export declare class Pilot {
    private readonly snapshotManager;
    private previousSteps;
    private stepPerformerPromptCreator;
    private stepPerformer;
    private cacheHandler;
    private running;
    private autoPerformer;
    private screenCapturer;
    private snapshotComparator;
    private testContext;
    constructor(config: Config);
    /**
     * Checks if Pilot is currently running a test flow.
     * @returns true if running, false otherwise
     */
    private assertIsRunning;
    /**
     * Starts a new test flow session.
     * Must be called before any test operations to ensure a clean state, as Pilot uses operation history for context.
     */
    start(): void;
    /**
     * Ends the current test flow session and handles cache management.
     * @param shouldSaveInCache - If true, the current test flow will be saved in cache (default: true)
     */
    end(shouldSaveInCache?: boolean): void;
    /**
     * Extends the testing framework's API capabilities.
     * @param categories - Additional API categories to add
     * @param context - Testing framework variables to expose (optional)
     * @example
     * pilot.extendAPICatalog([
     *   {
     *     title: 'Custom Actions',
     *     items: [
     *       {
     *         signature: 'customAction(param: string)',
     *         description: 'Performs a custom action',
     *         example: 'await customAction("param")',
     *         guidelines: ['Use this action for specific test scenarios']
     *       }
     *     ]
     *   }
     * ], { customAction });
     */
    extendAPICatalog(categories: TestingFrameworkAPICatalogCategory[], context?: any): void;
    /**
     * Performs one or more test steps using the provided intents.
     * @param steps The intents describing the test steps to perform.
     * @returns The result of the last executed step.
     */
    perform(...steps: string[]): Promise<any>;
    private performStep;
    private didPerformStep;
    /**
     * Performs an entire test flow using the provided goal.
     * @param goal A string which describes the flow should be executed.
     * @param reviewConfigs Optional review types to include in the autopilot report.
     * @returns pilot report with info about the actions thoughts etc ...
     */
    autopilot(goal: string, reviewConfigs?: AutoReviewSectionConfig[]): Promise<AutoReport>;
    /**
     * Loads the cache from the cache file.
     */
    private loadCache;
}
