/// <reference types="./create-trigger-builder.d.ts" />
import { CreateTriggerNode } from '../operation-node/create-trigger-node.js';
import { QueryNode } from '../operation-node/query-node.js';
import { parseValueBinaryOperationOrExpression, } from '../parser/binary-operation-parser.js';
import { parseOrderedColumnName, parseReferenceExpressionOrList, } from '../parser/reference-parser.js';
import { ImmediateValueTransformer } from '../plugin/immediate-value/immediate-value-transformer.js';
import { freeze } from '../util/object-utils.js';
import { preventAwait } from '../util/prevent-await.js';
import { IdentifierNode } from '../operation-node/identifier-node.js';
import { TriggerOrderNode } from '../operation-node/trigger-order-node.js';
import { TriggerEventNode, } from '../operation-node/trigger-event-node.js';
import { FunctionNode } from '../operation-node/function-node.js';
import { TriggerQueryCreator } from '../trigger-query-creator.js';
import { TableNode } from '../operation-node/table-node.js';
/**
 * This builder can be used to create a `create table` query.
 */
export class CreateTriggerBuilder {
    #props;
    constructor(props) {
        this.#props = freeze(props);
    }
    before() {
        return new CreateTriggerBuilder({
            ...this.#props,
            node: CreateTriggerNode.cloneWith(this.#props.node, {
                time: 'before',
            }),
        });
    }
    after() {
        return new CreateTriggerBuilder({
            ...this.#props,
            node: CreateTriggerNode.cloneWith(this.#props.node, {
                time: 'after',
            }),
        });
    }
    insteadOf() {
        return new CreateTriggerBuilder({
            ...this.#props,
            node: CreateTriggerNode.cloneWith(this.#props.node, {
                time: 'instead of',
            }),
        });
    }
    addEvent(event, columns) {
        return new CreateTriggerBuilder({
            ...this.#props,
            node: CreateTriggerNode.cloneWithEvent(this.#props.node, TriggerEventNode.create(event, columns?.map(parseOrderedColumnName))),
        });
    }
    forEachRow() {
        return new CreateTriggerBuilder({
            ...this.#props,
            node: CreateTriggerNode.cloneWith(this.#props.node, {
                forEach: `row`,
            }),
        });
    }
    forEachStatement() {
        return new CreateTriggerBuilder({
            ...this.#props,
            node: CreateTriggerNode.cloneWith(this.#props.node, {
                forEach: `statement`,
            }),
        });
    }
    follows(otherTriggerName) {
        return new CreateTriggerBuilder({
            ...this.#props,
            node: CreateTriggerNode.cloneWith(this.#props.node, {
                order: TriggerOrderNode.create('follows', IdentifierNode.create(otherTriggerName)),
            }),
        });
    }
    precedes(otherTriggerName) {
        return new CreateTriggerBuilder({
            ...this.#props,
            node: CreateTriggerNode.cloneWith(this.#props.node, {
                order: TriggerOrderNode.create('precedes', IdentifierNode.create(otherTriggerName)),
            }),
        });
    }
    /**
     * Specifies the table for the trigger.
     */
    onTable(table, schema) {
        return new CreateTriggerBuilder({
            ...this.#props,
            node: CreateTriggerNode.cloneWith(this.#props.node, {
                table: schema
                    ? TableNode.createWithSchema(schema, table)
                    : TableNode.create(table),
            }),
        });
    }
    /**
     * Adds the "temporary" modifier.
     *
     * Use this to create a temporary trigger.
     */
    temporary() {
        return new CreateTriggerBuilder({
            ...this.#props,
            node: CreateTriggerNode.cloneWith(this.#props.node, {
                temporary: true,
            }),
        });
    }
    /**
     * Adds the "if not exists" modifier.
     *
     * If the trigger already exists, no error is thrown if this method has been called.
     */
    ifNotExists() {
        return new CreateTriggerBuilder({
            ...this.#props,
            node: CreateTriggerNode.cloneWith(this.#props.node, {
                ifNotExists: true,
            }),
        });
    }
    /**
     * Only supported on PostgreSQL
     */
    orReplace() {
        return new CreateTriggerBuilder({
            ...this.#props,
            node: CreateTriggerNode.cloneWith(this.#props.node, {
                orReplace: true,
            }),
        });
    }
    /**
     * Adds a query to the trigger.
     */
    addQuery(build) {
        const node = build(new TriggerQueryCreator({ executor: this.#props.executor })).toOperationNode();
        if (!QueryNode.is(node))
            throw new Error('Must be a query node.');
        return new CreateTriggerBuilder({
            ...this.#props,
            node: CreateTriggerNode.cloneWithQuery(this.#props.node, node),
        });
    }
    function(name, args) {
        return new CreateTriggerBuilder({
            ...this.#props,
            node: CreateTriggerNode.cloneWith(this.#props.node, {
                function: FunctionNode.create(name, parseReferenceExpressionOrList(args)),
            }),
        });
    }
    when(...args) {
        const transformer = new ImmediateValueTransformer();
        return new CreateTriggerBuilder({
            ...this.#props,
            node: CreateTriggerNode.cloneWith(this.#props.node, {
                when: transformer.transformNode(parseValueBinaryOperationOrExpression(args)),
            }),
        });
    }
    $call(func) {
        return func(this);
    }
    toOperationNode() {
        return this.#props.executor.transformQuery(this.#props.node, this.#props.queryId);
    }
    compile() {
        return this.#props.executor.compileQuery(this.toOperationNode(), this.#props.queryId);
    }
    async execute() {
        await this.#props.executor.executeQuery(this.compile(), this.#props.queryId);
    }
}
preventAwait(CreateTriggerBuilder, "don't await CreateTriggerBuilder instances directly. To execute the query you need to call `execute`");
