"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CreateTriggerBuilder = void 0;
const create_trigger_node_js_1 = require("../operation-node/create-trigger-node.js");
const query_node_js_1 = require("../operation-node/query-node.js");
const binary_operation_parser_js_1 = require("../parser/binary-operation-parser.js");
const reference_parser_js_1 = require("../parser/reference-parser.js");
const immediate_value_transformer_js_1 = require("../plugin/immediate-value/immediate-value-transformer.js");
const object_utils_js_1 = require("../util/object-utils.js");
const prevent_await_js_1 = require("../util/prevent-await.js");
const identifier_node_js_1 = require("../operation-node/identifier-node.js");
const trigger_order_node_js_1 = require("../operation-node/trigger-order-node.js");
const trigger_event_node_js_1 = require("../operation-node/trigger-event-node.js");
const function_node_js_1 = require("../operation-node/function-node.js");
const trigger_query_creator_js_1 = require("../trigger-query-creator.js");
const table_node_js_1 = require("../operation-node/table-node.js");
/**
 * This builder can be used to create a `create table` query.
 */
class CreateTriggerBuilder {
    #props;
    constructor(props) {
        this.#props = (0, object_utils_js_1.freeze)(props);
    }
    before() {
        return new CreateTriggerBuilder({
            ...this.#props,
            node: create_trigger_node_js_1.CreateTriggerNode.cloneWith(this.#props.node, {
                time: 'before',
            }),
        });
    }
    after() {
        return new CreateTriggerBuilder({
            ...this.#props,
            node: create_trigger_node_js_1.CreateTriggerNode.cloneWith(this.#props.node, {
                time: 'after',
            }),
        });
    }
    insteadOf() {
        return new CreateTriggerBuilder({
            ...this.#props,
            node: create_trigger_node_js_1.CreateTriggerNode.cloneWith(this.#props.node, {
                time: 'instead of',
            }),
        });
    }
    addEvent(event, columns) {
        return new CreateTriggerBuilder({
            ...this.#props,
            node: create_trigger_node_js_1.CreateTriggerNode.cloneWithEvent(this.#props.node, trigger_event_node_js_1.TriggerEventNode.create(event, columns?.map(reference_parser_js_1.parseOrderedColumnName))),
        });
    }
    forEachRow() {
        return new CreateTriggerBuilder({
            ...this.#props,
            node: create_trigger_node_js_1.CreateTriggerNode.cloneWith(this.#props.node, {
                forEach: `row`,
            }),
        });
    }
    forEachStatement() {
        return new CreateTriggerBuilder({
            ...this.#props,
            node: create_trigger_node_js_1.CreateTriggerNode.cloneWith(this.#props.node, {
                forEach: `statement`,
            }),
        });
    }
    follows(otherTriggerName) {
        return new CreateTriggerBuilder({
            ...this.#props,
            node: create_trigger_node_js_1.CreateTriggerNode.cloneWith(this.#props.node, {
                order: trigger_order_node_js_1.TriggerOrderNode.create('follows', identifier_node_js_1.IdentifierNode.create(otherTriggerName)),
            }),
        });
    }
    precedes(otherTriggerName) {
        return new CreateTriggerBuilder({
            ...this.#props,
            node: create_trigger_node_js_1.CreateTriggerNode.cloneWith(this.#props.node, {
                order: trigger_order_node_js_1.TriggerOrderNode.create('precedes', identifier_node_js_1.IdentifierNode.create(otherTriggerName)),
            }),
        });
    }
    /**
     * Specifies the table for the trigger.
     */
    onTable(table, schema) {
        return new CreateTriggerBuilder({
            ...this.#props,
            node: create_trigger_node_js_1.CreateTriggerNode.cloneWith(this.#props.node, {
                table: schema
                    ? table_node_js_1.TableNode.createWithSchema(schema, table)
                    : table_node_js_1.TableNode.create(table),
            }),
        });
    }
    /**
     * Adds the "temporary" modifier.
     *
     * Use this to create a temporary trigger.
     */
    temporary() {
        return new CreateTriggerBuilder({
            ...this.#props,
            node: create_trigger_node_js_1.CreateTriggerNode.cloneWith(this.#props.node, {
                temporary: true,
            }),
        });
    }
    /**
     * Adds the "if not exists" modifier.
     *
     * If the trigger already exists, no error is thrown if this method has been called.
     */
    ifNotExists() {
        return new CreateTriggerBuilder({
            ...this.#props,
            node: create_trigger_node_js_1.CreateTriggerNode.cloneWith(this.#props.node, {
                ifNotExists: true,
            }),
        });
    }
    /**
     * Only supported on PostgreSQL
     */
    orReplace() {
        return new CreateTriggerBuilder({
            ...this.#props,
            node: create_trigger_node_js_1.CreateTriggerNode.cloneWith(this.#props.node, {
                orReplace: true,
            }),
        });
    }
    /**
     * Adds a query to the trigger.
     */
    addQuery(build) {
        const node = build(new trigger_query_creator_js_1.TriggerQueryCreator({ executor: this.#props.executor })).toOperationNode();
        if (!query_node_js_1.QueryNode.is(node))
            throw new Error('Must be a query node.');
        return new CreateTriggerBuilder({
            ...this.#props,
            node: create_trigger_node_js_1.CreateTriggerNode.cloneWithQuery(this.#props.node, node),
        });
    }
    function(name, args) {
        return new CreateTriggerBuilder({
            ...this.#props,
            node: create_trigger_node_js_1.CreateTriggerNode.cloneWith(this.#props.node, {
                function: function_node_js_1.FunctionNode.create(name, (0, reference_parser_js_1.parseReferenceExpressionOrList)(args)),
            }),
        });
    }
    when(...args) {
        const transformer = new immediate_value_transformer_js_1.ImmediateValueTransformer();
        return new CreateTriggerBuilder({
            ...this.#props,
            node: create_trigger_node_js_1.CreateTriggerNode.cloneWith(this.#props.node, {
                when: transformer.transformNode((0, binary_operation_parser_js_1.parseValueBinaryOperationOrExpression)(args)),
            }),
        });
    }
    $call(func) {
        return func(this);
    }
    toOperationNode() {
        return this.#props.executor.transformQuery(this.#props.node, this.#props.queryId);
    }
    compile() {
        return this.#props.executor.compileQuery(this.toOperationNode(), this.#props.queryId);
    }
    async execute() {
        await this.#props.executor.executeQuery(this.compile(), this.#props.queryId);
    }
}
exports.CreateTriggerBuilder = CreateTriggerBuilder;
(0, prevent_await_js_1.preventAwait)(CreateTriggerBuilder, "don't await CreateTriggerBuilder instances directly. To execute the query you need to call `execute`");
