import type { Compiler as RspackCompiler } from '@rspack/core';
import type { Compiler as WebpackCompiler } from 'webpack';
import type { Rule } from '../../types.js';
/**
 * {@link HermesBytecodePlugin} configuration options.
 */
export interface HermesBytecodePluginConfig {
    /**
     * Whether the plugin is enabled.
     *
     * Since hermes compilation of chunks is not necessary for every build, this
     * option allows one to enable/disable the plugin. Normally, you would only
     * enable this plugin for production builds.
     */
    enabled: boolean;
    /** Matching files will be converted to Hermes bytecode. */
    test: Rule | Rule[];
    /** Include matching files in conversion to Hermes bytecode. */
    include?: Rule | Rule[];
    /** Exclude matching files from conversion to Hermes bytecode. */
    exclude?: Rule | Rule[];
    /** Path to the Hermes compiler binary. */
    hermesCLIPath?: string;
    /** Path to React-Native package inside node_modules */
    reactNativePath?: string;
    /** Force enable `compareBeforeEmit` webpack output option which this plugin disables by default. */
    compareBeforeEmit?: boolean;
}
/**
 * Enable Hermes bytecode compilation for the given chunks.
 * This plugin is intended to be used with the `webpack-bundle` command.
 * It will transform the bundle into Hermes bytecode and replace the original bundle with the bytecode.
 * It will also compose the source maps generated by webpack and Hermes.
 *
 * Note: This plugin should only be used for production builds.
 * It is not possible to use this plugin for development builds.
 *
 * Note: You should exclude `index.bundle` from being transformed.
 * The `index.bundle` file is transformed by `react-native` after enabling Hermes in your project.
 *
 * @example ```js
 * // webpack.config.mjs
 * import * as Repack from '@callstack/repack';
 *
 * // ...
 * plugins: [
 *   new Repack.HermesBytecodePlugin({
 *    enabled: mode === 'production',
 *    test: /\.(js)?bundle$/,
 *    exclude: /index.bundle$/,
 *   }),
 * ]
 * ```
 *
 * @category Webpack Plugin
 */
export declare class HermesBytecodePlugin {
    private config;
    constructor(config: HermesBytecodePluginConfig);
    apply(compiler: RspackCompiler): void;
    apply(compiler: WebpackCompiler): void;
}
