"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ReanimatedPlugin = void 0;
const node_path_1 = __importDefault(require("node:path"));
const semver_1 = __importDefault(require("semver"));
const rules_js_1 = require("./rules.js");
class ReanimatedPlugin {
    constructor(options = {}) {
        this.options = options;
    }
    apply(__compiler) {
        const compiler = __compiler;
        const reanimatedPath = this.ensureDependencyInstalled(compiler.context, 'react-native-reanimated');
        const reanimatedVersion = this.getReanimatedVersion(reanimatedPath);
        if (reanimatedVersion.major >= 4) {
            this.ensureDependencyInstalled(compiler.context, 'react-native-worklets');
        }
        if (!this.options.unstable_disableTransform) {
            // add rules for transpiling with reanimated loader
            // TODO made obsolete by the new babel-swc-loader, remove in 6.0
            compiler.options.module.rules.push((0, rules_js_1.createReanimatedModuleRules)(reanimatedVersion.major, this.options.babelPluginOptions));
        }
        // ignore the 'setUpTests' warning from reanimated which is not relevant
        compiler.options.ignoreWarnings = compiler.options.ignoreWarnings ?? [];
        compiler.options.ignoreWarnings.push((warning) => /'`setUpTests` is available only in Jest environment\.'/.test(warning.message));
    }
    ensureDependencyInstalled(context, dependency) {
        try {
            // resolve the path to the dependency package.json
            // since its always in the root dir of the dependency
            const dependencyPackageJsonPath = node_path_1.default.join(dependency, 'package.json');
            const dependencyPath = require.resolve(dependencyPackageJsonPath, {
                paths: [context],
            });
            return node_path_1.default.dirname(dependencyPath);
        }
        catch {
            const error = new Error(`[RepackReanimatedPlugin] Dependency named '${dependency}' is required but not found in your project. ` +
                'Did you forget to install it?');
            // remove the stack trace to make the error more readable
            error.stack = undefined;
            throw error;
        }
    }
    getReanimatedVersion(reanimatedPath) {
        const reanimatedPackageJsonPath = node_path_1.default.join(reanimatedPath, 'package.json');
        const reanimatedPackageJson = require(reanimatedPackageJsonPath);
        const version = semver_1.default.parse(reanimatedPackageJson.version);
        if (!version) {
            throw new Error(`[RepackReanimatedPlugin] Unable to parse version for react-native-reanimated: ${reanimatedPackageJson.version}`);
        }
        return version;
    }
}
exports.ReanimatedPlugin = ReanimatedPlugin;
