"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.runAdbReverse = runAdbReverse;
var _nodePath = _interopRequireDefault(require("node:path"));
var _execa = _interopRequireDefault(require("execa"));
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
function getAdbPath() {
  const androidHome = process.env.ANDROID_HOME;
  return androidHome ? _nodePath.default.join(androidHome, 'platform-tools', 'adb') : 'adb';
}
function parseAdbError(error) {
  const errorMessage = error.message;
  const message = errorMessage.includes('error:') ? errorMessage.split('error:')[1] : errorMessage;
  return message.trim();
}
async function executeAdbCommand(command, logger) {
  const adbPath = getAdbPath();
  try {
    const result = await _execa.default.command(`${adbPath} ${command}`);
    logger.debug(`[ADB] "adb ${command}" executed successfully.`);
    return result;
  } catch (error) {
    const message = parseAdbError(error);
    logger.debug(`[ADB] "adb ${command}" failed: "${message}"`);
    throw new Error(message);
  }
}
async function waitForDevice(logger) {
  try {
    await executeAdbCommand('wait-for-device', logger);
  } catch (error) {
    const message = error.message;
    // Ignore the error if there are multiple devices/emulators
    // we only care about about at least 1 device being online
    if (/more than one device\/emulator/.test(message)) {
      return;
    }
    throw error;
  }
}
async function reversePort(port, device, logger) {
  await executeAdbCommand(`-s ${device} reverse tcp:${port} tcp:${port}`, logger);
}
async function getDevices(logger) {
  const {
    stdout
  } = await executeAdbCommand('devices', logger);
  const devices = stdout.split('\n').slice(1).map(line => line.split('\t')[0]).filter(Boolean);
  logger.debug(`[ADB] Found ${devices.length} devices/emulators.`);
  return devices;
}

/**
 * Runs the adb reverse command to reverse the specified port on all available devices.
 * Performs the following steps:
 * 1. (Optional) Waits for the device to be available.
 * 2. Get a list of all connected devices.
 * 3. Attempts to reverse the specified port using adb for all devices.
 */
async function runAdbReverse({
  logger = console,
  port,
  verbose = false,
  wait = false
}) {
  try {
    if (wait) {
      await waitForDevice(logger);
    }
    const devices = await getDevices(logger);
    for (const device of devices) {
      await reversePort(port, device, logger);
    }
    if (verbose) {
      logger.info('[ADB] port reverse success');
    }
  } catch (error) {
    const message = error.message;
    if (verbose) {
      logger.warn(`[ADB] port reverse failed: "${message}"`);
    }
  }
}
//# sourceMappingURL=runAdbReverse.js.map