/**
 * MIT License
 *
 * Copyright (c) [2022 - Present] Stɑrry Shivɑm
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */


package com.starry.greenstash.utils

import android.Manifest
import android.content.Context
import android.content.ContextWrapper
import android.content.pm.PackageManager
import android.os.Build
import android.view.HapticFeedbackConstants
import android.view.View
import android.widget.Toast
import androidx.appcompat.app.AppCompatActivity
import androidx.compose.foundation.lazy.LazyListState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.derivedStateOf
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableIntStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.res.stringResource
import androidx.core.content.ContextCompat
import com.starry.greenstash.R
import com.starry.greenstash.database.goal.GoalPriority
import com.starry.greenstash.ui.screens.home.FilterField
import com.starry.greenstash.ui.screens.home.FilterSortType
import java.io.File
import java.io.PrintWriter

/**
 * Gets the activity from the context.
 * @return the activity if the context is an instance of [AppCompatActivity], null otherwise.
 */
fun Context.getActivity(): AppCompatActivity? = when (this) {
    is AppCompatActivity -> this
    is ContextWrapper -> baseContext.getActivity()
    else -> null
}

/**
 * Checks if the app has the notification permission.
 * @return true if the app has the notification permission, false otherwise.
 */
fun Context.hasNotificationPermission() =
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
        ContextCompat.checkSelfPermission(
            this, Manifest.permission.POST_NOTIFICATIONS
        ) == PackageManager.PERMISSION_GRANTED
    } else {
        true
    }

/**
 * Checks if the lazy list is scrolling up.
 * @return true if the list is scrolling up, false otherwise.
 */
@Composable
fun LazyListState.isScrollingUp(): Boolean {
    var previousIndex by remember(this) { mutableIntStateOf(firstVisibleItemIndex) }
    var previousScrollOffset by remember(this) { mutableIntStateOf(firstVisibleItemScrollOffset) }
    return remember(this) {
        derivedStateOf {
            if (previousIndex != firstVisibleItemIndex) {
                previousIndex > firstVisibleItemIndex
            } else {
                previousScrollOffset >= firstVisibleItemScrollOffset
            }.also {
                previousIndex = firstVisibleItemIndex
                previousScrollOffset = firstVisibleItemScrollOffset
            }
        }
    }.value
}

/**
 * Creates and shows a toast message.
 */
fun String.toToast(context: Context, length: Int = Toast.LENGTH_SHORT) {
    Toast.makeText(context, this, length).show()
}

/**
 * Validates the amount.
 * @return true if the amount is valid, false otherwise.
 */
fun String.validateAmount() =
    this.isNotEmpty() && this.isNotBlank()
            && !this.matches("[0.]+".toRegex())
            && !this.endsWith(".")

/**
 * Clears the text content of the file.
 */
fun File.clearText() {
    PrintWriter(this).also {
        it.print("")
        it.close()
    }
}

/**
 * Updates the text content of the file.
 */
fun File.updateText(content: String) {
    clearText()
    appendText(content)
}

/**
 * Performs a slight haptic feedback.
 */
fun View.weakHapticFeedback() {
    this.performHapticFeedback(HapticFeedbackConstants.CLOCK_TICK)
}

/**
 * Performs a strong haptic feedback.
 */
fun View.strongHapticFeedback() {
    this.performHapticFeedback(HapticFeedbackConstants.LONG_PRESS)
}

/**
 * Translates FilterField enum values
 */
@Composable
fun FilterField.displayName(): String {
    return when (this) {
        FilterField.Title -> stringResource(R.string.filter_title)
        FilterField.Amount -> stringResource(R.string.filter_amount)
        FilterField.Priority -> stringResource(R.string.filter_priority)
    }
}

/**
 * Translates FilterSortType enum values
 */
@Composable
fun FilterSortType.displayName(): String {
    return when (this) {
        FilterSortType.Ascending -> stringResource(R.string.sort_ascending)
        FilterSortType.Descending -> stringResource(R.string.sort_descending)
    }
}

/**
 * Translates GoalPriority enum values
 */
@Composable
fun GoalPriority.displayName(): String {
    return when (this) {
        GoalPriority.High -> stringResource(R.string.priority_high)
        GoalPriority.Normal -> stringResource(R.string.priority_normal)
        GoalPriority.Low -> stringResource(R.string.priority_low)
    }
}