/**
 * MIT License
 *
 * Copyright (c) [2022 - Present] Stɑrry Shivɑm
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */


package com.starry.greenstash.ui.common

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.animation.animateContentSize
import androidx.compose.animation.expandVertically
import androidx.compose.animation.shrinkVertically
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Lightbulb
import androidx.compose.material3.Button
import androidx.compose.material3.Card
import androidx.compose.material3.CardDefaults
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.vector.ImageVector
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import com.starry.greenstash.ui.theme.greenstashFont


@Composable
fun TipCard(
    modifier: Modifier = Modifier,
    icon: ImageVector = Icons.Filled.Lightbulb,
    description: String,
    showTipCard: Boolean,
    onDismissRequest: () -> Unit
) {
    TipCard(
        modifier = modifier,
        icon = icon,
        description = description,
        showTipCard = showTipCard,
        showDismissButton = true,
        onDismissRequest = onDismissRequest
    )
}


@Composable
fun TipCardNoDismiss(
    modifier: Modifier = Modifier,
    icon: ImageVector = Icons.Filled.Lightbulb,
    description: String,
    showTipCard: Boolean,
) {
    TipCard(
        modifier = modifier,
        icon = icon,
        description = description,
        showTipCard = showTipCard,
        showDismissButton = false,
        onDismissRequest = {}
    )
}

@Composable
private fun TipCard(
    modifier: Modifier = Modifier,
    icon: ImageVector = Icons.Filled.Lightbulb,
    description: String,
    showTipCard: Boolean,
    showDismissButton: Boolean = true,
    onDismissRequest: () -> Unit
) {
    Column(
        modifier = Modifier
            .fillMaxWidth()
            .animateContentSize(),
        horizontalAlignment = Alignment.CenterHorizontally
    ) {
        AnimatedVisibility(
            visible = showTipCard,
            enter = expandVertically(),
            exit = shrinkVertically()
        ) {
            Card(
                modifier = modifier
                    .fillMaxWidth()
                    .padding(bottom = 10.dp),
                shape = RoundedCornerShape(16.dp),
                colors = CardDefaults.cardColors(
                    containerColor = MaterialTheme.colorScheme.secondaryContainer
                )
            ) {
                Column(
                    modifier = Modifier.padding(16.dp)
                ) {
                    Row(
                        modifier = Modifier.fillMaxWidth(),
                        verticalAlignment = Alignment.CenterVertically
                    ) {
                        Icon(
                            imageVector = icon,
                            contentDescription = null,
                            modifier = Modifier.size(32.dp)
                        )
                        Spacer(modifier = Modifier.width(16.dp))
                        Text(
                            text = description,
                            style = MaterialTheme.typography.titleSmall,
                            fontFamily = greenstashFont,
                        )
                    }
                    Spacer(modifier = Modifier.height(16.dp))
                    if (showDismissButton) {
                        Button(
                            onClick = { onDismissRequest() },
                            modifier = Modifier.align(Alignment.End)
                        ) {
                            Text(text = "OK")
                        }
                    }
                }
            }
        }
    }
}

@Preview
@Composable
private fun TipCardPV() {
    TipCard(
        description = "This is a tip",
        showTipCard = true,
        onDismissRequest = {}
    )
}
