"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TagNotesIndex = void 0;
const utils_1 = require("@standardnotes/utils");
const common_1 = require("@standardnotes/common");
const Tag_1 = require("../../../Syncable/Tag/Tag");
const Item_1 = require("../../../Abstract/Item");
class TagNotesIndex {
    constructor(collection, observers = []) {
        this.collection = collection;
        this.observers = observers;
        this.tagToNotesMap = {};
        this.allCountableNotes = new Set();
        this.isNoteCountable = (note) => {
            if ((0, Item_1.isDecryptedItem)(note)) {
                return !note.archived && !note.trashed;
            }
            return false;
        };
    }
    addCountChangeObserver(observer) {
        this.observers.push(observer);
        const thislessEventObservers = this.observers;
        return () => {
            (0, utils_1.removeFromArray)(thislessEventObservers, observer);
        };
    }
    notifyObservers(tagUuid) {
        for (const observer of this.observers) {
            observer(tagUuid);
        }
    }
    allCountableNotesCount() {
        return this.allCountableNotes.size;
    }
    countableNotesForTag(tag) {
        var _a;
        return ((_a = this.tagToNotesMap[tag.uuid]) === null || _a === void 0 ? void 0 : _a.size) || 0;
    }
    onChange(delta) {
        const notes = [...delta.changed, ...delta.inserted, ...delta.discarded].filter((i) => i.content_type === common_1.ContentType.Note);
        const tags = [...delta.changed, ...delta.inserted].filter(Item_1.isDecryptedItem).filter(Tag_1.isTag);
        this.receiveNoteChanges(notes);
        this.receiveTagChanges(tags);
    }
    receiveTagChanges(tags) {
        for (const tag of tags) {
            const uuids = tag.noteReferences.map((ref) => ref.uuid);
            const countableUuids = uuids.filter((uuid) => this.allCountableNotes.has(uuid));
            const previousSet = this.tagToNotesMap[tag.uuid];
            this.tagToNotesMap[tag.uuid] = new Set(countableUuids);
            if ((previousSet === null || previousSet === void 0 ? void 0 : previousSet.size) !== countableUuids.length) {
                this.notifyObservers(tag.uuid);
            }
        }
    }
    receiveNoteChanges(notes) {
        const previousAllCount = this.allCountableNotes.size;
        for (const note of notes) {
            const isCountable = this.isNoteCountable(note);
            if (isCountable) {
                this.allCountableNotes.add(note.uuid);
            }
            else {
                this.allCountableNotes.delete(note.uuid);
            }
            const associatedTagUuids = this.collection.uuidsThatReferenceUuid(note.uuid);
            for (const tagUuid of associatedTagUuids) {
                const set = this.setForTag(tagUuid);
                const previousCount = set.size;
                if (isCountable) {
                    set.add(note.uuid);
                }
                else {
                    set.delete(note.uuid);
                }
                if (previousCount !== set.size) {
                    this.notifyObservers(tagUuid);
                }
            }
        }
        if (previousAllCount !== this.allCountableNotes.size) {
            this.notifyObservers(undefined);
        }
    }
    setForTag(uuid) {
        let set = this.tagToNotesMap[uuid];
        if (!set) {
            set = new Set();
            this.tagToNotesMap[uuid] = set;
        }
        return set;
    }
}
exports.TagNotesIndex = TagNotesIndex;
