"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NotificationPayload = void 0;
const ValueObject_1 = require("../Core/ValueObject");
const Result_1 = require("../Core/Result");
const NotificationType_1 = require("./NotificationType");
const Uuid_1 = require("../Common/Uuid");
const NotificationPayloadIdentifierType_1 = require("./NotificationPayloadIdentifierType");
class NotificationPayload extends ValueObject_1.ValueObject {
    constructor(props) {
        super(props);
    }
    toString() {
        return JSON.stringify({
            version: this.props.version,
            type: this.props.type.value,
            primaryIdentifier: this.props.primaryIdentifier.value,
            primaryIndentifierType: this.props.primaryIndentifierType.value,
            secondaryIdentifier: this.props.secondaryIdentifier?.value,
            secondaryIdentifierType: this.props.secondaryIdentifierType?.value,
        });
    }
    static createFromString(jsonPayload) {
        try {
            const props = JSON.parse(jsonPayload);
            const typeOrError = NotificationType_1.NotificationType.create(props.type);
            if (typeOrError.isFailed()) {
                return Result_1.Result.fail(typeOrError.getError());
            }
            const type = typeOrError.getValue();
            const primaryIdentifierOrError = Uuid_1.Uuid.create(props.primaryIdentifier);
            if (primaryIdentifierOrError.isFailed()) {
                return Result_1.Result.fail(primaryIdentifierOrError.getError());
            }
            const primaryIdentifier = primaryIdentifierOrError.getValue();
            const primaryIndentifierTypeOrError = NotificationPayloadIdentifierType_1.NotificationPayloadIdentifierType.create(props.primaryIndentifierType);
            if (primaryIndentifierTypeOrError.isFailed()) {
                return Result_1.Result.fail(primaryIndentifierTypeOrError.getError());
            }
            const primaryIndentifierType = primaryIndentifierTypeOrError.getValue();
            let secondaryIdentifier;
            if (props.secondaryIdentifier) {
                const secondaryIdentifierOrError = Uuid_1.Uuid.create(props.secondaryIdentifier);
                if (secondaryIdentifierOrError.isFailed()) {
                    return Result_1.Result.fail(secondaryIdentifierOrError.getError());
                }
                secondaryIdentifier = secondaryIdentifierOrError.getValue();
            }
            let secondaryIdentifierType;
            if (props.secondaryIdentifierType) {
                const secondaryIdentifierTypeOrError = NotificationPayloadIdentifierType_1.NotificationPayloadIdentifierType.create(props.secondaryIdentifierType);
                if (secondaryIdentifierTypeOrError.isFailed()) {
                    return Result_1.Result.fail(secondaryIdentifierTypeOrError.getError());
                }
                secondaryIdentifierType = secondaryIdentifierTypeOrError.getValue();
            }
            return NotificationPayload.create({
                version: props.version,
                type,
                primaryIdentifier,
                primaryIndentifierType,
                secondaryIdentifier,
                secondaryIdentifierType,
            });
        }
        catch (error) {
            return Result_1.Result.fail(error.message);
        }
    }
    static create(props) {
        if (props.secondaryIdentifier === undefined &&
            props.type.equals(NotificationType_1.NotificationType.create(NotificationType_1.NotificationType.TYPES.SharedVaultItemRemoved).getValue())) {
            return Result_1.Result.fail(`Item uuid is required for ${NotificationType_1.NotificationType.TYPES.SharedVaultItemRemoved} notification type`);
        }
        return Result_1.Result.ok(new NotificationPayload(props));
    }
}
exports.NotificationPayload = NotificationPayload;
