import { Uuid } from '@standardnotes/common';
import { SNRootKey } from '@standardnotes/encryption';
import * as Encryption from '@standardnotes/encryption';
import * as Services from '@standardnotes/services';
import { DiagnosticInfo } from '@standardnotes/services';
import { FullyFormedPayloadInterface, DeletedPayloadInterface, Environment } from '@standardnotes/models';
/**
 * The storage service is responsible for persistence of both simple key-values, and payload
 * storage. It does so by relying on deviceInterface to save and retrieve raw values and payloads.
 * For simple key/values, items are grouped together in an in-memory hash, and persisted to disk
 * as a single object (encrypted, when possible). It handles persisting payloads in the local
 * database by encrypting the payloads when possible.
 * The storage service also exposes methods that allow the application to initially
 * decrypt the persisted key/values, and also a method to determine whether a particular
 * key can decrypt wrapped storage.
 */
export declare class DiskStorageService extends Services.AbstractService implements Services.StorageServiceInterface {
    private deviceInterface;
    private identifier;
    private environment;
    protected internalEventBus: Services.InternalEventBusInterface;
    private encryptionProvider;
    private storagePersistable;
    private persistencePolicy;
    private encryptionPolicy;
    private needsPersist;
    private currentPersistPromise?;
    private values;
    constructor(deviceInterface: Services.DeviceInterface, identifier: string, environment: Environment, internalEventBus: Services.InternalEventBusInterface);
    provideEncryptionProvider(provider: Encryption.EncryptionProviderInterface): void;
    deinit(): void;
    handleApplicationStage(stage: Services.ApplicationStage): Promise<void>;
    setPersistencePolicy(persistencePolicy: Services.StoragePersistencePolicies): Promise<void>;
    setEncryptionPolicy(encryptionPolicy: Services.StorageEncryptionPolicy, persist?: boolean): void;
    isEphemeralSession(): boolean;
    initializeFromDisk(): Promise<void>;
    private setInitialValues;
    isStorageWrapped(): boolean;
    canDecryptWithKey(key: SNRootKey): Promise<boolean>;
    private decryptWrappedValue;
    decryptStorage(): Promise<void>;
    /** @todo This function should be debounced. */
    private persistValuesToDisk;
    awaitPersist(): Promise<void>;
    private immediatelyPersistValuesToDisk;
    /**
     * Generates a payload that can be persisted to disk,
     * either as a plain object, or an encrypted item.
     */
    private generatePersistableValues;
    setValue(key: string, value: unknown, mode?: Services.StorageValueModes): void;
    setValueAndAwaitPersist(key: string, value: unknown, mode?: Services.StorageValueModes): Promise<void>;
    private setValueWithNoPersist;
    getValue<T>(key: string, mode?: Services.StorageValueModes, defaultValue?: T): T;
    removeValue(key: string, mode?: Services.StorageValueModes): Promise<void>;
    getStorageEncryptionPolicy(): Services.StorageEncryptionPolicy;
    /**
     * Default persistence key. Platforms can override as needed.
     */
    private getPersistenceKey;
    private defaultValuesObject;
    static DefaultValuesObject(wrapped?: Services.WrappedStorageValue, unwrapped?: Services.ValuesObjectRecord, nonwrapped?: Services.ValuesObjectRecord): Services.StorageValuesObject;
    private domainKeyForMode;
    /**
     * Clears simple values from storage only. Does not affect payloads.
     */
    clearValues(): Promise<void>;
    getAllRawPayloads(): Promise<import("@standardnotes/models").FullyFormedTransferPayload[]>;
    savePayload(payload: FullyFormedPayloadInterface): Promise<void>;
    savePayloads(payloads: FullyFormedPayloadInterface[]): Promise<void>;
    deletePayloads(payloads: DeletedPayloadInterface[]): Promise<void>;
    forceDeletePayloads(payloads: FullyFormedPayloadInterface[]): Promise<void>;
    deletePayloadWithId(uuid: Uuid): Promise<void>;
    clearAllPayloads(): Promise<void>;
    clearAllData(): Promise<void>;
    getDiagnostics(): Promise<DiagnosticInfo | undefined>;
}
