import { ContentType, Uuid } from '@standardnotes/common';
import { PayloadsChangeObserverCallback } from './Types';
import { ImmutablePayloadCollection, EncryptedPayloadInterface, DeletedPayloadInterface, DecryptedPayloadInterface, PayloadCollection, PayloadEmitSource, FullyFormedPayloadInterface, HistoryMap, DeltaEmit } from '@standardnotes/models';
import { AbstractService, PayloadManagerInterface, InternalEventBusInterface, DiagnosticInfo } from '@standardnotes/services';
import { IntegrityPayload } from '@standardnotes/responses';
/**
 * The payload manager is responsible for keeping state regarding what items exist in the
 * global application state. It does so by exposing functions that allow consumers to 'map'
 * a detached payload into global application state. Whenever a change is made or retrieved
 * from any source, it must be mapped in order to be properly reflected in global application state.
 * The model manager deals only with in-memory state, and does not deal directly with storage.
 * It also serves as a query store, and can be queried for current notes, tags, etc.
 * It exposes methods that allow consumers to listen to mapping events. This is how
 * applications 'stream' items to display in the interface.
 */
export declare class PayloadManager extends AbstractService implements PayloadManagerInterface {
    protected internalEventBus: InternalEventBusInterface;
    private changeObservers;
    collection: PayloadCollection<FullyFormedPayloadInterface>;
    private emitQueue;
    constructor(internalEventBus: InternalEventBusInterface);
    /**
     * Our payload collection keeps the latest mapped payload for every payload
     * that passes through our mapping function. Use this to query current state
     * as needed to make decisions, like about duplication or uuid alteration.
     */
    getMasterCollection(): ImmutablePayloadCollection<FullyFormedPayloadInterface<import("@standardnotes/models").ItemContent>>;
    deinit(): void;
    resetState(): void;
    find(uuids: Uuid[]): FullyFormedPayloadInterface[];
    findOne(uuid: Uuid): FullyFormedPayloadInterface | undefined;
    all(contentType: ContentType): FullyFormedPayloadInterface[];
    get integrityPayloads(): IntegrityPayload[];
    get nonDeletedItems(): FullyFormedPayloadInterface[];
    get invalidPayloads(): EncryptedPayloadInterface[];
    emitDeltaEmit<P extends FullyFormedPayloadInterface = FullyFormedPayloadInterface>(emit: DeltaEmit<P>, sourceKey?: string): Promise<P[]>;
    /**
     * One of many mapping helpers available.
     * This function maps a payload to an item
     * @returns every paylod altered as a result of this operation, to be
     * saved to storage by the caller
     */
    emitPayload<P extends FullyFormedPayloadInterface = FullyFormedPayloadInterface>(payload: P, source: PayloadEmitSource, sourceKey?: string): Promise<P[]>;
    /**
     * This function maps multiple payloads to items, and is the authoratative mapping
     * function that all other mapping helpers rely on
     * @returns every paylod altered as a result of this operation, to be
     * saved to storage by the caller
     */
    emitPayloads<P extends FullyFormedPayloadInterface = FullyFormedPayloadInterface>(payloads: P[], source: PayloadEmitSource, sourceKey?: string): Promise<P[]>;
    private popQueue;
    private applyPayloads;
    /**
     * Notifies observers when an item has been mapped.
     * @param types - An array of content types to listen for
     * @param priority - The lower the priority, the earlier the function is called
     *  wrt to other observers
     */
    addObserver(types: ContentType | ContentType[], callback: PayloadsChangeObserverCallback, priority?: number): () => void;
    /**
     * This function is mostly for internal use, but can be used externally by consumers who
     * explicitely understand what they are doing (want to propagate model state without mapping)
     */
    notifyChangeObservers(changed: FullyFormedPayloadInterface[], inserted: FullyFormedPayloadInterface[], discarded: DeletedPayloadInterface[], ignored: EncryptedPayloadInterface[], unerrored: DecryptedPayloadInterface[], source: PayloadEmitSource, sourceKey?: string): void;
    /**
     * Imports an array of payloads from an external source (such as a backup file)
     * and marks the items as dirty.
     * @returns Resulting items
     */
    importPayloads(payloads: DecryptedPayloadInterface[], historyMap: HistoryMap): Promise<Uuid[]>;
    removePayloadLocally(payload: FullyFormedPayloadInterface): void;
    erroredPayloadsForContentType(contentType: ContentType): EncryptedPayloadInterface[];
    deleteErroredPayloads(payloads: EncryptedPayloadInterface[]): Promise<void>;
    getDiagnostics(): Promise<DiagnosticInfo | undefined>;
}
