import { DecryptedPayloadInterface, EncryptedItemInterface } from '@standardnotes/models';
import { SNSyncService } from '../Sync/SyncService';
import { DiskStorageService } from '../Storage/DiskStorageService';
import { PayloadManager } from '../Payloads/PayloadManager';
import { ChallengeService } from '../Challenge';
import { SNApiService } from '../Api/ApiService';
import { ItemManager } from '../Items/ItemManager';
import { ClientDisplayableError } from '@standardnotes/responses';
import { AlertService, AbstractService, InternalEventBusInterface, ApplicationStage, DiagnosticInfo, EncryptionService, UserService } from '@standardnotes/services';
import { KeyRecoveryEvent } from './Types';
/**
 * The key recovery service listens to items key changes to detect any that cannot be decrypted.
 * If it detects an items key that is not properly decrypted, it will present a key recovery
 * wizard (using existing UI like Challenges and AlertService) that will attempt to recover
 * the root key for those keys.
 *
 * When we encounter an items key we cannot decrypt, this is a sign that the user's password may
 * have recently changed (even though their session is still valid). If the user has been
 * previously signed in, we take this opportunity to reach out to the server to get the
 * user's current key_params. We ensure these key params' version is equal to or greater than our own.

 * - If this key's key params are equal to the retrieved parameters,
    and this keys created date is greater than any existing valid items key,
    or if we do not have any items keys:
       1. Use the decryption of this key as a source of validation
       2. If valid, replace our local root key with this new root key and emit the decrypted items key
 * - Else, if the key params are not equal,
     or its created date is less than an existing valid items key
        1. Attempt to decrypt this key using its attached key paramas
        2. If valid, emit decrypted items key. DO NOT replace local root key.
 * - If by the end we did not find an items key with matching key params to the retrieved
     key params, AND the retrieved key params are newer than what we have locally, we must
     issue a sign in request to the server.

 * If the user is not signed in and we detect an undecryptable items key, we present a detached
 * recovery wizard that doesn't affect our local root key.
 *
 * When an items key is emitted, protocol service will automatically try to decrypt any
 * related items that are in an errored state.
 *
 * In the item observer, `ignored` items represent items who have encrypted overwrite
 * protection enabled (only items keys). This means that if the incoming payload is errored,
 * but our current copy is not, we will ignore the incoming value until we can properly
 * decrypt it.
 */
export declare class SNKeyRecoveryService extends AbstractService<KeyRecoveryEvent, DecryptedPayloadInterface[]> {
    private itemManager;
    private payloadManager;
    private apiService;
    private protocolService;
    private challengeService;
    private alertService;
    private storageService;
    private syncService;
    private userService;
    protected internalEventBus: InternalEventBusInterface;
    private removeItemObserver;
    private decryptionQueue;
    private isProcessingQueue;
    constructor(itemManager: ItemManager, payloadManager: PayloadManager, apiService: SNApiService, protocolService: EncryptionService, challengeService: ChallengeService, alertService: AlertService, storageService: DiskStorageService, syncService: SNSyncService, userService: UserService, internalEventBus: InternalEventBusInterface);
    deinit(): void;
    handleApplicationStage(stage: ApplicationStage): Promise<void>;
    /**
     * Ignored items keys are items keys which arrived from a remote source, which we were
     * not able to decrypt, and for which we already had an existing items key that was
     * properly decrypted. Since items keys key contents are immutable, if we already have a
     * successfully decrypted version, yet we can't decrypt the new version, we should
     * temporarily ignore the new version until we can properly decrypt it (through the recovery flow),
     * and not overwrite the local copy.
     *
     * Ignored items are persisted to disk in isolated storage so that they may be decrypted
     * whenever. When they are finally decryptable, we will emit them and update our database
     * with the new decrypted value.
     *
     * When the app first launches, we will query the isolated storage to see if there are any
     * keys we need to decrypt.
     */
    private handleIgnoredItemsKeys;
    private handleUndecryptableItemsKeys;
    presentKeyRecoveryWizard(): void;
    canAttemptDecryptionOfItem(item: EncryptedItemInterface): ClientDisplayableError | true;
    processPersistedUndecryptables(): Promise<void>;
    private getUndecryptables;
    private persistUndecryptables;
    private saveToUndecryptables;
    private removeFromUndecryptables;
    private getClientKeyParams;
    private performServerSignIn;
    private getWrappingKeyIfApplicable;
    private addKeysToQueue;
    private readdQueueItem;
    private getLatestKeyParamsFromServer;
    private beginKeyRecovery;
    private potentiallyPerformFallbackSignInToUpdateOutdatedLocalRootKey;
    private processQueue;
    private processQueueItem;
    private handleDecryptionOfAllKeysMatchingCorrectRootKey;
    private removeElementsFromQueueForMatchingKeyParams;
    getDiagnostics(): Promise<DiagnosticInfo | undefined>;
}
