import { ContentType } from '@standardnotes/common';
import { SNItemsKey } from '@standardnotes/encryption';
import { PayloadManager } from '../Payloads/PayloadManager';
import { UuidString } from '../../Types/UuidString';
import * as Models from '@standardnotes/models';
import * as Services from '@standardnotes/services';
import { DiagnosticInfo, ItemsClientInterface, ItemRelationshipDirection } from '@standardnotes/services';
import { ApplicationDisplayOptions } from '../../Application/Options/OptionalOptions';
import { DecryptedItemInterface, ItemContent } from '@standardnotes/models';
/**
 * The item manager is backed by the payload manager. It listens for change events from the
 * global payload manager, and converts any payloads to items, then propagates those items to
 * listeners on the item manager. When the item manager makes a change to an item, it will modify
 * items using a mutator, then emit those payloads to the payload manager. The payload manager
 * will then notify its observers (which is us), we'll convert the payloads to items,
 * and then we'll propagate them to our listeners.
 */
export declare class ItemManager extends Services.AbstractService implements Services.ItemManagerInterface, ItemsClientInterface {
    private payloadManager;
    private readonly options;
    protected internalEventBus: Services.InternalEventBusInterface;
    private unsubChangeObserver;
    private observers;
    private collection;
    private systemSmartViews;
    private tagItemsIndex;
    private navigationDisplayController;
    private tagDisplayController;
    private itemsKeyDisplayController;
    private componentDisplayController;
    private themeDisplayController;
    private fileDisplayController;
    private smartViewDisplayController;
    constructor(payloadManager: PayloadManager, options: ApplicationDisplayOptions, internalEventBus: Services.InternalEventBusInterface);
    private rebuildSystemSmartViews;
    private createCollection;
    private get allDisplayControllers();
    get invalidItems(): Models.EncryptedItemInterface[];
    createItemFromPayload(payload: Models.DecryptedPayloadInterface): Models.DecryptedItemInterface;
    createPayloadFromObject(object: Models.DecryptedTransferPayload): Models.DecryptedPayloadInterface;
    setPrimaryItemDisplayOptions(options: Models.DisplayOptions): void;
    getDisplayableNotes(): Models.SNNote[];
    getDisplayableFiles(): Models.FileItem[];
    getDisplayableNotesAndFiles(): (Models.SNNote | Models.FileItem)[];
    getDisplayableTags(): Models.SNTag[];
    getDisplayableItemsKeys(): SNItemsKey[];
    getDisplayableComponents(): (Models.SNComponent | Models.SNTheme)[];
    deinit(): void;
    resetState(): void;
    findItem<T extends Models.DecryptedItemInterface = Models.DecryptedItemInterface>(uuid: UuidString): T | undefined;
    findAnyItem(uuid: UuidString): Models.ItemInterface | undefined;
    findAnyItems(uuids: UuidString[]): Models.ItemInterface[];
    private findSystemSmartView;
    findSureItem<T extends Models.DecryptedItemInterface = Models.DecryptedItemInterface>(uuid: UuidString): T;
    /**
     * Returns all items matching given ids
     */
    findItems<T extends Models.DecryptedItemInterface>(uuids: UuidString[]): T[];
    /**
     * If item is not found, an `undefined` element
     * will be inserted into the array.
     */
    findItemsIncludingBlanks<T extends Models.DecryptedItemInterface>(uuids: UuidString[]): (T | undefined)[];
    get items(): Models.DecryptedItemInterface[];
    allTrackedItems(): Models.ItemInterface[];
    hasTagsNeedingFoldersMigration(): boolean;
    addNoteCountChangeObserver(observer: Models.TagItemCountChangeObserver): () => void;
    allCountableNotesCount(): number;
    countableNotesForTag(tag: Models.SNTag | Models.SmartView): number;
    getNoteCount(): number;
    addObserver<I extends Models.DecryptedItemInterface = Models.DecryptedItemInterface>(contentType: ContentType | ContentType[], callback: Services.ItemManagerChangeObserverCallback<I>): () => void;
    /**
     * Returns the items that reference the given item, or an empty array if no results.
     */
    itemsReferencingItem(itemToLookupUuidFor: Models.DecryptedItemInterface, contentType?: ContentType): Models.DecryptedItemInterface[];
    /**
     * Returns all items that an item directly references
     */
    referencesForItem(itemToLookupUuidFor: Models.DecryptedItemInterface, contentType?: ContentType): Models.DecryptedItemInterface[];
    private setPayloads;
    private notifyObserversByUiAdjustingDelta;
    private notifyObservers;
    /**
     * Consumers wanting to modify an item should run it through this block,
     * so that data is properly mapped through our function, and latest state
     * is properly reconciled.
     */
    changeItem<M extends Models.DecryptedItemMutator = Models.DecryptedItemMutator, I extends Models.DecryptedItemInterface = Models.DecryptedItemInterface>(itemToLookupUuidFor: I, mutate?: (mutator: M) => void, mutationType?: Models.MutationType, emitSource?: Models.PayloadEmitSource, payloadSourceKey?: string): Promise<I>;
    /**
     * @param mutate If not supplied, the intention would simply be to mark the item as dirty.
     */
    changeItems<M extends Models.DecryptedItemMutator = Models.DecryptedItemMutator, I extends Models.DecryptedItemInterface = Models.DecryptedItemInterface>(itemsToLookupUuidsFor: I[], mutate?: (mutator: M) => void, mutationType?: Models.MutationType, emitSource?: Models.PayloadEmitSource, payloadSourceKey?: string): Promise<I[]>;
    /**
     * Run unique mutations per each item in the array, then only propagate all changes
     * once all mutations have been run. This differs from `changeItems` in that changeItems
     * runs the same mutation on all items.
     */
    runTransactionalMutations(transactions: Models.TransactionalMutation[], emitSource?: Models.PayloadEmitSource, payloadSourceKey?: string): Promise<(Models.DecryptedItemInterface | undefined)[]>;
    runTransactionalMutation(transaction: Models.TransactionalMutation, emitSource?: Models.PayloadEmitSource, payloadSourceKey?: string): Promise<Models.DecryptedItemInterface | undefined>;
    changeNote(itemToLookupUuidFor: Models.SNNote, mutate: (mutator: Models.NoteMutator) => void, mutationType?: Models.MutationType, emitSource?: Models.PayloadEmitSource, payloadSourceKey?: string): Promise<Models.DecryptedPayloadInterface[]>;
    changeTag(itemToLookupUuidFor: Models.SNTag, mutate: (mutator: Models.TagMutator) => void, mutationType?: Models.MutationType, emitSource?: Models.PayloadEmitSource, payloadSourceKey?: string): Promise<Models.SNTag>;
    changeComponent(itemToLookupUuidFor: Models.SNComponent, mutate: (mutator: Models.ComponentMutator) => void, mutationType?: Models.MutationType, emitSource?: Models.PayloadEmitSource, payloadSourceKey?: string): Promise<Models.SNComponent>;
    changeFeatureRepo(itemToLookupUuidFor: Models.SNFeatureRepo, mutate: (mutator: Models.FeatureRepoMutator) => void, mutationType?: Models.MutationType, emitSource?: Models.PayloadEmitSource, payloadSourceKey?: string): Promise<Models.SNFeatureRepo>;
    changeActionsExtension(itemToLookupUuidFor: Models.SNActionsExtension, mutate: (mutator: Models.ActionsExtensionMutator) => void, mutationType?: Models.MutationType, emitSource?: Models.PayloadEmitSource, payloadSourceKey?: string): Promise<Models.SNActionsExtension>;
    changeItemsKey(itemToLookupUuidFor: Models.ItemsKeyInterface, mutate: (mutator: Models.ItemsKeyMutatorInterface) => void, mutationType?: Models.MutationType, emitSource?: Models.PayloadEmitSource, payloadSourceKey?: string): Promise<Models.ItemsKeyInterface>;
    private applyTransform;
    /**
     * Sets the item as needing sync. The item is then run through the mapping function,
     * and propagated to mapping observers.
     * @param isUserModified - Whether to update the item's "user modified date"
     */
    setItemDirty(itemToLookupUuidFor: Models.DecryptedItemInterface, isUserModified?: boolean): Promise<Models.DecryptedItemInterface<Models.ItemContent>>;
    setItemsDirty(itemsToLookupUuidsFor: Models.DecryptedItemInterface[], isUserModified?: boolean): Promise<Models.DecryptedItemInterface[]>;
    /**
     * Returns an array of items that need to be synced.
     */
    getDirtyItems(): (Models.DecryptedItemInterface | Models.DeletedItemInterface)[];
    /**
     * Duplicates an item and maps it, thus propagating the item to observers.
     * @param isConflict - Whether to mark the duplicate as a conflict of the original.
     */
    duplicateItem<T extends Models.DecryptedItemInterface>(itemToLookupUuidFor: T, isConflict?: boolean, additionalContent?: Partial<Models.ItemContent>): Promise<T>;
    createItem<T extends Models.DecryptedItemInterface, C extends Models.ItemContent = Models.ItemContent>(contentType: ContentType, content: C, needsSync?: boolean): Promise<T>;
    createTemplateItem<C extends Models.ItemContent = Models.ItemContent, I extends Models.DecryptedItemInterface<C> = Models.DecryptedItemInterface<C>>(contentType: ContentType, content?: C, override?: Partial<Models.DecryptedPayload<C>>): I;
    /**
     * @param item item to be checked
     * @returns Whether the item is a template (unmanaged)
     */
    isTemplateItem(item: Models.DecryptedItemInterface): boolean;
    insertItem(item: Models.DecryptedItemInterface): Promise<Models.DecryptedItemInterface>;
    insertItems(items: Models.DecryptedItemInterface[], emitSource?: Models.PayloadEmitSource): Promise<Models.DecryptedItemInterface[]>;
    emitItemFromPayload(payload: Models.DecryptedPayloadInterface, emitSource: Models.PayloadEmitSource): Promise<Models.DecryptedItemInterface>;
    emitItemsFromPayloads(payloads: Models.DecryptedPayloadInterface[], emitSource: Models.PayloadEmitSource): Promise<Models.DecryptedItemInterface[]>;
    setItemToBeDeleted(itemToLookupUuidFor: Models.DecryptedItemInterface | Models.EncryptedItemInterface, source?: Models.PayloadEmitSource): Promise<void>;
    setItemsToBeDeleted(itemsToLookupUuidsFor: (Models.DecryptedItemInterface | Models.EncryptedItemInterface)[]): Promise<void>;
    getItems<T extends Models.DecryptedItemInterface>(contentType: ContentType | ContentType[]): T[];
    getAnyItems(contentType: ContentType | ContentType[]): Models.ItemInterface[];
    itemsMatchingPredicate<T extends Models.DecryptedItemInterface>(contentType: ContentType, predicate: Models.PredicateInterface<T>): T[];
    itemsMatchingPredicates<T extends Models.DecryptedItemInterface>(contentType: ContentType, predicates: Models.PredicateInterface<T>[]): T[];
    subItemsMatchingPredicates<T extends Models.DecryptedItemInterface>(items: T[], predicates: Models.PredicateInterface<T>[]): T[];
    getRootTags(): Models.SNTag[];
    findTagByTitle(title: string): Models.SNTag | undefined;
    findTagByTitleAndParent(title: string, parentItemToLookupUuidFor?: Models.SNTag): Models.SNTag | undefined;
    /**
     * Finds tags with title or component starting with a search query and (optionally) not associated with a note
     * @param searchQuery - The query string to match
     * @param note - The note whose tags should be omitted from results
     * @returns Array containing tags matching search query and not associated with note
     */
    searchTags(searchQuery: string, note?: Models.SNNote): Models.SNTag[];
    getTagParent(itemToLookupUuidFor: Models.SNTag): Models.SNTag | undefined;
    getTagPrefixTitle(tag: Models.SNTag): string | undefined;
    getTagLongTitle(tag: Models.SNTag): string;
    getTagParentChain(itemToLookupUuidFor: Models.SNTag): Models.SNTag[];
    findOrCreateTagParentChain(titlesHierarchy: string[]): Promise<Models.SNTag>;
    getTagChildren(itemToLookupUuidFor: Models.SNTag): Models.SNTag[];
    isTagAncestor(tagToLookUpUuidFor: Models.SNTag, childToLookUpUuidFor: Models.SNTag): boolean;
    isValidTagParent(parentTagToLookUpUuidFor: Models.SNTag, childToLookUpUuidFor: Models.SNTag): boolean;
    /**
     * @returns The changed child tag
     */
    setTagParent(parentTag: Models.SNTag, childTag: Models.SNTag): Promise<Models.SNTag>;
    /**
     * @returns The changed child tag
     */
    unsetTagParent(childTag: Models.SNTag): Promise<Models.SNTag>;
    associateFileWithNote(file: Models.FileItem, note: Models.SNNote): Promise<Models.FileItem>;
    disassociateFileWithNote(file: Models.FileItem, note: Models.SNNote): Promise<Models.FileItem>;
    addTagToNote(note: Models.SNNote, tag: Models.SNTag, addHierarchy: boolean): Promise<Models.SNTag[]>;
    addTagToFile(file: Models.FileItem, tag: Models.SNTag, addHierarchy: boolean): Promise<Models.SNTag[]>;
    linkNoteToNote(note: Models.SNNote, otherNote: Models.SNNote): Promise<Models.SNNote>;
    linkFileToFile(file: Models.FileItem, otherFile: Models.FileItem): Promise<Models.FileItem>;
    unlinkItems(itemA: DecryptedItemInterface<ItemContent>, itemB: DecryptedItemInterface<ItemContent>): Promise<Models.DecryptedItemInterface<Models.ItemContent>>;
    /**
     * Get tags for a note sorted in natural order
     * @param item - The item whose tags will be returned
     * @returns Array containing tags associated with an item
     */
    getSortedTagsForItem(item: DecryptedItemInterface<ItemContent>): Models.SNTag[];
    createTag(title: string, parentItemToLookupUuidFor?: Models.SNTag): Promise<Models.SNTag>;
    createSmartView<T extends Models.DecryptedItemInterface>(title: string, predicate: Models.PredicateInterface<T>): Promise<Models.SmartView>;
    createSmartViewFromDSL<T extends Models.DecryptedItemInterface>(dsl: string): Promise<Models.SmartView>;
    createTagOrSmartView(title: string): Promise<Models.SNTag | Models.SmartView>;
    isSmartViewTitle(title: string): boolean;
    /**
     * Finds or creates a tag with a given title
     */
    findOrCreateTagByTitle(title: string, parentItemToLookupUuidFor?: Models.SNTag): Promise<Models.SNTag>;
    notesMatchingSmartView(view: Models.SmartView): Models.SNNote[];
    get allNotesSmartView(): Models.SmartView;
    get archivedSmartView(): Models.SmartView;
    get trashSmartView(): Models.SmartView;
    get untaggedNotesSmartView(): Models.SmartView;
    get trashedItems(): Models.SNNote[];
    /**
     * Permanently deletes any items currently in the trash. Consumer must manually call sync.
     */
    emptyTrash(): Promise<void>;
    /**
     * Returns all smart views, sorted by title.
     */
    getSmartViews(): Models.SmartView[];
    /**
     * The number of notes currently managed
     */
    get noteCount(): number;
    /**
     * Immediately removes all items from mapping state and notifies observers
     * Used primarily when signing into an account and wanting to discard any current
     * local data.
     */
    removeAllItemsFromMemory(): Promise<void>;
    removeItemLocally(item: Models.DecryptedItemInterface | Models.DeletedItemInterface): void;
    renameFile(file: Models.FileItem, name: string): Promise<Models.FileItem>;
    setLastSyncBeganForItems(itemsToLookupUuidsFor: (Models.DecryptedItemInterface | Models.DeletedItemInterface)[], date: Date, globalDirtyIndex: number): Promise<(Models.DecryptedItemInterface | Models.DeletedItemInterface)[]>;
    relationshipDirectionBetweenItems(itemA: Models.DecryptedItemInterface<Models.ItemContent>, itemB: Models.DecryptedItemInterface<Models.ItemContent>): ItemRelationshipDirection;
    getDiagnostics(): Promise<DiagnosticInfo | undefined>;
}
