import { ItemManager } from '../Items/ItemManager';
import { SNApiService } from '../Api/ApiService';
import { DiskStorageService } from '../Storage/DiskStorageService';
import * as Models from '@standardnotes/models';
import * as Responses from '@standardnotes/responses';
import { SNNote } from '@standardnotes/models';
import { AbstractService, EncryptionService, DeviceInterface, InternalEventBusInterface } from '@standardnotes/services';
/**
 * The history manager is responsible for:
 * 1. Transient session history, which include keeping track of changes made in the
 *    current application session. These change logs (unless otherwise configured) are
 *    ephemeral and do not persist past application restart. Session history entries are
 *    added via change observers that trigger when an item changes.
 * 2. Remote server history. Entries are automatically added by the server and must be
 *    retrieved per item via an API call.
 */
export declare class SNHistoryManager extends AbstractService {
    private itemManager;
    private storageService;
    private apiService;
    private protocolService;
    deviceInterface: DeviceInterface;
    protected internalEventBus: InternalEventBusInterface;
    private removeChangeObserver;
    /**
     * When no history exists for an item yet, we first put it in the staging map.
     * Then, the next time the item changes and it has no history, we check the staging map.
     * If the entry from the staging map differs from the incoming change, we now add the incoming
     * change to the history map and remove it from staging. This is a way to detect when the first
     * actual change of an item occurs (especially new items), rather than tracking a change
     * as an item propagating through the different PayloadSource
     * lifecycles (created, local saved, presyncsave, etc)
     */
    private historyStaging;
    private history;
    private itemRevisionThreshold;
    constructor(itemManager: ItemManager, storageService: DiskStorageService, apiService: SNApiService, protocolService: EncryptionService, deviceInterface: DeviceInterface, internalEventBus: InternalEventBusInterface);
    deinit(): void;
    private recordNewHistoryForItems;
    sessionHistoryForItem(item: Models.SNNote): Models.HistoryEntry[];
    getHistoryMapCopy(): Models.HistoryMap;
    /**
     * Fetches a list of revisions from the server for an item. These revisions do not
     * include the item's content. Instead, each revision's content must be fetched
     * individually upon selection via `fetchRemoteRevision`.
     */
    remoteHistoryForItem(item: Models.SNNote): Promise<Responses.RevisionListEntry[] | undefined>;
    /**
     * Expands on a revision fetched via `remoteHistoryForItem` by getting a revision's
     * complete fields (including encrypted content).
     */
    fetchRemoteRevision(note: Models.SNNote, entry: Responses.RevisionListEntry): Promise<Models.HistoryEntry | undefined>;
    deleteRemoteRevision(note: SNNote, entry: Responses.RevisionListEntry): Promise<Responses.MinimalHttpResponse>;
    /**
     * Clean up if there are too many revisions. Note itemRevisionThreshold
     * is the amount of revisions which above, call for an optimization. An
     * optimization may not remove entries above this threshold. It will
     * determine what it should keep and what it shouldn't. So, it is possible
     * to have a threshold of 60 but have 600 entries, if the item history deems
     * those worth keeping.
     *
     * Rules:
     * - Keep an entry if it is the oldest entry
     * - Keep an entry if it is the latest entry
     * - Keep an entry if it is Significant
     * - If an entry is Significant and it is a deletion change, keep the entry before this entry.
     */
    optimizeHistoryForItem(uuid: string): void;
}
