"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SNComponent = exports.isComponentOrTheme = exports.isComponent = void 0;
const utils_1 = require("@standardnotes/utils");
const common_1 = require("@standardnotes/common");
const features_1 = require("@standardnotes/features");
const AppDataField_1 = require("../../Abstract/Item/Types/AppDataField");
const ConflictStrategy_1 = require("../../Abstract/Item/Types/ConflictStrategy");
const DecryptedItem_1 = require("../../Abstract/Item/Implementations/DecryptedItem");
const Predicate_1 = require("../../Runtime/Predicate/Predicate");
const isComponent = (x) => x.content_type === common_1.ContentType.Component;
exports.isComponent = isComponent;
const isComponentOrTheme = (x) => x.content_type === common_1.ContentType.Component || x.content_type === common_1.ContentType.Theme;
exports.isComponentOrTheme = isComponentOrTheme;
/**
 * Components are mostly iframe based extensions that communicate with the SN parent
 * via the postMessage API. However, a theme can also be a component, which is activated
 * only by its url.
 */
class SNComponent extends DecryptedItem_1.DecryptedItem {
    constructor(payload) {
        super(payload);
        this.permissions = [];
        /** Custom data that a component can store in itself */
        this.componentData = this.payload.content.componentData || {};
        if (payload.content.hosted_url && (0, utils_1.isValidUrl)(payload.content.hosted_url)) {
            this.hosted_url = payload.content.hosted_url;
        }
        else if (payload.content.url && (0, utils_1.isValidUrl)(payload.content.url)) {
            this.hosted_url = payload.content.url;
        }
        else if (payload.content.legacy_url && (0, utils_1.isValidUrl)(payload.content.legacy_url)) {
            this.hosted_url = payload.content.legacy_url;
        }
        this.local_url = payload.content.local_url;
        this.valid_until = new Date(payload.content.valid_until || 0);
        this.offlineOnly = payload.content.offlineOnly;
        this.name = payload.content.name;
        this.area = payload.content.area;
        this.package_info = payload.content.package_info || {};
        this.permissions = payload.content.permissions || [];
        this.active = payload.content.active;
        this.autoupdateDisabled = payload.content.autoupdateDisabled;
        this.disassociatedItemIds = payload.content.disassociatedItemIds || [];
        this.associatedItemIds = payload.content.associatedItemIds || [];
        this.isMobileDefault = payload.content.isMobileDefault;
        /**
         * @legacy
         * We don't want to set this.url directly, as we'd like to phase it out.
         * If the content.url exists, we'll transfer it to legacy_url. We'll only
         * need to set this if content.hosted_url is blank, otherwise,
         * hosted_url is the url replacement.
         */
        this.legacy_url = !payload.content.hosted_url ? payload.content.url : undefined;
    }
    /** Do not duplicate components under most circumstances. Always keep original */
    strategyWhenConflictingWithItem(_item, _previousRevision) {
        return ConflictStrategy_1.ConflictStrategy.KeepBase;
    }
    get isSingleton() {
        return true;
    }
    get displayName() {
        var _a;
        return ((_a = (0, features_1.FindNativeFeature)(this.identifier)) === null || _a === void 0 ? void 0 : _a.name) || this.name;
    }
    singletonPredicate() {
        const uniqueIdentifierPredicate = new Predicate_1.Predicate('identifier', '=', this.identifier);
        return uniqueIdentifierPredicate;
    }
    isEditor() {
        return this.area === features_1.ComponentArea.Editor;
    }
    isTheme() {
        return this.content_type === common_1.ContentType.Theme || this.area === features_1.ComponentArea.Themes;
    }
    isDefaultEditor() {
        return this.getAppDomainValue(AppDataField_1.AppDataField.DefaultEditor) === true;
    }
    getLastSize() {
        return this.getAppDomainValue(AppDataField_1.AppDataField.LastSize);
    }
    /**
     * The key used to look up data that this component may have saved to an item.
     * This data will be stored on the item using this key.
     */
    getClientDataKey() {
        if (this.legacy_url) {
            return this.legacy_url;
        }
        else {
            return this.uuid;
        }
    }
    hasValidHostedUrl() {
        return (this.hosted_url || this.legacy_url) != undefined;
    }
    contentKeysToIgnoreWhenCheckingEquality() {
        const componentKeys = ['active', 'disassociatedItemIds', 'associatedItemIds'];
        const superKeys = super.contentKeysToIgnoreWhenCheckingEquality();
        return [...componentKeys, ...superKeys];
    }
    /**
     * An associative component depends on being explicitly activated for a
     * given item, compared to a dissaciative component, which is enabled by
     * default in areas unrelated to a certain item.
     */
    static associativeAreas() {
        return [features_1.ComponentArea.Editor];
    }
    isAssociative() {
        return SNComponent.associativeAreas().includes(this.area);
    }
    isExplicitlyEnabledForItem(uuid) {
        return this.associatedItemIds.indexOf(uuid) !== -1;
    }
    isExplicitlyDisabledForItem(uuid) {
        return this.disassociatedItemIds.indexOf(uuid) !== -1;
    }
    get isExpired() {
        return this.valid_until.getTime() > 0 && this.valid_until <= new Date();
    }
    get identifier() {
        return this.package_info.identifier;
    }
    get thirdPartyPackageInfo() {
        return this.package_info;
    }
    get noteType() {
        return this.package_info.note_type || features_1.NoteType.Plain;
    }
    get isDeprecated() {
        var _a;
        let flags = (_a = this.package_info.flags) !== null && _a !== void 0 ? _a : [];
        flags = flags.map((flag) => flag.toLowerCase());
        return flags.includes(features_1.ComponentFlag.Deprecated);
    }
    get deprecationMessage() {
        return this.package_info.deprecation_message;
    }
}
exports.SNComponent = SNComponent;
