/**
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 */

'use strict';

var extension = require('@lexical/extension');
var ReactExtension = require('@lexical/react/ReactExtension');
var ReactProviderExtension = require('@lexical/react/ReactProviderExtension');
var lexical = require('lexical');
var react = require('react');
var jsxRuntime = require('react/jsx-runtime');

/**
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 */
/**
 * The equivalent of LexicalComposer for an extension. Make sure that your extension
 * argument is stable (e.g. using module scope or useMemo) so
 * that you are not re-creating the editor on every render!
 *
 * @example
 * Module scoped extension
 * ```tsx
 * const extension = defineExtension({
 *   name: "[root]",
 *   dependencies: [RichTextExtension, HistoryExtension, EmojiExtension]
 * });
 * function MyEditor({ children }) {
 *   return (<LexicalExtensionComposer extension={extension}>{children}</LexicalExtensionComposer>);
 * }
 * ```
 *
 * @example
 * useMemo extension
 * ```tsx
 * function MyEditor({ emojiBaseUrl, children }) {
 *   const extension = useMemo(() => {
 *     return defineExtension({
 *       name: "[root]",
 *       dependencies: [
 *         RichTextExtension,
 *         HistoryExtension,
 *         configExtension(EmojiExtension, { emojiBaseUrl }),
 *       ],
 *     });
 *   }, [emojiBaseUrl]);
 *   return (<LexicalExtensionComposer extension={extension}>{children}</LexicalExtensionComposer>);
 * }
 * ```
 *
 * @example
 * Incorrect usage with unstable extension
 * ```tsx
 * function MyBrokenEditor({ emojiBaseUrl }) {
 *   // This argument is not stable, the editor is re-created every render and
 *   // all state is lost!
 *   const extension = defineExtension({
 *     name: "[root]",
 *     dependencies: [RichTextExtension, HistoryExtension, EmojiExtension]
 *   });
 *   return (<LexicalExtensionComposer extension={extension}>{children}</LexicalExtensionComposer>);
 * }
 * ```
 */
function LexicalExtensionComposer({
  extension: extension$1,
  children,
  contentEditable
}) {
  const editor = react.useMemo(() => {
    const builder = extension.LexicalBuilder.fromExtensions([ReactProviderExtension.ReactProviderExtension, lexical.configExtension(ReactExtension.ReactExtension, contentEditable === undefined ? {} : {
      contentEditable
    }), extension$1]);
    return builder.buildEditor();
  }, [contentEditable, extension$1]);
  react.useEffect(() => {
    // Strict mode workaround
    let didMount = false;
    queueMicrotask(() => {
      didMount = true;
    });
    return () => {
      if (didMount) {
        editor.dispose();
      }
    };
  }, [editor]);
  const {
    Component
  } = extension.getExtensionDependencyFromEditor(editor, ReactExtension.ReactExtension).output;
  return /*#__PURE__*/jsxRuntime.jsx(Component, {
    children: children
  });
}

exports.LexicalExtensionComposer = LexicalExtensionComposer;
