/**
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 */
import type { EditorState, LexicalEditor } from 'lexical';
import { ReadonlySignal } from '@lexical/extension';
export type HistoryStateEntry = {
    editor: LexicalEditor;
    editorState: EditorState;
};
export type HistoryState = {
    current: null | HistoryStateEntry;
    redoStack: Array<HistoryStateEntry>;
    undoStack: Array<HistoryStateEntry>;
};
/**
 * Registers necessary listeners to manage undo/redo history stack and related editor commands.
 * It returns `unregister` callback that cleans up all listeners and should be called on editor unmount.
 * @param editor - The lexical editor.
 * @param historyState - The history state, containing the current state and the undo/redo stack.
 * @param delay - The time (in milliseconds) the editor should delay generating a new history stack,
 * instead of merging the current changes with the current stack.
 * @returns The listeners cleanup callback function.
 */
export declare function registerHistory(editor: LexicalEditor, historyState: HistoryState, delay: number | ReadonlySignal<number>): () => void;
/**
 * Creates an empty history state.
 * @returns - The empty history state, as an object.
 */
export declare function createEmptyHistoryState(): HistoryState;
export interface HistoryConfig {
    /**
     * The time (in milliseconds) the editor should delay generating a new history stack,
     * instead of merging the current changes with the current stack. The default is 300ms.
     */
    delay: number;
    /**
     * The initial history state, the default is {@link createEmptyHistoryState}.
     */
    createInitialHistoryState: (editor: LexicalEditor) => HistoryState;
    /**
     * Whether history is disabled or not
     */
    disabled: boolean;
}
/**
 * Registers necessary listeners to manage undo/redo history stack and related
 * editor commands, via the \@lexical/history module.
 */
export declare const HistoryExtension: import("lexical").LexicalExtension<HistoryConfig, "@lexical/history/History", import("@lexical/extension").NamedSignalsOutput<{
    delay: number;
    disabled: boolean;
    historyState: HistoryState;
}>, unknown>;
/**
 * Registers necessary listeners to manage undo/redo history stack and related
 * editor commands, via the \@lexical/history module, only if the parent editor
 * has a history plugin implementation.
 */
export declare const SharedHistoryExtension: import("lexical").LexicalExtension<import("lexical").ExtensionConfigBase, "@lexical/history/SharedHistory", unknown, unknown>;
