/**
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 */
import { type AnyLexicalExtension, type AnyLexicalExtensionArgument, type AnyNormalizedLexicalExtensionArgument, type CreateEditorArgs, type InitialEditorConfig, type LexicalEditor, type LexicalEditorWithDispose, type LexicalExtensionConfig } from 'lexical';
import { ExtensionRep } from './ExtensionRep';
/** @internal Use a well-known symbol for dev tools purposes */
export declare const builderSymbol: unique symbol;
type BuildCreateEditorArgs = Omit<CreateEditorArgs, 'onError'> & Pick<InitialEditorConfig, 'onError' | '$initialEditorState'>;
/**
 * Build a LexicalEditor by combining together one or more extensions, optionally
 * overriding some of their configuration.
 *
 * @param extensions - Extension arguments (extensions or extensions with config overrides)
 * @returns An editor handle
 *
 * @example
 * A single root extension with multiple dependencies
 *
 * ```ts
 * const editor = buildEditorFromExtensions(
 *   defineExtension({
 *     name: "[root]",
 *     dependencies: [
 *       RichTextExtension,
 *       configExtension(EmojiExtension, { emojiBaseUrl: "/assets/emoji" }),
 *     ],
 *     register: (editor: LexicalEditor) => {
 *       console.log("Editor Created");
 *       return () => console.log("Editor Disposed");
 *     },
 *   }),
 * );
 * ```
 *
 * @example
 * A very similar minimal configuration without the register hook
 *
 * ```ts
 * const editor = buildEditorFromExtensions(
 *   RichTextExtension,
 *   configExtension(EmojiExtension, { emojiBaseUrl: "/assets/emoji" }),
 * );
 * ```
 */
export declare function buildEditorFromExtensions(...extensions: AnyLexicalExtensionArgument[]): LexicalEditorWithDispose;
interface WithBuilder {
    [builderSymbol]?: LexicalBuilder | undefined;
}
/** @internal */
export declare class LexicalBuilder {
    roots: readonly AnyNormalizedLexicalExtensionArgument[];
    extensionNameMap: Map<string, ExtensionRep<AnyLexicalExtension>>;
    outgoingConfigEdges: Map<string, Map<string, LexicalExtensionConfig<AnyLexicalExtension>[]>>;
    incomingEdges: Map<string, Set<string>>;
    conflicts: Map<string, string>;
    _sortedExtensionReps?: readonly ExtensionRep<AnyLexicalExtension>[];
    PACKAGE_VERSION: string;
    constructor(roots: AnyNormalizedLexicalExtensionArgument[]);
    static fromExtensions(extensions: AnyLexicalExtensionArgument[]): LexicalBuilder;
    static maybeFromEditor(editor: LexicalEditor): undefined | LexicalBuilder;
    /** Look up the editor that was created by this LexicalBuilder or throw */
    static fromEditor(editor: LexicalEditor): LexicalBuilder;
    constructEditor(): LexicalEditor & WithBuilder;
    buildEditor(): LexicalEditorWithDispose;
    hasExtensionByName(name: string): boolean;
    getExtensionRep<Extension extends AnyLexicalExtension>(extension: Extension): ExtensionRep<Extension> | undefined;
    addEdge(fromExtensionName: string, toExtensionName: string, configs: LexicalExtensionConfig<AnyLexicalExtension>[]): void;
    addExtension(arg: AnyLexicalExtensionArgument): void;
    sortedExtensionReps(): readonly ExtensionRep<AnyLexicalExtension>[];
    registerEditor(editor: LexicalEditor): () => void;
    buildCreateEditorArgs(): BuildCreateEditorArgs;
}
export {};
