"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VersionCommand = exports.factory = void 0;
const tslib_1 = require("tslib");
const chalk_1 = tslib_1.__importDefault(require("chalk"));
const dedent_1 = tslib_1.__importDefault(require("dedent"));
const minimatch_1 = tslib_1.__importDefault(require("minimatch"));
const os_1 = tslib_1.__importDefault(require("os"));
const p_map_1 = tslib_1.__importDefault(require("p-map"));
const p_pipe_1 = tslib_1.__importDefault(require("p-pipe"));
const p_reduce_1 = tslib_1.__importDefault(require("p-reduce"));
const semver_1 = tslib_1.__importDefault(require("semver"));
const core_1 = require("@lerna-lite/core");
const get_current_branch_1 = require("./lib/get-current-branch");
const create_release_1 = require("./lib/create-release");
const is_anything_committed_1 = require("./lib/is-anything-committed");
const remote_branch_exists_1 = require("./lib/remote-branch-exists");
const is_behind_upstream_1 = require("./lib/is-behind-upstream");
const is_breaking_change_1 = require("./lib/is-breaking-change");
const git_add_1 = require("./lib/git-add");
const git_commit_1 = require("./lib/git-commit");
const git_tag_1 = require("./lib/git-tag");
const git_push_1 = require("./lib/git-push");
const prompt_version_1 = require("./lib/prompt-version");
const update_lockfile_version_1 = require("./lib/update-lockfile-version");
const conventional_commits_1 = require("./conventional-commits");
function factory(argv) {
    return new VersionCommand(argv);
}
exports.factory = factory;
class VersionCommand extends core_1.Command {
    get otherCommandConfigs() {
        // back-compat
        return ['publish'];
    }
    get requiresGit() {
        // prettier-ignore
        return (this.commitAndTag ||
            this.pushToRemote ||
            this.options.allowBranch ||
            this.options.conventionalCommits);
    }
    constructor(argv) {
        super(argv);
        /** command name */
        this.name = 'version';
        this.globalVersion = '';
        this.packagesToVersion = [];
        this.currentBranch = '';
        this.gitRemote = '';
        this.tagPrefix = '';
        this.commitAndTag = true;
        this.pushToRemote = true;
        this.hasRootedLeaf = false;
        this.releaseNotes = [];
        this.savePrefix = '';
        this.tags = [];
        this.updates = [];
    }
    configureProperties() {
        super.configureProperties();
        // Defaults are necessary here because yargs defaults
        // override durable options provided by a config file
        const { amend, changelogIncludeCommitsClientLogin, changelogIncludeCommitsGitAuthor, commitHooks = true, gitRemote = 'origin', gitTagVersion = true, granularPathspec = true, push = true, signGitCommit, signoffGitCommit, signGitTag, forceGitTag, tagVersionPrefix = 'v', } = this.options;
        this.gitRemote = gitRemote;
        this.tagPrefix = tagVersionPrefix;
        this.commitAndTag = gitTagVersion;
        this.pushToRemote = gitTagVersion && amend !== true && push;
        this.changelogIncludeCommitsClientLogin =
            changelogIncludeCommitsClientLogin === '' ? true : changelogIncludeCommitsClientLogin;
        this.changelogIncludeCommitsGitAuthor =
            changelogIncludeCommitsGitAuthor === '' ? true : changelogIncludeCommitsGitAuthor;
        // never automatically push to remote when amending a commit
        // prettier-ignore
        this.releaseClient = (this.pushToRemote &&
            this.options.createRelease &&
            (0, create_release_1.createReleaseClient)(this.options.createRelease));
        this.releaseNotes = [];
        if (this.releaseClient && this.options.conventionalCommits !== true) {
            throw new core_1.ValidationError('ERELEASE', 'To create a release, you must enable --conventional-commits');
        }
        if (this.releaseClient && this.options.changelog === false) {
            throw new core_1.ValidationError('ERELEASE', 'To create a release, you cannot pass --no-changelog');
        }
        this.gitOpts = {
            amend,
            commitHooks,
            granularPathspec,
            signGitCommit,
            signoffGitCommit,
            signGitTag,
            forceGitTag,
        };
        // https://docs.npmjs.com/misc/config#save-prefix
        this.savePrefix = this.options.exact ? '' : '^';
    }
    async initialize() {
        var _a;
        const isIndependent = this.project.isIndependent();
        const describeTag = this.project.config.describeTag;
        if (!isIndependent) {
            this.logger.info('current project version', (_a = this.project.version) !== null && _a !== void 0 ? _a : '');
        }
        if (this.requiresGit) {
            // git validation, if enabled, should happen before updates are calculated and versions picked
            if (!(0, is_anything_committed_1.isAnythingCommitted)(this.execOpts, this.options.dryRun)) {
                throw new core_1.ValidationError('ENOCOMMIT', 'No commits in this repository. Please commit something before using version.');
            }
            this.currentBranch = (0, get_current_branch_1.getCurrentBranch)(this.execOpts, false);
            if (this.currentBranch === 'HEAD') {
                throw new core_1.ValidationError('ENOGIT', 'Detached git HEAD, please checkout a branch to choose versions.');
            }
            if (this.pushToRemote &&
                !(0, remote_branch_exists_1.remoteBranchExists)(this.gitRemote, this.currentBranch, this.execOpts, this.options.dryRun)) {
                throw new core_1.ValidationError('ENOREMOTEBRANCH', (0, dedent_1.default) `
            Branch "${this.currentBranch}" doesn't exist in remote "${this.gitRemote}".
            If this is a new branch, please make sure you push it to the remote first.
          `);
            }
            if (this.options.allowBranch &&
                ![].concat(this.options.allowBranch).some((x) => (0, minimatch_1.default)(this.currentBranch, x))) {
                throw new core_1.ValidationError('ENOTALLOWED', (0, dedent_1.default) `
            Branch "${this.currentBranch}" is restricted from versioning due to allowBranch config.
            Please consider the reasons for this restriction before overriding the option.
          `);
            }
            if (this.commitAndTag &&
                this.pushToRemote &&
                (0, is_behind_upstream_1.isBehindUpstream)(this.gitRemote, this.currentBranch, this.execOpts, this.options.dryRun)) {
                // eslint-disable-next-line max-len
                const message = `Local branch "${this.currentBranch}" is behind remote upstream ${this.gitRemote}/${this.currentBranch}`;
                if (!this.options.ci) {
                    // interrupt interactive execution
                    throw new core_1.ValidationError('EBEHIND', (0, dedent_1.default) `
              ${message}
              Please merge remote changes into "${this.currentBranch}" with "git pull"
            `);
                }
                // CI execution should not error, but warn & exit
                this.logger.warn('EBEHIND', `${message}, exiting`);
                // still exits zero, aka "ok"
                return false;
            }
        }
        else {
            this.logger.notice('FYI', 'git repository validation has been skipped, please ensure your version bumps are correct');
        }
        if (this.options.conventionalPrerelease && this.options.conventionalGraduate) {
            throw new core_1.ValidationError('ENOTALLOWED', (0, dedent_1.default) `
          --conventional-prerelease cannot be combined with --conventional-graduate.
        `);
        }
        if (this.options.manuallyUpdateRootLockfile && this.options.syncWorkspaceLock) {
            throw new core_1.ValidationError('ENOTALLOWED', (0, dedent_1.default) `
          --manually-update-root-lockfile cannot be combined with --sync-workspace-lock.
        `);
        }
        if (this.changelogIncludeCommitsClientLogin && this.changelogIncludeCommitsGitAuthor) {
            throw new core_1.ValidationError('ENOTALLOWED', (0, dedent_1.default) `
          --changelog-include-commits-client-login cannot be combined with --changelog-include-commits-git-author.
        `);
        }
        if (this.options.changelogIncludeCommitAuthorFullname) {
            this.logger.warn('deprecated', '--changelog-include-commit-author-fullname has been renamed to --changelog-include-commits-git-author.');
        }
        // fetch all commits from remote server of the last release when user wants to include client login associated to each commits
        const remoteClient = this.options.createRelease || this.options.remoteClient;
        if (this.options.conventionalCommits && this.changelogIncludeCommitsClientLogin) {
            if (!remoteClient) {
                throw new core_1.ValidationError('EMISSINGCLIENT', (0, dedent_1.default) `
            --changelog-include-commits-client-login requires one of these two option --remote-client or --create-release to be defined.
          `);
            }
            this.commitsSinceLastRelease = await (0, conventional_commits_1.getCommitsSinceLastRelease)(remoteClient, this.options.gitRemote, this.currentBranch, isIndependent, this.execOpts);
        }
        this.updates = (0, core_1.collectUpdates)(this.packageGraph.rawPackageList, this.packageGraph, this.execOpts, {
            ...this.options,
            isIndependent,
            describeTag,
        }).filter((node) => {
            // --no-private completely removes private packages from consideration
            if (node.pkg.private && this.options.private === false) {
                // TODO: (major) make --no-private the default
                return false;
            }
            if (!node.version) {
                // a package may be unversioned only if it is private
                if (node.pkg.private) {
                    this.logger.info('version', 'Skipping unversioned private package %j', node.name);
                }
                else {
                    throw new core_1.ValidationError('ENOVERSION', (0, dedent_1.default) `
              A version field is required in ${node.name}'s package.json file.
              If you wish to keep the package unversioned, it must be made private.
            `);
                }
            }
            return !!node.version;
        });
        if (!this.updates.length) {
            this.logger.success(`No changed packages to ${this.composed ? 'publish' : 'version'}`);
            // still exits zero, aka "ok"
            return false;
        }
        // a "rooted leaf" is the regrettable pattern of adding "." to the "packages" config in lerna.json
        this.hasRootedLeaf = this.packageGraph.has(this.project.manifest.name);
        if (this.hasRootedLeaf && !this.composed) {
            this.logger.info('version', 'rooted leaf detected, skipping synthetic root lifecycles');
        }
        this.runPackageLifecycle = (0, core_1.createRunner)({ ...this.options, stdio: 'inherit' });
        // don't execute recursively if run from a poorly-named script
        this.runRootLifecycle = /^(pre|post)?version$/.test(process.env.npm_lifecycle_event)
            ? (stage) => this.logger.warn('lifecycle', 'Skipping root %j because it has already been called', stage)
            : (stage) => this.runPackageLifecycle(this.project.manifest, stage);
        // amending a commit probably means the working tree is dirty
        if (this.commitAndTag && this.gitOpts.amend !== true) {
            const { forcePublish, conventionalCommits, conventionalGraduate } = this.options;
            const checkUncommittedOnly = forcePublish || (conventionalCommits && conventionalGraduate);
            const check = checkUncommittedOnly ? core_1.throwIfUncommitted : core_1.checkWorkingTree;
            await check(this.execOpts, this.options.dryRun);
        }
        else {
            this.logger.warn('version', 'Skipping working tree validation, proceed at your own risk');
        }
        const versionsForUpdate = await this.getVersionsForUpdates();
        await this.setUpdatesForVersions(versionsForUpdate);
        return this.confirmVersions();
    }
    async execute() {
        await this.updatePackageVersions();
        if (this.commitAndTag) {
            await this.commitAndTagUpdates();
        }
        else {
            this.logger.info('execute', 'Skipping git tag/commit');
        }
        if (this.pushToRemote) {
            await this.gitPushToRemote();
        }
        else {
            this.logger.info('execute', 'Skipping git push');
        }
        if (this.releaseClient) {
            this.logger.info('execute', 'Creating releases...');
            await (0, create_release_1.createRelease)(this.releaseClient, { tags: this.tags, releaseNotes: this.releaseNotes }, { gitRemote: this.options.gitRemote, execOpts: this.execOpts }, this.options.dryRun);
        }
        else {
            this.logger.info('execute', 'Skipping releases');
        }
        if (!this.composed) {
            this.logger.success('version', 'finished');
        }
        return {
            updates: this.updates,
            updatesVersions: this.updatesVersions,
        };
    }
    getVersionsForUpdates() {
        const independentVersions = this.project.isIndependent();
        const { bump, conventionalCommits, preid } = this.options;
        const repoVersion = (bump ? semver_1.default.clean(bump) : '');
        const increment = (bump && !semver_1.default.valid(bump) ? bump : '');
        const resolvePrereleaseId = (existingPreid) => preid || existingPreid || 'alpha';
        const makeGlobalVersionPredicate = (nextVersion) => {
            this.globalVersion = nextVersion;
            return () => nextVersion;
        };
        // decide the predicate in the conditionals below
        let predicate;
        if (repoVersion) {
            predicate = makeGlobalVersionPredicate((0, conventional_commits_1.applyBuildMetadata)(repoVersion, this.options.buildMetadata));
        }
        else if (increment && independentVersions) {
            // compute potential prerelease ID for each independent update
            predicate = (node) => (0, conventional_commits_1.applyBuildMetadata)(semver_1.default.inc(node.version, increment, resolvePrereleaseId(node.prereleaseId)), this.options.buildMetadata);
        }
        else if (increment) {
            // compute potential prerelease ID once for all fixed updates
            const prereleaseId = prereleaseIdFromVersion(this.project.version);
            const nextVersion = (0, conventional_commits_1.applyBuildMetadata)(semver_1.default.inc(this.project.version, increment, resolvePrereleaseId(prereleaseId)), this.options.buildMetadata);
            predicate = makeGlobalVersionPredicate(nextVersion);
        }
        else if (conventionalCommits) {
            // it's a bit weird to have a return here, true
            return this.recommendVersions(resolvePrereleaseId);
        }
        else if (independentVersions) {
            // prompt for each independent update with potential prerelease ID
            predicate = (0, prompt_version_1.makePromptVersion)(resolvePrereleaseId, this.options.buildMetadata);
        }
        else {
            // prompt once with potential prerelease ID
            const prereleaseId = prereleaseIdFromVersion(this.project.version);
            const node = { version: this.project.version, prereleaseId };
            predicate = (0, prompt_version_1.makePromptVersion)(resolvePrereleaseId, this.options.buildMetadata);
            predicate = predicate(node).then(makeGlobalVersionPredicate);
        }
        return Promise.resolve(predicate).then((getVersion) => this.reduceVersions(getVersion));
    }
    reduceVersions(getVersion) {
        const iterator = (versionMap, node) => Promise.resolve(getVersion(node)).then((version) => versionMap.set(node.name, version));
        return (0, p_reduce_1.default)(this.updates, iterator, new Map());
    }
    setUpdatesForVersions(versions) {
        var _a, _b, _c;
        if (this.project.isIndependent()) {
            // only partial fixed versions need to be checked
            this.updatesVersions = versions;
        }
        else {
            let hasBreakingChange = false;
            for (const [name, bump] of versions) {
                hasBreakingChange = hasBreakingChange || (0, is_breaking_change_1.isBreakingChange)((_a = this.packageGraph) === null || _a === void 0 ? void 0 : _a.get(name).version, bump);
            }
            if (hasBreakingChange) {
                // _all_ packages need a major version bump whenever _any_ package does
                this.updates = Array.from((_c = (_b = this.packageGraph) === null || _b === void 0 ? void 0 : _b.values()) !== null && _c !== void 0 ? _c : []);
                // --no-private completely removes private packages from consideration
                if (this.options.private === false) {
                    // TODO: (major) make --no-private the default
                    this.updates = this.updates.filter((node) => !node.pkg.private);
                }
                this.updatesVersions = new Map(this.updates.map((node) => [node.name, this.globalVersion]));
            }
            else {
                this.updatesVersions = versions;
            }
        }
        this.packagesToVersion = this.updates.map((node) => node.pkg);
    }
    getPrereleasePackageNames() {
        const prereleasePackageNames = this.getPackagesForOption(this.options.conventionalPrerelease);
        const isCandidate = prereleasePackageNames.has('*')
            ? () => true
            : (_node, name) => prereleasePackageNames.has(name);
        return (0, core_1.collectPackages)(this.packageGraph, { isCandidate }).map((pkg) => { var _a; return (_a = pkg === null || pkg === void 0 ? void 0 : pkg.name) !== null && _a !== void 0 ? _a : ''; });
    }
    /**
     * @param {boolean|string|string[]} option
     * @returns {Set<string>} A set of package names (or wildcard) derived from option value.
     */
    getPackagesForOption(option) {
        // new Set(null) is equivalent to new Set([])
        // i.e., an empty Set
        let inputs = null;
        if (option === true) {
            // option passed without specific packages, eg. --force-publish
            inputs = ['*'];
        }
        else if (typeof option === 'string') {
            // option passed with one or more comma separated package names, eg.:
            // --force-publish=*
            // --force-publish=foo
            // --force-publish=foo,bar
            inputs = option.split(',');
        }
        else if (Array.isArray(option)) {
            // option passed multiple times with individual package names
            // --force-publish foo --force-publish baz
            inputs = [...option];
        }
        return new Set(inputs);
    }
    async recommendVersions(resolvePrereleaseId) {
        var _a;
        const independentVersions = this.project.isIndependent();
        const { buildMetadata, changelogPreset, conventionalGraduate, conventionalBumpPrerelease } = this.options;
        const rootPath = this.project.manifest.location;
        const type = independentVersions ? 'independent' : 'fixed';
        const prereleasePackageNames = this.getPrereleasePackageNames();
        const graduatePackageNames = Array.from(this.getPackagesForOption(conventionalGraduate));
        const shouldPrerelease = (name) => prereleasePackageNames === null || prereleasePackageNames === void 0 ? void 0 : prereleasePackageNames.includes(name);
        const shouldGraduate = (name) => graduatePackageNames.includes('*') || graduatePackageNames.includes(name);
        const getPrereleaseId = (node) => {
            if (!shouldGraduate(node.name) && (shouldPrerelease(node.name) || node.prereleaseId)) {
                return resolvePrereleaseId(node.prereleaseId);
            }
        };
        if (type === 'fixed') {
            this.setGlobalVersionFloor();
        }
        const versions = await this.reduceVersions((node) => (0, conventional_commits_1.recommendVersion)(node, type, {
            changelogPreset,
            rootPath,
            tagPrefix: this.tagPrefix,
            prereleaseId: getPrereleaseId(node),
            conventionalBumpPrerelease,
            buildMetadata,
        }));
        if (type === 'fixed') {
            this.globalVersion = (_a = this.setGlobalVersionCeiling(versions)) !== null && _a !== void 0 ? _a : '';
        }
        return versions;
    }
    confirmVersions() {
        const changes = this.packagesToVersion.map((pkg) => {
            var _a, _b, _c;
            let line = ` - ${(_a = pkg.name) !== null && _a !== void 0 ? _a : '[n/a]'}: ${pkg.version} => ${chalk_1.default.cyan((_b = this.updatesVersions) === null || _b === void 0 ? void 0 : _b.get((_c = pkg === null || pkg === void 0 ? void 0 : pkg.name) !== null && _c !== void 0 ? _c : ''))}`;
            if (pkg.private) {
                line += ` (${chalk_1.default.red('private')})`;
            }
            return line;
        });
        (0, core_1.logOutput)('');
        (0, core_1.logOutput)(`Changes (${changes.length} packages):`);
        (0, core_1.logOutput)(changes.join(core_1.EOL));
        (0, core_1.logOutput)('');
        if (this.options.yes) {
            this.logger.info('auto-confirmed', '');
            return true;
        }
        // When composed from `lerna publish`, use this opportunity to confirm publishing
        let confirmMessage = this.options.dryRun ? chalk_1.default.bgMagenta('[dry-run]') : '';
        confirmMessage += this.composed
            ? ' Are you sure you want to publish these packages?'
            : ' Are you sure you want to create these versions?';
        return (0, core_1.promptConfirmation)(confirmMessage.trim());
    }
    updatePackageVersions() {
        const { conventionalCommits, changelogPreset, changelogHeaderMessage, changelogVersionMessage, changelog = true, } = this.options;
        const independentVersions = this.project.isIndependent();
        const rootPath = this.project.manifest.location;
        const changedFiles = new Set();
        const npmClient = this.options.npmClient || 'npm';
        let chain = Promise.resolve();
        // preversion:  Run BEFORE bumping the package version.
        // version:     Run AFTER bumping the package version, but BEFORE commit.
        // postversion: Run AFTER bumping the package version, and AFTER commit.
        // @see https://docs.npmjs.com/misc/scripts
        if (!this.hasRootedLeaf) {
            // exec preversion lifecycle in root (before all updates)
            chain = chain.then(() => this.runRootLifecycle('preversion'));
        }
        const actions = [
            (pkg) => this.runPackageLifecycle(pkg, 'preversion').then(() => pkg),
            // manifest may be mutated by any previous lifecycle
            (pkg) => pkg.refresh(),
            (pkg) => {
                var _a, _b, _c, _d;
                // set new version
                pkg.set('version', (_a = this.updatesVersions) === null || _a === void 0 ? void 0 : _a.get((_b = pkg === null || pkg === void 0 ? void 0 : pkg.name) !== null && _b !== void 0 ? _b : ''));
                // update pkg dependencies
                for (const [depName, resolved] of (_c = this.packageGraph) === null || _c === void 0 ? void 0 : _c.get(pkg.name).localDependencies) {
                    const depVersion = (_d = this.updatesVersions) === null || _d === void 0 ? void 0 : _d.get(depName);
                    if (depVersion && resolved.type !== 'directory') {
                        // don't overwrite local file: specifiers, they only change during publish
                        pkg.updateLocalDependency(resolved, depVersion, this.savePrefix, this.options.allowPeerDependenciesUpdate, this.options.workspaceStrictMatch, this.commandName);
                    }
                }
                return Promise.all([(0, update_lockfile_version_1.updateClassicLockfileVersion)(pkg), pkg.serialize()]).then(([lockfilePath]) => {
                    // commit the updated manifest
                    changedFiles.add(pkg.manifestLocation);
                    if (lockfilePath) {
                        changedFiles.add(lockfilePath);
                    }
                    return pkg;
                });
            },
            (pkg) => this.runPackageLifecycle(pkg, 'version').then(() => pkg),
        ];
        if (conventionalCommits && changelog) {
            // we can now generate the Changelog, based on the
            // the updated version that we're about to release.
            const type = independentVersions ? 'independent' : 'fixed';
            actions.push((pkg) => (0, conventional_commits_1.updateChangelog)(pkg, type, {
                changelogPreset,
                rootPath,
                tagPrefix: this.tagPrefix,
                changelogIncludeCommitsGitAuthor: this.changelogIncludeCommitsGitAuthor,
                changelogIncludeCommitsClientLogin: this.changelogIncludeCommitsClientLogin,
                changelogHeaderMessage,
                changelogVersionMessage,
                commitsSinceLastRelease: this.commitsSinceLastRelease,
            }).then(({ logPath, newEntry }) => {
                // commit the updated changelog
                changedFiles.add(logPath);
                // add release notes
                if (independentVersions) {
                    this.releaseNotes.push({
                        name: pkg.name,
                        notes: newEntry,
                    });
                }
                return pkg;
            }));
        }
        const mapUpdate = (0, p_pipe_1.default)(...actions);
        chain = chain.then(() => (0, core_1.runTopologically)(this.packagesToVersion, mapUpdate, {
            concurrency: this.concurrency,
            rejectCycles: this.options.rejectCycles,
            graphType: this.options.allowPeerDependenciesUpdate ? 'allPlusPeerDependencies' : 'allDependencies',
        }));
        // update the project root npm lock file, we will read and write back to the lock file
        // this is currently the default update and if none of the flag are enabled (or all undefined) then we'll consider this as enabled
        if (npmClient === 'npm' &&
            (this.options.manuallyUpdateRootLockfile ||
                (this.options.manuallyUpdateRootLockfile === undefined && !this.options.syncWorkspaceLock))) {
            this.logger.warn('npm', 'we recommend using --sync-workspace-lock which will sync your lock file via your favorite npm client instead of relying on Lerna-Lite itself to update it.');
            chain = chain.then(() => 
            // update modern lockfile (version 2 or higher) when exist in the project root
            (0, update_lockfile_version_1.loadPackageLockFileWhenExists)(rootPath).then((lockFileResponse) => {
                if (lockFileResponse && lockFileResponse.lockfileVersion >= 2) {
                    this.logger.verbose(`lock`, `start process loop of manually updating npm lock file`);
                    for (const pkg of this.packagesToVersion) {
                        this.logger.verbose(`lock`, `updating root "package-lock-json" for package "${pkg.name}"`);
                        (0, update_lockfile_version_1.updateTempModernLockfileVersion)(pkg, lockFileResponse.json);
                    }
                    // save the lockfile, only once, after all package versions were updated
                    return (0, update_lockfile_version_1.saveUpdatedLockJsonFile)(lockFileResponse.path, lockFileResponse.json).then((lockfilePath) => {
                        if (lockfilePath) {
                            changedFiles.add(lockfilePath);
                        }
                    });
                }
            }));
        }
        else if (this.options.syncWorkspaceLock) {
            // update lock file, with npm client defined when `--sync-workspace-lock` is enabled
            chain = chain.then(() => (0, update_lockfile_version_1.runInstallLockFileOnly)(npmClient, this.project.manifest.location, this.options.npmClientArgs || []).then((lockfilePath) => {
                if (lockfilePath) {
                    changedFiles.add(lockfilePath);
                }
            }));
        }
        if (!independentVersions) {
            this.project.version = this.globalVersion;
            if (conventionalCommits && changelog) {
                chain = chain.then(() => (0, conventional_commits_1.updateChangelog)(this.project.manifest, 'root', {
                    changelogPreset,
                    rootPath,
                    tagPrefix: this.tagPrefix,
                    version: this.globalVersion,
                    changelogIncludeCommitsGitAuthor: this.changelogIncludeCommitsGitAuthor,
                    changelogIncludeCommitsClientLogin: this.changelogIncludeCommitsClientLogin,
                    changelogHeaderMessage,
                    changelogVersionMessage,
                    commitsSinceLastRelease: this.commitsSinceLastRelease,
                }).then(({ logPath, newEntry }) => {
                    // commit the updated changelog
                    changedFiles.add(logPath);
                    // add release notes
                    this.releaseNotes.push({
                        name: 'fixed',
                        notes: newEntry,
                    });
                }));
            }
            chain = chain.then(() => this.project.serializeConfig().then((lernaConfigLocation) => {
                // commit the version update
                changedFiles.add(lernaConfigLocation);
            }));
        }
        if (!this.hasRootedLeaf) {
            // exec version lifecycle in root (after all updates)
            chain = chain.then(() => this.runRootLifecycle('version'));
        }
        if (this.commitAndTag) {
            chain = chain.then(() => (0, git_add_1.gitAdd)(Array.from(changedFiles), this.gitOpts, this.execOpts, this.options.dryRun));
        }
        return chain;
    }
    async commitAndTagUpdates() {
        if (this.project.isIndependent()) {
            this.tags = await this.gitCommitAndTagVersionForUpdates();
        }
        else {
            this.tags = await this.gitCommitAndTagVersion();
        }
        // run the postversion script for each update
        await (0, p_map_1.default)(this.packagesToVersion, (pkg) => this.runPackageLifecycle(pkg, 'postversion'));
        if (!this.hasRootedLeaf) {
            // run postversion, if set, in the root directory
            await this.runRootLifecycle('postversion');
        }
    }
    async gitCommitAndTagVersionForUpdates() {
        const tags = this.packagesToVersion.map((pkg) => { var _a; return `${pkg.name}@${(_a = this.updatesVersions) === null || _a === void 0 ? void 0 : _a.get(pkg.name)}`; });
        const subject = this.options.message || 'chore: Publish new release';
        const message = tags.reduce((msg, tag) => `${msg}${os_1.default.EOL} - ${tag}`, `${subject}${os_1.default.EOL}`);
        await (0, git_commit_1.gitCommit)(message, this.gitOpts, this.execOpts, this.options.dryRun);
        for (const tag of tags) {
            await (0, git_tag_1.gitTag)(tag, this.gitOpts, this.execOpts, this.options.gitTagCommand, this.options.dryRun);
        }
        return tags;
    }
    async gitCommitAndTagVersion() {
        const version = this.globalVersion;
        const tag = `${this.tagPrefix}${version}`;
        const message = this.options.message ? this.options.message.replace(/%s/g, tag).replace(/%v/g, version) : tag;
        await (0, git_commit_1.gitCommit)(message, this.gitOpts, this.execOpts, this.options.dryRun);
        await (0, git_tag_1.gitTag)(tag, this.gitOpts, this.execOpts, this.options.gitTagCommand, this.options.dryRun);
        return [tag];
    }
    gitPushToRemote() {
        this.logger.info('git', 'Pushing tags...');
        return (0, git_push_1.gitPush)(this.gitRemote, this.currentBranch, this.execOpts, this.options.dryRun);
    }
    setGlobalVersionFloor() {
        const globalVersion = this.project.version;
        for (const node of this.updates) {
            if (semver_1.default.lt(node.version, globalVersion)) {
                this.logger.verbose('version', `Overriding version of ${node.name} from ${node.version} to ${globalVersion}`);
                node.pkg.version = globalVersion;
            }
        }
    }
    setGlobalVersionCeiling(versions) {
        let highestVersion = this.project.version;
        versions.forEach((bump) => {
            if (bump && semver_1.default.gt(bump, highestVersion)) {
                highestVersion = bump;
            }
        });
        versions.forEach((_, name) => versions.set(name, highestVersion));
        return highestVersion;
    }
}
exports.VersionCommand = VersionCommand;
/**
 * @param {string} version
 * @returns {string|undefined}
 */
function prereleaseIdFromVersion(version) {
    return (semver_1.default.prerelease(version) || []).shift();
}
//# sourceMappingURL=version-command.js.map