"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateFileExists = exports.runInstallLockFileOnly = exports.updateNpmLockFileVersion2 = exports.saveUpdatedLockJsonFile = exports.updateTempModernLockfileVersion = exports.updateClassicLockfileVersion = exports.loadPackageLockFileWhenExists = void 0;
const tslib_1 = require("tslib");
const npmlog_1 = tslib_1.__importDefault(require("npmlog"));
const path_1 = tslib_1.__importDefault(require("path"));
const load_json_file_1 = tslib_1.__importDefault(require("load-json-file"));
const fs_1 = tslib_1.__importDefault(require("fs"));
const os_1 = tslib_1.__importDefault(require("os"));
const semver_1 = tslib_1.__importDefault(require("semver"));
const write_json_file_1 = tslib_1.__importDefault(require("write-json-file"));
const core_1 = require("@lerna-lite/core");
/**
 * From a folder path provided, try to load a `package-lock.json` file if it exists.
 * @param {String} lockFileFolderPath
 * @returns Promise<{path: string; json: Object; lockFileVersion: number; }>
 */
async function loadPackageLockFileWhenExists(lockFileFolderPath) {
    var _a;
    try {
        const lockFilePath = path_1.default.join(lockFileFolderPath, 'package-lock.json');
        const pkgLockFileObj = await (0, load_json_file_1.default)(lockFilePath);
        const lockfileVersion = +((_a = pkgLockFileObj === null || pkgLockFileObj === void 0 ? void 0 : pkgLockFileObj['lockfileVersion']) !== null && _a !== void 0 ? _a : 1);
        return {
            path: lockFilePath,
            json: pkgLockFileObj,
            lockfileVersion,
        };
    }
    catch (error) { } // eslint-disable-line
}
exports.loadPackageLockFileWhenExists = loadPackageLockFileWhenExists;
/**
 * Update NPM Lock File (when found), the lock file might be version 1 (exist in package folder) or version 2 (exist in workspace root)
 * Depending on the version type, the structure of the lock file will be different and will be updated accordingly
 * @param {Object} pkg
 * @returns Promise<string>
 */
async function updateClassicLockfileVersion(pkg) {
    var _a;
    try {
        // "lockfileVersion" = 1, package lock file might be located in the package folder
        const lockFilePath = path_1.default.join(pkg.location, 'package-lock.json');
        const pkgLockFileObj = await (0, load_json_file_1.default)(lockFilePath);
        if (pkgLockFileObj) {
            pkgLockFileObj.version = pkg.version;
            // update version for a npm lockfile v2 format
            if ((_a = pkgLockFileObj.packages) === null || _a === void 0 ? void 0 : _a['']) {
                pkgLockFileObj.packages[''].version = pkg.version;
                if (pkgLockFileObj.packages[''].dependencies) {
                    pkgLockFileObj.packages[''].dependencies = pkg.dependencies;
                }
                if (pkgLockFileObj.packages[''].devDependencies) {
                    pkgLockFileObj.packages[''].devDependencies = pkg.devDependencies;
                }
            }
            await (0, write_json_file_1.default)(lockFilePath, pkgLockFileObj, {
                detectIndent: true,
                indent: 2,
            });
            return lockFilePath;
        }
    }
    catch (error) { } // eslint-disable-line
}
exports.updateClassicLockfileVersion = updateClassicLockfileVersion;
/**
 * Update NPM Lock File (when found), the lock file must be version 2 or higher and is considered as modern lockfile,
 * its structure is different and all version properties will be updated accordingly
 * @param {Object} pkg
 * @param {Object} project
 * @returns Promise<string>
 */
function updateTempModernLockfileVersion(pkg, projLockFileObj) {
    // OR "lockfileVersion" >= 2 in the project root, will have a global package lock file located in the root folder and is formatted
    if (projLockFileObj) {
        updateNpmLockFileVersion2(projLockFileObj, pkg.name, pkg.version);
    }
}
exports.updateTempModernLockfileVersion = updateTempModernLockfileVersion;
/**
 * Save a lockfile by providing a full path and an updated json object
 * @param {String} filePath
 * @param {Object} updateLockFileObj
 * @returns Promise<String | undefined> - file path will be returned when it was found and updated
 */
async function saveUpdatedLockJsonFile(filePath, updateLockFileObj) {
    try {
        await (0, write_json_file_1.default)(filePath, updateLockFileObj, {
            detectIndent: true,
            indent: 2,
        });
        return filePath;
    }
    catch (error) { } // eslint-disable-line
}
exports.saveUpdatedLockJsonFile = saveUpdatedLockJsonFile;
/**
 * Update workspace root NPM Lock File Version Type 2 (considerd modern lockfile)
 * @param {Object} obj
 * @param {String} pkgName
 * @param {String} newVersion
 */
function updateNpmLockFileVersion2(obj, pkgName, newVersion) {
    if (typeof obj === 'object' && pkgName && newVersion) {
        for (const k in obj) {
            if (typeof obj[k] === 'object' && obj[k] !== null) {
                updateNpmLockFileVersion2(obj[k], pkgName, newVersion);
            }
            else {
                if (k === pkgName) {
                    // e.g.: "@lerna-lite/core": "^0.1.2",
                    const [_, versionPrefix, _versionStr] = obj[k].match(/^([\^~])?(.*)$/) || [];
                    obj[k] = `${versionPrefix}${newVersion}`;
                }
                else if (k === 'name' && obj[k] === pkgName && obj['version'] !== undefined) {
                    // e.g. "packages/version": { "name": "@lerna-lite/version", "version": "0.1.2" }
                    if (obj['version'] !== undefined) {
                        obj['version'] = newVersion;
                    }
                }
            }
        }
    }
}
exports.updateNpmLockFileVersion2 = updateNpmLockFileVersion2;
/**
 * Run `npm install --package-lock-only` or equivalent depending on the package manager defined in `npmClient`
 * @param {'npm' | 'pnpm' | 'yarn'} npmClient
 * @param {String} cwd
 * @returns {Promise<string | undefined>} lockfile name if executed successfully
 */
async function runInstallLockFileOnly(npmClient, cwd, npmArgs) {
    let inputLockfileName = '';
    let outputLockfileName;
    const npmClientArgsRaw = npmArgs || [];
    const npmClientArgs = npmClientArgsRaw.reduce((args, arg) => args.concat(arg.split(/\s|,/)), []);
    switch (npmClient) {
        case 'pnpm':
            inputLockfileName = 'pnpm-lock.yaml';
            if (await validateFileExists(path_1.default.join(cwd, inputLockfileName))) {
                npmlog_1.default.verbose('lock', `updating lock file via "pnpm install --lockfile-only --ignore-scripts"`);
                await (0, core_1.exec)('pnpm', ['install', '--lockfile-only', '--ignore-scripts', ...npmClientArgs], { cwd });
                outputLockfileName = inputLockfileName;
            }
            break;
        case 'yarn':
            inputLockfileName = 'yarn.lock';
            const yarnVersion = (0, core_1.execSync)('yarn', ['--version']);
            if (semver_1.default.gte(yarnVersion, '2.0.0') && (await validateFileExists(path_1.default.join(cwd, inputLockfileName)))) {
                npmlog_1.default.verbose('lock', `updating lock file via "yarn install --mode update-lockfile"`);
                await (0, core_1.exec)('yarn', ['install', '--mode', 'update-lockfile', ...npmClientArgs], { cwd });
                outputLockfileName = inputLockfileName;
            }
            break;
        case 'npm':
        default:
            inputLockfileName = 'package-lock.json';
            if (await validateFileExists(path_1.default.join(cwd, inputLockfileName))) {
                const localNpmVersion = (0, core_1.execSync)('npm', ['--version']);
                npmlog_1.default.silly(`npm`, `current local npm version is "${localNpmVersion}"`);
                // for npm version >=8.5.0 we can simply call "npm install --package-lock-only"
                // however, when lower then we need to call "npm shrinkwrap --package-lock-only" and then rename "npm-shrinkwrap.json" file back to "package-lock.json"
                if (semver_1.default.gte(localNpmVersion, '8.5.0')) {
                    npmlog_1.default.verbose('lock', `updating lock file via "npm install --package-lock-only"`);
                    await (0, core_1.exec)('npm', ['install', '--package-lock-only', ...npmClientArgs], { cwd });
                }
                else {
                    // TODO: remove this in the next major release
                    // with npm < 8.5.0, we need to update the lock file in 2 steps
                    // 1. using shrinkwrap will delete current lock file and create new "npm-shrinkwrap.json" but will avoid npm retrieving package version info from registry
                    npmlog_1.default.verbose('lock', `updating lock file via "npm shrinkwrap --package-lock-only".`);
                    npmlog_1.default.warn(`npm`, `Your npm version is lower than 8.5.0, we recommend upgrading your npm client to avoid the use of "npm shrinkwrap" instead of the regular (better) "npm install --package-lock-only".`);
                    await (0, core_1.exec)('npm', ['shrinkwrap', '--package-lock-only', ...npmClientArgs], { cwd });
                    // 2. rename "npm-shrinkwrap.json" back to "package-lock.json"
                    npmlog_1.default.verbose('lock', `renaming "npm-shrinkwrap.json" file back to "package-lock.json"`);
                    fs_1.default.renameSync('npm-shrinkwrap.json', 'package-lock.json');
                }
                outputLockfileName = inputLockfileName;
            }
            break;
    }
    if (!outputLockfileName) {
        npmlog_1.default.error('lock', [
            `we could not sync neither locate "${inputLockfileName}" by using "${npmClient}" client at location ${cwd}`,
            `Note: if you were expecting a different lock file name, make sure to configure "npmClient" into your "lerna.json" config file.`,
        ].join(os_1.default.EOL));
    }
    return outputLockfileName;
}
exports.runInstallLockFileOnly = runInstallLockFileOnly;
/**
 * Simply validates if a file exists
 * @param {String} filePath - file path
 * @returns {Boolean}
 */
async function validateFileExists(filePath) {
    try {
        await fs_1.default.promises.access(filePath);
        return true;
    }
    catch (_a) {
        return false;
    }
}
exports.validateFileExists = validateFileExists;
//# sourceMappingURL=update-lockfile-version.js.map