"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.promptVersion = exports.makePromptVersion = void 0;
const tslib_1 = require("tslib");
const semver_1 = tslib_1.__importDefault(require("semver"));
const core_1 = require("@lerna-lite/core");
const conventional_commits_1 = require("../conventional-commits");
/**
 * @param {(existingPreid: string) => string} resolvePrereleaseId
 * @param {string} buildMetadata
 */
function makePromptVersion(resolvePrereleaseId, buildMetadata) {
    return (node) => promptVersion(node.version, node.name, resolvePrereleaseId(node.prereleaseId), buildMetadata);
}
exports.makePromptVersion = makePromptVersion;
/**
 * A predicate that prompts user to select/construct a version bump.
 * It can be run per-package (independent) or globally (fixed).
 *
 * @param {PackageGraphNode|Object} node The metadata to process
 * @param {String} currentVersion
 * @param {String} name (Only used in independent mode)
 * @param {String} prereleaseId
 * @param {String} buildMetadata
 */
async function promptVersion(currentVersion, name, prereleaseId, buildMetadata) {
    const patch = (0, conventional_commits_1.applyBuildMetadata)(semver_1.default.inc(currentVersion, 'patch'), buildMetadata);
    const minor = (0, conventional_commits_1.applyBuildMetadata)(semver_1.default.inc(currentVersion, 'minor'), buildMetadata);
    const major = (0, conventional_commits_1.applyBuildMetadata)(semver_1.default.inc(currentVersion, 'major'), buildMetadata);
    const prepatch = (0, conventional_commits_1.applyBuildMetadata)(semver_1.default.inc(currentVersion, 'prepatch', prereleaseId), buildMetadata);
    const preminor = (0, conventional_commits_1.applyBuildMetadata)(semver_1.default.inc(currentVersion, 'preminor', prereleaseId), buildMetadata);
    const premajor = (0, conventional_commits_1.applyBuildMetadata)(semver_1.default.inc(currentVersion, 'premajor', prereleaseId), buildMetadata);
    const message = `Select a new version ${name ? `for ${name} ` : ''}(currently ${currentVersion})`;
    const choice = await (0, core_1.promptSelectOne)(message, {
        choices: [
            { value: patch, name: `Patch (${patch})` },
            { value: minor, name: `Minor (${minor})` },
            { value: major, name: `Major (${major})` },
            { value: prepatch, name: `Prepatch (${prepatch})` },
            { value: preminor, name: `Preminor (${preminor})` },
            { value: premajor, name: `Premajor (${premajor})` },
            { value: 'PRERELEASE', name: 'Custom Prerelease' },
            { value: 'CUSTOM', name: 'Custom Version' },
        ],
    });
    if (choice === 'CUSTOM') {
        return (0, core_1.promptTextInput)('Enter a custom version', {
            filter: semver_1.default.valid,
            // semver.valid() always returns null with invalid input
            validate: (v) => v !== null || 'Must be a valid semver version',
        });
    }
    if (choice === 'PRERELEASE') {
        const defaultVersion = semver_1.default.inc(currentVersion, 'prerelease', prereleaseId);
        const prompt = `(default: "${prereleaseId}", yielding ${defaultVersion})`;
        return (0, core_1.promptTextInput)(`Enter a prerelease identifier ${prompt}`, {
            filter: (v) => (0, conventional_commits_1.applyBuildMetadata)(semver_1.default.inc(currentVersion, 'prerelease', v || prereleaseId), buildMetadata),
        });
    }
    return choice;
}
exports.promptVersion = promptVersion;
//# sourceMappingURL=prompt-version.js.map