"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createRelease = exports.createReleaseClient = void 0;
const tslib_1 = require("tslib");
const core_1 = require("@lerna-lite/core");
const chalk_1 = tslib_1.__importDefault(require("chalk"));
const npmlog_1 = tslib_1.__importDefault(require("npmlog"));
const new_github_release_url_1 = tslib_1.__importDefault(require("new-github-release-url"));
const semver_1 = tslib_1.__importDefault(require("semver"));
const git_clients_1 = require("../git-clients");
function createReleaseClient(type) {
    switch (type) {
        case 'gitlab':
            return (0, git_clients_1.createGitLabClient)();
        case 'github':
            return (0, git_clients_1.createGitHubClient)();
        /* istanbul ignore next: guarded by yargs.choices() */
        default:
            throw new core_1.ValidationError('ERELEASE', 'Invalid release client type');
    }
}
exports.createReleaseClient = createReleaseClient;
/**
 * @param {ReturnType<typeof createReleaseClient>} client
 * @param {{ tags: string[]; releaseNotes: { name: string; notes: string; }[] }} commandProps
 * @param {{ gitRemote: string; execOpts: import('@lerna/child-process').ExecOpts }} opts
 */
function createRelease(client, { tags, releaseNotes }, { gitRemote, execOpts }, dryRun = false) {
    const { GH_TOKEN } = process.env;
    const repo = (0, git_clients_1.parseGitRepo)(gitRemote, execOpts);
    return Promise.all(releaseNotes.map(({ notes, name }) => {
        const tag = name === 'fixed' ? tags[0] : tags.find((t) => t.startsWith(`${name}@`));
        /* istanbul ignore if */
        if (!tag) {
            return Promise.resolve();
        }
        const prereleaseParts = semver_1.default.prerelease(tag.replace(`${name}@`, '')) || [];
        // when the `GH_TOKEN` environment variable is not set,
        // we'll create a link to GitHub web interface form with the fields pre-populated
        if (!GH_TOKEN) {
            const releaseUrl = (0, new_github_release_url_1.default)({
                user: repo.owner,
                repo: repo.name,
                tag,
                isPrerelease: prereleaseParts.length > 0,
                title: tag,
                body: notes,
            });
            npmlog_1.default.verbose('github', 'GH_TOKEN environment variable is not set');
            npmlog_1.default.info('github', `🏷️ (GitHub Release web interface) - 🔗 ${releaseUrl}`);
            return Promise.resolve();
        }
        const releaseOptions = {
            owner: repo.owner,
            repo: repo.name,
            tag_name: tag,
            name: tag,
            body: notes,
            draft: false,
            prerelease: prereleaseParts.length > 0,
        };
        if (dryRun) {
            npmlog_1.default.info(chalk_1.default.bold.magenta('[dry-run] >'), `Create Release with repo options: `, JSON.stringify(releaseOptions));
            return Promise.resolve();
        }
        return client.repos.createRelease(releaseOptions);
    }));
}
exports.createRelease = createRelease;
//# sourceMappingURL=create-release.js.map