"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.updateChangelog = void 0;
const tslib_1 = require("tslib");
const core_1 = require("@lerna-lite/core");
const conventional_changelog_core_1 = tslib_1.__importDefault(require("conventional-changelog-core"));
const fs_extra_1 = tslib_1.__importDefault(require("fs-extra"));
const get_stream_1 = tslib_1.__importDefault(require("get-stream"));
const npmlog_1 = tslib_1.__importDefault(require("npmlog"));
const constants_1 = require("./constants");
const get_changelog_config_1 = require("./get-changelog-config");
const make_bump_only_filter_1 = require("./make-bump-only-filter");
const read_existing_changelog_1 = require("./read-existing-changelog");
const writer_opts_transform_1 = require("./writer-opts-transform");
/**
 * Update changelog with the commits of the new release
 * @param {Package} pkg
 * @param {ChangelogType} type
 * @param {UpdateChangelogOption} commandOptions
 */
async function updateChangelog(pkg, type, updateOptions) {
    npmlog_1.default.silly(type, 'for %s at %s', pkg.name, pkg.location);
    const { changelogPreset, changelogIncludeCommitsGitAuthor, changelogIncludeCommitsClientLogin, changelogHeaderMessage = '', changelogVersionMessage = '', commitsSinceLastRelease, rootPath, tagPrefix = 'v', version, } = updateOptions;
    const config = await get_changelog_config_1.GetChangelogConfig.getChangelogConfig(changelogPreset, rootPath);
    const options = {};
    const context = {}; // pass as positional because cc-core's merge-config is wack
    const writerOpts = {};
    // cc-core mutates input :P
    if (config.conventionalChangelog) {
        // "new" preset API
        options.config = Object.assign({}, config.conventionalChangelog);
    }
    else {
        // "old" preset API
        options.config = Object.assign({}, config);
    }
    // NOTE: must pass as positional argument due to weird bug in merge-config
    const gitRawCommitsOpts = Object.assign({}, options.config.gitRawCommitsOpts);
    // are we including commit author name/email or remote client login name
    if (changelogIncludeCommitsGitAuthor || changelogIncludeCommitsGitAuthor === '') {
        (0, writer_opts_transform_1.setConfigChangelogCommitGitAuthor)(config, gitRawCommitsOpts, writerOpts, changelogIncludeCommitsGitAuthor);
    }
    else if ((changelogIncludeCommitsClientLogin || changelogIncludeCommitsClientLogin === '') &&
        commitsSinceLastRelease) {
        // prettier-ignore
        (0, writer_opts_transform_1.setConfigChangelogCommitClientLogin)(config, gitRawCommitsOpts, writerOpts, commitsSinceLastRelease, changelogIncludeCommitsClientLogin);
    }
    if (type === 'root') {
        context.version = version;
        // preserve tagPrefix because cc-core can't find the currentTag otherwise
        context.currentTag = `${tagPrefix}${version}`;
        // root changelogs are only enabled in fixed mode, and need the proper tag prefix
        options.tagPrefix = tagPrefix;
    }
    else {
        // "fixed" or "independent"
        gitRawCommitsOpts.path = pkg.location;
        options.pkg = { path: pkg.manifestLocation };
        if (type === 'independent') {
            options.lernaPackage = pkg.name;
        }
        else {
            // only fixed mode can have a custom tag prefix
            options.tagPrefix = tagPrefix;
            // preserve tagPrefix because cc-core can't find the currentTag otherwise
            context.currentTag = `${tagPrefix}${pkg.version}`;
            context.version = pkg.version; // TODO investigate why Lerna doesn't have this line
        }
    }
    // generate the markdown for the upcoming release.
    const changelogStream = (0, conventional_changelog_core_1.default)(options, context, gitRawCommitsOpts, undefined, writerOpts);
    return Promise.all([
        // prettier-ignore
        (0, get_stream_1.default)(changelogStream).then((0, make_bump_only_filter_1.makeBumpOnlyFilter)(pkg)),
        (0, read_existing_changelog_1.readExistingChangelog)(pkg),
    ]).then(([inputEntry, [changelogFileLoc, changelogContents]]) => {
        let newEntry = inputEntry;
        npmlog_1.default.silly(type, 'writing new entry: %j', newEntry);
        const changelogVersion = type === 'root' ? changelogVersionMessage : '';
        const changelogHeader = constants_1.CHANGELOG_HEADER.replace(/%s/g, (changelogHeaderMessage === null || changelogHeaderMessage === void 0 ? void 0 : changelogHeaderMessage.length) > 0 ? changelogHeaderMessage + core_1.EOL : '');
        const content = [changelogHeader, changelogVersion, newEntry, changelogContents]
            .join(constants_1.BLANK_LINE)
            .trim()
            .replace(/[\r\n]{2,}/gm, '\n\n'); // conventional-changelog adds way too many extra line breaks, let's remove a few of them
        return fs_extra_1.default.writeFile(changelogFileLoc, content + core_1.EOL).then(() => {
            npmlog_1.default.verbose(type, 'wrote', changelogFileLoc);
            return {
                logPath: changelogFileLoc,
                content,
                newEntry,
            };
        });
    });
}
exports.updateChangelog = updateChangelog;
//# sourceMappingURL=update-changelog.js.map