"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.recommendVersion = void 0;
const tslib_1 = require("tslib");
const conventional_recommended_bump_1 = tslib_1.__importDefault(require("conventional-recommended-bump"));
const npmlog_1 = tslib_1.__importDefault(require("npmlog"));
const semver_1 = tslib_1.__importDefault(require("semver"));
const get_changelog_config_1 = require("./get-changelog-config");
const apply_build_metadata_1 = require("./apply-build-metadata");
/**
 * @param {import('@lerna/package').Package} pkg
 * @param {import('..').VersioningStrategy} type
 * @param {import('..').BaseChangelogOptions & { prereleaseId?: string, buildMetadata?: string }} commandOptions
 */
async function recommendVersion(pkg, type, recommendationOptions) {
    const { changelogPreset, rootPath, tagPrefix, prereleaseId, conventionalBumpPrerelease, buildMetadata } = recommendationOptions;
    npmlog_1.default.silly(type, 'for %s at %s', pkg.name, pkg.location);
    const options = {
        path: pkg.location,
    };
    if (type === 'independent') {
        options.lernaPackage = pkg.name;
    }
    else {
        // only fixed mode can have a custom tag prefix
        options.tagPrefix = tagPrefix;
    }
    const shouldBumpPrerelease = (releaseType, version) => {
        if (!semver_1.default.prerelease(version)) {
            return true;
        }
        switch (releaseType) {
            case 'major':
                return semver_1.default.minor(version) !== 0 || semver_1.default.patch(version) !== 0;
            case 'minor':
                return semver_1.default.patch(version) !== 0;
            default:
                return false;
        }
    };
    // 'new' preset API
    options.config = (await get_changelog_config_1.GetChangelogConfig.getChangelogConfig(changelogPreset, rootPath));
    // Ensure potential ValidationError in getChangelogConfig() is propagated correctly
    return new Promise((resolve, reject) => {
        (0, conventional_recommended_bump_1.default)(options, (err, data) => {
            if (err) {
                return reject(err);
            }
            // result might be undefined because some presets are not consistent with angular
            // we still need to bump _something_ because lerna-lite saw a change here
            let releaseType = data.releaseType || 'patch';
            if (prereleaseId) {
                const shouldBump = conventionalBumpPrerelease || shouldBumpPrerelease(releaseType, pkg.version);
                const prereleaseType = shouldBump ? `pre${releaseType}` : 'prerelease';
                npmlog_1.default.verbose(type, 'increment %s by %s - %s', pkg.version, prereleaseType, pkg.name);
                resolve((0, apply_build_metadata_1.applyBuildMetadata)(semver_1.default.inc(pkg.version, prereleaseType, prereleaseId), buildMetadata));
            }
            else {
                if (semver_1.default.major(pkg.version) === 0) {
                    // According to semver, major version zero (0.y.z) is for initial
                    // development. Anything MAY change at any time. The public API
                    // SHOULD NOT be considered stable. The version 1.0.0 defines
                    // the (initial stable) public API.
                    //
                    // To allow monorepos to use major version zero meaningfully,
                    // the transition from 0.x to 1.x must be explicitly requested
                    // by the user. Breaking changes MUST NOT automatically bump
                    // the major version from 0.x to 1.x.
                    //
                    // The usual convention is to use semver-patch bumps for bugfix
                    // releases and semver-minor for everything else, including
                    // breaking changes. This matches the behavior of `^` operator
                    // as implemented by `npm`.
                    //
                    if (releaseType === 'major') {
                        releaseType = 'minor';
                    }
                }
                npmlog_1.default.verbose(type, 'increment %s by %s - %s', pkg.version, releaseType, pkg.name);
                resolve((0, apply_build_metadata_1.applyBuildMetadata)(semver_1.default.inc(pkg.version, releaseType), buildMetadata));
            }
        });
    });
}
exports.recommendVersion = recommendVersion;
//# sourceMappingURL=recommend-version.js.map