"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getGithubCommits = void 0;
const tslib_1 = require("tslib");
const core_1 = require("@lerna-lite/core");
const dedent_1 = tslib_1.__importDefault(require("dedent"));
const npmlog_1 = tslib_1.__importDefault(require("npmlog"));
const git_clients_1 = require("../git-clients");
const QUERY_PAGE_SIZE = 100; // GitHub API is restricting max of 100 per query
/**
 * Get all commits from GitHub remote repository, using GitHub Graphql API, by providing a date to query from
 * https://docs.github.com/en/graphql/reference/objects#repository
 * @param {RemoteClientType} client
 * @param {String} gitRemote
 * @param {String} branchName
 * @param {ExecOpts} execOpts
 * @returns {Promise<RemoteCommit[]>}
 */
async function getGithubCommits(gitRemote, branchName, sinceDate, execOpts) {
    var _a, _b, _c, _d, _e, _f;
    const repo = (0, git_clients_1.parseGitRepo)(gitRemote, execOpts);
    const octokit = (0, git_clients_1.createGitHubClient)();
    const remoteCommits = [];
    let afterCursor = '';
    let hasNextPage = false;
    do {
        const afterCursorStr = afterCursor ? `, after: "${afterCursor}"` : '';
        const queryStr = (0, dedent_1.default)(`query getCommits($repo: String!, $owner: String!, $branchName: String!, $pageSize: Int!, $since: GitTimestamp!) {
          repository(name: $repo, owner: $owner) {
            ref(qualifiedName: $branchName) {
              target { ... on Commit {
                  history(first: $pageSize, since: $since ${afterCursorStr}) {
                    nodes { oid, message, author { name, user { login }}}
                    pageInfo { hasNextPage, endCursor }
        }}}}}}`).trim();
        const response = await octokit.graphql(queryStr, {
            owner: repo.owner,
            repo: repo.name,
            afterCursor,
            branchName,
            pageSize: QUERY_PAGE_SIZE,
            since: sinceDate,
        });
        const historyData = (0, core_1.getComplexObjectValue)(response, 'repository.ref.target.history');
        const pageInfo = historyData === null || historyData === void 0 ? void 0 : historyData.pageInfo;
        hasNextPage = (_a = pageInfo === null || pageInfo === void 0 ? void 0 : pageInfo.hasNextPage) !== null && _a !== void 0 ? _a : false;
        afterCursor = (_b = pageInfo === null || pageInfo === void 0 ? void 0 : pageInfo.endCursor) !== null && _b !== void 0 ? _b : '';
        if (historyData === null || historyData === void 0 ? void 0 : historyData.nodes) {
            for (const commit of historyData.nodes) {
                if ((commit === null || commit === void 0 ? void 0 : commit.oid) && (commit === null || commit === void 0 ? void 0 : commit.author)) {
                    remoteCommits.push({
                        shortHash: commit.oid.substring(0, 7),
                        authorName: commit === null || commit === void 0 ? void 0 : commit.author.name,
                        login: (_e = (_d = (_c = commit === null || commit === void 0 ? void 0 : commit.author) === null || _c === void 0 ? void 0 : _c.user) === null || _d === void 0 ? void 0 : _d.login) !== null && _e !== void 0 ? _e : '',
                        message: (_f = commit === null || commit === void 0 ? void 0 : commit.message) !== null && _f !== void 0 ? _f : '',
                    });
                }
            }
        }
    } while (hasNextPage);
    npmlog_1.default.verbose('github', 'found %s commits since last release timestamp %s', remoteCommits.length, sinceDate);
    return remoteCommits;
}
exports.getGithubCommits = getGithubCommits;
//# sourceMappingURL=get-github-commits.js.map