"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getOldestCommitSinceLastTag = exports.getCommitsSinceLastRelease = void 0;
const tslib_1 = require("tslib");
const core_1 = require("@lerna-lite/core");
const npmlog_1 = tslib_1.__importDefault(require("npmlog"));
const get_github_commits_1 = require("./get-github-commits");
/**
 * From the current branch, find all commits since the last tag release.
 * The output will be an array which include every commit short hash & user login
 * @param {RemoteClientType} client
 * @param {String} gitRemote
 * @param {String} branchName
 * @param {Boolean} [isIndependent]
 * @param {ExecOpts} [execOpts]
 * @returns {Promise<RemoteCommit[]>}
 */
async function getCommitsSinceLastRelease(client, gitRemote, branchName, isIndependent, execOpts) {
    // get the last release tag date or the first commit date if no release tag found
    const { commitDate } = getOldestCommitSinceLastTag(execOpts, isIndependent, false);
    switch (client) {
        case 'github':
            return (0, get_github_commits_1.getGithubCommits)(gitRemote, branchName, commitDate, execOpts);
        case 'gitlab':
        default:
            throw new core_1.ValidationError('EREMOTE', 'Invalid remote client type, "github" is currently the only supported client with the option --changelog-include-commits-client-login.');
    }
}
exports.getCommitsSinceLastRelease = getCommitsSinceLastRelease;
/**
 * Find the oldest commit details since the last release tag or else if not tag exists then return first commit info
 * @param {ExecOpts} [execOpts]
 * @param {Boolean} [includeMergedTags]
 * @param {Boolean} [isIndependent]
 * @returns {*} - oldest commit detail (hash, date)
 */
function getOldestCommitSinceLastTag(execOpts, isIndependent, includeMergedTags) {
    let commitResult = '';
    const describeOptions = { ...execOpts };
    if (isIndependent) {
        describeOptions.match = '*@*'; // independent tag pattern
    }
    const { lastTagName } = (0, core_1.describeRefSync)(describeOptions, includeMergedTags);
    if (lastTagName) {
        const gitCommandArgs = ['log', `${lastTagName}..HEAD`, '--format="%h %aI"', '--reverse'];
        npmlog_1.default.silly('git', 'getCurrentBranchOldestCommitSinceLastTag');
        npmlog_1.default.verbose('exec', `git ${gitCommandArgs.join(' ')}`);
        let stdout = (0, core_1.execSync)('git', gitCommandArgs, execOpts);
        if (!stdout) {
            // in some occasion the previous git command might return nothing, in that case we'll return the tag detail instead
            stdout = (0, core_1.execSync)('git', ['log', '-1', '--format="%h %aI"', lastTagName], execOpts);
        }
        [commitResult] = stdout.split('\n');
    }
    else {
        const gitCommandArgs = ['log', '--oneline', '--format="%h %aI"', '--reverse', '--max-parents=0', 'HEAD'];
        npmlog_1.default.silly('git', 'getCurrentBranchFirstCommit');
        npmlog_1.default.verbose('exec', `git ${gitCommandArgs.join(' ')}`);
        commitResult = (0, core_1.execSync)('git', gitCommandArgs, execOpts);
    }
    const [, commitHash, commitDate] = /^"?([0-9a-f]+)\s([0-9\-|\+T\:]*)"?$/.exec(commitResult) || [];
    // prettier-ignore
    npmlog_1.default.info('oldestCommitSinceLastTag', `commit found since last tag: ${lastTagName} - (SHA) ${commitHash} - ${commitDate}`);
    return { commitHash, commitDate };
}
exports.getOldestCommitSinceLastTag = getOldestCommitSinceLastTag;
//# sourceMappingURL=get-commits-since-last-release.js.map