"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GetChangelogConfig = void 0;
const tslib_1 = require("tslib");
const core_1 = require("@lerna-lite/core");
const npmlog_1 = tslib_1.__importDefault(require("npmlog"));
const npm_package_arg_1 = tslib_1.__importDefault(require("npm-package-arg"));
const pify_1 = tslib_1.__importDefault(require("pify"));
class GetChangelogConfig {
    static isFunction(config) {
        return Object.prototype.toString.call(config) === '[object Function]';
    }
    static resolveConfigPromise(presetPackageName, presetConfig) {
        npmlog_1.default.verbose('getChangelogConfig', 'Attempting to resolve preset %j', presetPackageName);
        let config = require(presetPackageName);
        npmlog_1.default.info('getChangelogConfig', 'Successfully resolved preset %j', presetPackageName);
        if (this.isFunction(config)) {
            try {
                // try assuming config builder function first
                config = config(presetConfig);
            }
            catch (_) {
                // legacy presets export an errback function instead of Q.all()
                config = (0, pify_1.default)(config)();
            }
        }
        return config;
    }
    /**
     * @param {ChangelogPresetConfig} [changelogPreset]
     * @param {string} [rootPath]
     */
    static getChangelogConfig(changelogPreset = 'conventional-changelog-angular', rootPath) {
        const presetName = typeof changelogPreset === 'string' ? changelogPreset : changelogPreset.name;
        const presetConfig = typeof changelogPreset === 'object' ? changelogPreset : {};
        const cacheKey = `${presetName}${presetConfig ? JSON.stringify(presetConfig) : ''}`;
        let config = GetChangelogConfig.cfgCache.get(cacheKey);
        if (!config) {
            let presetPackageName = presetName;
            // https://github.com/npm/npm-package-arg#result-object
            const parsed = (0, npm_package_arg_1.default)(presetPackageName, rootPath);
            npmlog_1.default.verbose('getChangelogConfig', 'using preset %j', presetPackageName);
            npmlog_1.default.silly('npa', parsed);
            if (parsed.type === 'directory') {
                if (parsed.raw[0] === '@') {
                    // npa parses scoped subpath reference as a directory
                    parsed.name = parsed.raw;
                    parsed.scope = parsed.raw.substring(0, parsed.raw.indexOf('/'));
                    // un-scoped subpath shorthand handled in first catch block
                }
                else {
                    presetPackageName = parsed.fetchSpec;
                }
            }
            else if (parsed.type === 'git' && parsed.hosted && parsed.hosted.default === 'shortcut') {
                // probably a shorthand subpath, e.g. 'foo/bar'
                parsed.name = parsed.raw;
            }
            // Maybe it doesn't need an implicit 'conventional-changelog-' prefix?
            try {
                config = this.resolveConfigPromise(presetPackageName, presetConfig);
                GetChangelogConfig.cfgCache.set(cacheKey, config);
                // early exit, yay
                return Promise.resolve(config);
            }
            catch (err) {
                npmlog_1.default.verbose('getChangelogConfig', err.message);
                npmlog_1.default.info('getChangelogConfig', 'Auto-prefixing conventional-changelog preset %j', presetName);
                // probably a deep shorthand subpath :P
                parsed.name = parsed.raw;
            }
            if (parsed.name.indexOf('conventional-changelog-') < 0) {
                // implicit 'conventional-changelog-' prefix
                const parts = parsed.name.split('/');
                const start = parsed.scope ? 1 : 0;
                //        foo =>        conventional-changelog-foo
                // @scope/foo => @scope/conventional-changelog-foo
                parts.splice(start, 1, `conventional-changelog-${parts[start]}`);
                // _technically_ supports 'foo/lib/bar.js', but that's gross
                presetPackageName = parts.join('/');
            }
            try {
                config = this.resolveConfigPromise(presetPackageName, presetConfig);
                GetChangelogConfig.cfgCache.set(cacheKey, config);
            }
            catch (err) {
                npmlog_1.default.warn('getChangelogConfig', err.message);
                throw new core_1.ValidationError('EPRESET', `Unable to load conventional-changelog preset "${presetName}"${presetName !== presetPackageName ? ` (${presetPackageName})` : ''}`);
            }
        }
        // the core presets are bloody Q.all() spreads
        return Promise.resolve(config);
    }
}
exports.GetChangelogConfig = GetChangelogConfig;
GetChangelogConfig.cfgCache = new Map();
//# sourceMappingURL=get-changelog-config.js.map