"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RunCommand = exports.factory = void 0;
const tslib_1 = require("tslib");
const core_1 = require("@lerna-lite/core");
const filter_packages_1 = require("@lerna-lite/filter-packages");
const profiler_1 = require("@lerna-lite/profiler");
const chalk_1 = tslib_1.__importDefault(require("chalk"));
const fs_extra_1 = require("fs-extra");
const p_map_1 = tslib_1.__importDefault(require("p-map"));
const path_1 = tslib_1.__importDefault(require("path"));
const perf_hooks_1 = require("perf_hooks");
const lib_1 = require("./lib");
function factory(argv) {
    return new RunCommand(argv);
}
exports.factory = factory;
class RunCommand extends core_1.Command {
    get requiresGit() {
        return false;
    }
    constructor(argv) {
        super(argv);
        /** command name */
        this.name = 'run';
        this.args = [];
        this.bail = false;
        this.prefix = false;
        this.npmClient = 'npm';
        this.packagesWithScript = [];
        this.script = '';
    }
    initialize() {
        const { script, npmClient = 'npm' } = this.options;
        this.script = script;
        this.args = this.options['--'] || [];
        this.npmClient = npmClient;
        if (!script) {
            throw new core_1.ValidationError('ENOSCRIPT', 'You must specify a lifecycle script to run');
        }
        // inverted boolean options
        this.bail = this.options.bail !== false;
        this.prefix = this.options.prefix !== false;
        let chain = Promise.resolve();
        if (!this.options.log) {
            this.options.log = this.logger;
        }
        this.options.isIndependent = this.project.isIndependent();
        chain = chain.then(() => (0, filter_packages_1.getFilteredPackages)(this.packageGraph, this.execOpts, this.options));
        chain = chain.then((filteredPackages) => {
            this.packagesWithScript =
                script === 'env' ? filteredPackages : filteredPackages.filter((pkg) => pkg.scripts && pkg.scripts[script]);
        });
        return chain.then(() => {
            this.count = this.packagesWithScript.length;
            this.packagePlural = this.count === 1 ? 'package' : 'packages';
            this.joinedCommand = [this.npmClient, 'run', this.script].concat(this.args).join(' ');
            if (!this.count) {
                this.logger.info('run', `No packages found with the lifecycle script "${script}"`);
                // still exits zero, aka 'ok'
                return false;
            }
        });
    }
    execute() {
        if (!this.options.useNx) {
            this.logger.info('', 'Executing command in %d %s: %j', this.count, this.packagePlural, this.joinedCommand);
        }
        let chain = Promise.resolve();
        const getElapsed = (0, lib_1.timer)();
        if (this.options.useNx) {
            chain = chain.then(() => this.runScriptsUsingNx());
        }
        else if (this.options.parallel) {
            this.logger.verbose('Parallel', this.joinedCommand);
            chain = chain.then(() => this.runScriptInPackagesParallel());
        }
        else if (this.toposort) {
            this.logger.verbose('Topological', this.joinedCommand);
            chain = chain.then(() => this.runScriptInPackagesTopological());
        }
        else {
            this.logger.verbose('Lexical', this.joinedCommand);
            chain = chain.then(() => this.runScriptInPackagesLexical());
        }
        if (this.bail) {
            // only the first error is caught
            chain = chain.catch((err) => {
                process.exitCode = err.exitCode;
                // rethrow to halt chain and log properly
                throw err;
            });
        }
        else {
            // detect error (if any) from collected results
            chain = chain.then((results) => {
                /* istanbul ignore else */
                if (results === null || results === void 0 ? void 0 : results.some((result) => result === null || result === void 0 ? void 0 : result.failed)) {
                    // propagate 'highest' error code, it's probably the most useful
                    const codes = results.filter((result) => result === null || result === void 0 ? void 0 : result.failed).map((result) => result.exitCode);
                    const exitCode = Math.max(...codes, 1);
                    this.logger.error('', 'Received non-zero exit code %d during execution', exitCode);
                    if (!this.options.stream) {
                        results
                            .filter((result) => result === null || result === void 0 ? void 0 : result.failed)
                            .forEach((result) => {
                            var _a, _b;
                            this.logger.error('', (_b = (_a = result.pkg) === null || _a === void 0 ? void 0 : _a.name) !== null && _b !== void 0 ? _b : '', result.stderr);
                        });
                    }
                    process.exitCode = exitCode;
                }
                return results;
            });
        }
        return chain.then((results) => {
            const someFailed = results === null || results === void 0 ? void 0 : results.some((result) => result === null || result === void 0 ? void 0 : result.failed);
            const logType = someFailed ? 'error' : 'success';
            this.logger[logType]('run', `Ran npm script '%s' in %d %s in %ss:`, this.script, this.count, this.packagePlural, (getElapsed() / 1000).toFixed(1));
            if (!this.bail && !this.options.stream) {
                results.forEach((result) => {
                    var _a, _b, _c, _d;
                    if (result === null || result === void 0 ? void 0 : result.failed) {
                        this.logger.error('', ` - ${(_b = (_a = result.pkg) === null || _a === void 0 ? void 0 : _a.name) !== null && _b !== void 0 ? _b : ''}`);
                    }
                    else {
                        this.logger.success('', ` - ${(_d = (_c = result.pkg) === null || _c === void 0 ? void 0 : _c.name) !== null && _d !== void 0 ? _d : ''}`);
                    }
                });
            }
            else {
                this.logger.success('', this.packagesWithScript.map((pkg) => `- ${pkg.name}`).join('\n'));
            }
        });
    }
    getOpts(pkg) {
        // these options are NOT passed directly to execa, they are composed in npm-run-script
        return {
            args: this.args,
            npmClient: this.npmClient,
            prefix: this.prefix,
            reject: this.bail,
            pkg,
        };
    }
    getRunner() {
        return this.options.stream
            ? (pkg) => this.runScriptInPackageStreaming(pkg)
            : (pkg) => this.runScriptInPackageCapturing(pkg);
    }
    runScriptInPackagesTopological() {
        let profiler;
        let runner;
        if (this.options.profile) {
            profiler = new profiler_1.Profiler({
                concurrency: this.concurrency,
                log: this.logger,
                outputDirectory: this.options.profileLocation,
            });
            const callback = this.getRunner();
            runner = (pkg) => profiler.run(() => callback(pkg), pkg.name);
        }
        else {
            runner = this.getRunner();
        }
        let chain = (0, core_1.runTopologically)(this.packagesWithScript, runner, {
            concurrency: this.concurrency,
            rejectCycles: this.options.rejectCycles,
        });
        if (profiler) {
            chain = chain.then((results) => profiler.output().then(() => results));
        }
        return chain;
    }
    /** Nx requires quotes around script names of the form script:name*/
    escapeScriptNameQuotes(scriptName) {
        return scriptName.includes(':') ? `"${scriptName}"` : scriptName;
    }
    async runScriptsUsingNx() {
        if (this.options.ci) {
            process.env.CI = 'true';
        }
        if (this.options.profile) {
            const absolutePath = (0, profiler_1.generateProfileOutputPath)(this.options.profileLocation);
            // Nx requires a workspace relative path for this
            process.env.NX_PROFILE = path_1.default.relative(this.project.rootPath, absolutePath);
        }
        perf_hooks_1.performance.mark('init-local');
        await this.configureNxOutput();
        const { extraOptions, targetDependencies, options } = await this.prepNxOptions();
        if (this.packagesWithScript.length === 1) {
            const { runOne } = await Promise.resolve().then(() => tslib_1.__importStar(require('nx/src/command-line/run-one')));
            const fullQualifiedTarget = this.packagesWithScript.map((p) => p.name)[0] + ':' + this.escapeScriptNameQuotes(this.script);
            return runOne(process.cwd(), {
                'project:target:configuration': fullQualifiedTarget,
                ...options,
            }, targetDependencies, extraOptions);
        }
        else {
            const { runMany } = await Promise.resolve().then(() => tslib_1.__importStar(require('nx/src/command-line/run-many')));
            const projects = this.packagesWithScript.map((p) => p.name).join(',');
            return runMany({
                projects,
                target: this.script,
                ...options,
            }, targetDependencies);
        }
    }
    async prepNxOptions() {
        var _a;
        const nxJsonExists = (0, fs_extra_1.existsSync)(path_1.default.join(this.project.rootPath, 'nx.json'));
        const { readNxJson } = await Promise.resolve().then(() => tslib_1.__importStar(require('nx/src/config/configuration')));
        const nxJson = readNxJson();
        const targetDependenciesAreDefined = Object.keys(nxJson.targetDependencies || nxJson.targetDefaults || {}).length > 0;
        const hasProjectSpecificNxConfiguration = this.packagesWithScript.some((p) => !!p.get('nx'));
        const hasCustomizedNxConfiguration = (nxJsonExists && targetDependenciesAreDefined) || hasProjectSpecificNxConfiguration;
        const mimicLernaDefaultBehavior = !hasCustomizedNxConfiguration;
        const targetDependencies = this.toposort && !this.options.parallel && mimicLernaDefaultBehavior
            ? {
                [this.script]: [
                    {
                        projects: 'dependencies',
                        target: this.script,
                    },
                ],
            }
            : {};
        if (this.options.prefix === false && !this.options.stream) {
            this.logger.warn(this.name, `"no-prefix" is ignored when not using streaming output.`);
        }
        // prettier-ignore
        const outputStyle = this.options.stream
            ? this.prefix
                ? 'stream'
                : 'stream-without-prefixes'
            : 'dynamic';
        const options = {
            outputStyle,
            /**
             * To match lerna's own behavior (via pMap's default concurrency), we set parallel to a very large number if
             * the flag has been set (we can't use Infinity because that would cause issues with the task runner).
             */
            parallel: this.options.parallel && mimicLernaDefaultBehavior ? 999 : this.concurrency,
            nxBail: this.bail,
            nxIgnoreCycles: !this.options.rejectCycles,
            skipNxCache: this.options.skipNxCache,
            verbose: this.options.verbose,
            __overrides__: this.args.map((t) => t.toString()),
        };
        if (hasCustomizedNxConfiguration) {
            this.logger.verbose(this.name, 'Nx target configuration was found. Task dependencies will be automatically included.');
            if (this.options.parallel || this.options.sort !== undefined) {
                this.logger.warn(this.name, `"parallel", "sort", and "no-sort" are ignored when Nx targets are configured.`);
            }
            if (this.options.includeDependencies) {
                this.logger.info(this.name, `Using the "include-dependencies" option when Nx targets are configured will include both task dependencies detected by Nx and project dependencies detected by Lerna.`);
            }
            if (this.options.ignore) {
                this.logger.info(this.name, `Using the "ignore" option when Nx targets are configured will exclude only tasks that are not determined to be required by Nx.`);
            }
        }
        else {
            this.logger.verbose(this.name, 'Nx target configuration was not found. Task dependencies will not be automatically included.');
        }
        const extraOptions = {
            excludeTaskDependencies: mimicLernaDefaultBehavior,
            loadDotEnvFiles: (_a = this.options.loadEnvFiles) !== null && _a !== void 0 ? _a : true,
        };
        return { targetDependencies, options, extraOptions };
    }
    runScriptInPackagesParallel() {
        return (0, p_map_1.default)(this.packagesWithScript, (pkg) => this.runScriptInPackageStreaming(pkg));
    }
    runScriptInPackagesLexical() {
        return (0, p_map_1.default)(this.packagesWithScript, this.getRunner(), { concurrency: this.concurrency });
    }
    runScriptInPackageStreaming(pkg) {
        if (this.options.dryRun) {
            return this.dryRunScript(this.script, pkg.name);
        }
        const chain = (0, lib_1.npmRunScriptStreaming)(this.script, this.getOpts(pkg));
        if (!this.bail) {
            chain.then((result) => ({ ...result, pkg }));
        }
        return chain;
    }
    runScriptInPackageCapturing(pkg) {
        const getElapsed = (0, lib_1.timer)();
        if (this.options.dryRun) {
            return this.dryRunScript(this.script, pkg.name);
        }
        return (0, lib_1.npmRunScript)(this.script, this.getOpts(pkg)).then((result) => {
            this.logger.info('run', `Ran npm script '%s' in '%s' in %ss:`, this.script, pkg.name, (getElapsed() / 1000).toFixed(1));
            (0, core_1.logOutput)(result.stdout);
            if (!this.bail) {
                return { ...result, pkg };
            }
            return result;
        });
    }
    async configureNxOutput() {
        try {
            const nxOutput = await Promise.resolve().then(() => tslib_1.__importStar(require('nx/src/utils/output')));
            nxOutput.output.cliName = 'Lerna (powered by Nx)';
            nxOutput.output.formatCommand = (taskId) => taskId;
            return nxOutput;
        }
        catch (e) {
            this.logger.error('\n', `You have set 'useNx: true' in lerna.json, but you haven't installed Nx as a dependency.\n` +
                `To do it run 'npm install -D nx@latest' or 'yarn add -D -W nx@latest'.\n` +
                `Optional: To configure the caching and distribution run 'npx nx init' after installing it.`);
            process.exit(1);
        }
    }
    dryRunScript(scriptName, pkgName) {
        this.logger.info(chalk_1.default.bold.magenta('[dry-run] >'), `Run npm script '%s' in '%s'`, scriptName, pkgName);
        (0, core_1.logOutput)(`${chalk_1.default.bold.magenta('[dry-run] >')} ${pkgName}`);
        return Promise.resolve();
    }
}
exports.RunCommand = RunCommand;
//# sourceMappingURL=run-command.js.map